#include "cache.h"
#include "manager.h"

#include <library/cpp/protobuf/util/is_equal.h>

#include <util/string/builder.h>
#include <util/system/tempfile.h>
#include <util/system/fs.h>

namespace NYP::NServiceDiscovery {
    int TEndpointSetCache::DefaultFileMode() {
        return S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP | S_IROTH | S_IWOTH;
    }

    TFsPath ChooseDirPath(const TString& dir, const TEndpointSetKey& key) {
        if (key.FilePath) {
            return ""; //use FilePath in readonly mode
        } else if (dir) {
            return TFsPath(dir) / EndpointSetDirName(key);
        } else {
            return ""; //use in memory store
        }
    }

    TFsPath ChooseFilePath(const TFsPath& endpointSetDir, const TEndpointSetKey& key) {
        if (key.FilePath) {
            return key.FilePath;
        } else if (endpointSetDir.IsDefined()) {
            return endpointSetDir / "current";
        } else {
            return "";
        }
    }

    TEndpointSetCache::TEndpointSetCache(const TString& dir, const TEndpointSetKey& key, const TValidationOptions& validationOptions)
        : Key_(key)
        , DirPath_(ChooseDirPath(dir, key))
        , CacheFile_(ChooseFilePath(DirPath_, key))
        , ValidationOptions_(validationOptions)
    {
        if (DirPath_.IsDefined()) {
            try {
                ReadFromFile(DirPath_ / "applied");
                LastStored_ = CurrentData_;
                return;
            } catch (...) {
            }

            try {
                LoadFromFile();
                LastStored_ = CurrentData_;
            } catch (...) {
            }
        }
    }

    void ValidateEndpointSet(const NApi::TEndpointSet& endpointSet, const TValidationOptions& options, const TEndpointSetInfo& currentInfo) {
        if (!(options.AllowEmptyEndpointSetsOnStart && currentInfo.Version == 0)) {
            Y_ENSURE(endpointSet.endpoints_size() > 0, "empty endpointset");
        }

        for (ssize_t i = 0; i < endpointSet.endpoints_size(); ++i) {
            const auto& endpoint = endpointSet.endpoints(i);
            Y_ENSURE(endpoint.fqdn(), "empty fqdn");
            Y_ENSURE(endpoint.port() > 0, "zero port");
            Y_ENSURE(endpoint.ip4_address() || endpoint.ip6_address(), "no address");
        }
    }

    bool TEndpointSetCache::Store(const NApi::TEndpointSet& endpoints, const TEndpointSetInfo& info) {
        TStatEnv nullEnv;
        return Store(endpoints, info, nullEnv);
    }

    bool TEndpointSetCache::Store(const NApi::TEndpointSet& endpoints, const TEndpointSetInfo& info, TStatEnv& statEnv) {
        if (CurrentData_.Info.Version > 0 && NProtoBuf::IsEqual(LastStored_, endpoints) && NProtoBuf::IsEqual(CurrentData_, endpoints)) {
            return false;
        }

        TStringBuilder obsoleteTimestampInfo;
        if (info.yp_timestamp() < LastStored_.Info.yp_timestamp()) {
            obsoleteTimestampInfo << "too old yp_timestamp (last stored " << LastStored_.Info.yp_timestamp() << ", incoming " << info.yp_timestamp() << ")";
        } else if (info.yp_timestamp() < CurrentData_.Info.yp_timestamp()) {
            obsoleteTimestampInfo << "too old yp_timestamp (current " << CurrentData_.Info.yp_timestamp() << ", incoming " << info.yp_timestamp() << ")";
        }

        if (obsoleteTimestampInfo) {
            if (auto* stat = statEnv.CommonStat) {
                ++stat->ObsoleteTimestamp;
            }
            if (auto * stat = statEnv.EndpointSetStat) {
                ++stat->ObsoleteTimestamp;
            }

            if (auto* log = statEnv.Log) {
                (*log) << TLOG_INFO << "store cache [" << Key_.ToString() << "] " << obsoleteTimestampInfo;
            }

            return false;
        }

        if (auto* log = statEnv.Log) {
            (*log) << TLOG_INFO << "store cache [" << Key_.ToString() << "] " << info.ShortDebugString() << " " << endpoints.ShortDebugString();
        }

        if (auto * stat = statEnv.EndpointSetStat) {
            ++stat->CacheStoreCounter;
        }

        ValidateEndpointSet(endpoints, ValidationOptions_, CurrentData_.Info);

        LastStored_ = TEndpointSetEx{endpoints, info};
        HasChanges_ = true;

        if (!DirPath_.IsDefined()) {
            return true;
        }

        StoreToFile(endpoints, info, statEnv);

        return true;
    }

    void TEndpointSetCache::StoreToFile(const NApi::TEndpointSet& endpoints, const TEndpointSetInfo& info, TStatEnv& statEnv) {
        Y_VERIFY(CacheFile_.IsDefined() && CacheFile_.Parent() == DirPath_);

        CheckDir();

        TTempFile tmpFile(MakeTempName(DirPath_.c_str()));
        WriteFile(tmpFile.Name(), endpoints, info);
        Chmod(tmpFile.Name().c_str(), DefaultFileMode());

        if (IsOnlyReadinessChanged(CurrentData_, endpoints)) {
            // If only ready-flags were changed, just update top of the cache and prevent updating previous file.
            NFs::Rename(tmpFile.Name(), CacheFile_);
            return;
        }

        TTempFile prevTmpFile(MakeTempName(DirPath_.c_str()));
        bool haveNewPrevFile = false;

        try {
            // Readiness in prev file may become invalid if prev file moved manually.
            // So here we fix all endpoints to have set ready to true.

            NApi::TEndpointSet prevEs;
            TEndpointSetInfo prevInfo;
            ReadFile(CacheFile_.GetPath(), prevEs, prevInfo);

            for (auto& ep : *prevEs.mutable_endpoints()) {
                ep.set_ready(true);
            }

            WriteFile(prevTmpFile.Name(), prevEs, prevInfo);
            Chmod(prevTmpFile.Name().c_str(), DefaultFileMode());

            haveNewPrevFile = true;
        } catch (...) {
            if (CacheFile_.Exists()) {
                if (auto* log = statEnv.Log) {
                    (*log) << TLOG_ERR << "store cache [" << Key_.ToString() << "] can't make prev file " << CurrentExceptionMessage();
                }
            }
        }

        TFsPath prevFile = DirPath_ / "prev";

        NFs::Rename(tmpFile.Name(), CacheFile_);

        if (haveNewPrevFile) {
            NFs::Rename(prevTmpFile.Name(), prevFile);
        }
    }

    bool HasChanges(const TFileStat& f1, const TFileStat& f2) {
        return f1.MTime != f2.MTime || f1.CTime != f2.CTime || f1.INode != f2.INode || f1.Size != f2.Size;
    }

    void TEndpointSetCache::SetLastApplied(const TEndpointSetEx& endpointset) {
        if (NProtoBuf::IsEqual(endpointset, LastApplied_)) {
            return;
        }

        CheckDir();

        TTempFile tmpFile(MakeTempName(DirPath_.c_str()));
        WriteFile(tmpFile.Name(), endpointset, endpointset.Info);
        Chmod(tmpFile.Name().c_str(), DefaultFileMode());

        TFsPath appliedFile = DirPath_ / "applied";
        NFs::Rename(tmpFile.Name(), appliedFile);
        LastApplied_ = endpointset;
    }

    void TEndpointSetCache::LoadFromFile() {
        TFileStat stat(CacheFile_.GetPath());

        Y_ENSURE(!stat.IsNull(), "can't get file stat");

        if (!HasChanges(FileStat_, stat)) {
            Y_ENSURE(LastUpdateSucceeded_, "last update has failed");
            return;
        }

        LastUpdateSucceeded_ = false;
        FileStat_ = stat;

        ReadFromFile(CacheFile_);

        LastUpdateSucceeded_ = true;
    }

    void TEndpointSetCache::ReadFromFile(const TFsPath& path) {
        TEndpointSetEx tmp;
        ReadFile(path, tmp, tmp.Info);

        SortEndpointSet(&tmp);
        ValidateEndpointSet(tmp, ValidationOptions_, CurrentData_.Info);

        if (CurrentData_.Info.Version == 0 || !NProtoBuf::IsEqual(tmp, CurrentData_)) {
            const size_t prevVersion = CurrentData_.Info.Version;
            CurrentData_ = std::move(tmp);
            CurrentData_.Info.Version = prevVersion + 1;
        }
    }

    void TEndpointSetCache::LoadFromMemory() {
        if (!HasChanges_) {
            return;
        }

        const size_t prevVersion = CurrentData_.Info.Version;
        CurrentData_ = LastStored_;
        CurrentData_.Info.Version = prevVersion + 1;
        HasChanges_ = false;
    }

    void TEndpointSetCache::Load() {
        if (CacheFile_.IsDefined()) {
            try {
                LoadFromFile();
            } catch (...) {
                if (CurrentData_.Info.Version == 0) {
                    LoadFromMemory();
                }
                throw;
            }
        } else {
            LoadFromMemory();
        }
    }
}
