#include "endpointset.h"

#include <library/cpp/protobuf/util/pb_io.h>
#include <util/stream/output.h>

static constexpr ui32 RevisionDontIgnoreReadiness = 1;

namespace NYP::NServiceDiscovery {
    TString EndpointSetDirName(const TEndpointSetKey& key) {
        return key.cluster_name() + '#' + key.endpoint_set_id();
    }

    void ReadFile(const TString& filename, NApi::TEndpointSet& endpointSet, TEndpointSetInfo& info) {
        NYP::NServiceDiscovery::TEndpointSetFile fileData;
        ParseFromTextFormat(filename, fileData);
        endpointSet.Swap(fileData.mutable_endpoint_set());
        info.Swap(fileData.mutable_info());

        SortEndpointSet(&endpointSet);

        if (fileData.revision() < RevisionDontIgnoreReadiness) {
            for (int i = 0; i < endpointSet.mutable_endpoints()->size(); i++) {
                endpointSet.mutable_endpoints(i)->set_ready(true);
            }
        }
    }

    void WriteFile(const TString& filename, const NApi::TEndpointSet& endpointSet, const TEndpointSetInfo& info) {
        NYP::NServiceDiscovery::TEndpointSetFile fileData;
        *fileData.mutable_endpoint_set() = endpointSet;
        *fileData.mutable_info() = info;
        fileData.set_revision(RevisionDontIgnoreReadiness);
        SerializeToTextFormat(fileData, filename);
    }

    TString TEndpointSetKey::ToString() const {
        if (FilePath) {
            return FilePath;
        } else {
            return EndpointSetDirName(*this);
        }
    }

    void SortEndpointSet(NApi::TEndpointSet* es) {
        auto* v = es->mutable_endpoints();
        Sort(v->begin(), v->end(), [](const auto& a, const auto& b) {
            return std::tie(a.id(), a.fqdn()) < std::tie(b.id(), b.fqdn());
        });
    }

    bool EndpointsEqualExceptReady(const NApi::TEndpoint& a, const NApi::TEndpoint& b) {
        return a.id() == b.id() &&
            a.protocol() == b.protocol() &&
            a.fqdn() == b.fqdn() &&
            a.ip4_address() == b.ip4_address() &&
            a.ip6_address() == b.ip6_address() &&
            a.port() == b.port();
    }

    bool IsOnlyReadinessChanged(const NApi::TEndpointSet& prev, const NApi::TEndpointSet& next) {
        if (next.endpoints().size() != prev.endpoints().size()) {
            return false;
        }

        bool readinessChanged = false;
        for (int i = 0; i < next.endpoints().size(); i++) {
            if (!EndpointsEqualExceptReady(next.endpoints(i), prev.endpoints(i))) {
                return false;
            }

            if (next.endpoints(i).ready() != prev.endpoints(i).ready()) {
                readinessChanged = true;
            }
        }

        return readinessChanged;
    }
}
