#include "http_resolver.h"

#include <library/cpp/coroutine/engine/impl.h>
#include <library/cpp/coroutine/engine/network.h>

#include <library/cpp/http/io/stream.h>

#include <util/string/builder.h>
#include <util/stream/buffered.h>

namespace NYP::NServiceDiscovery {
    void THttpResolver::ResolveOne(const NApi::TReqResolveEndpoints& request, NApi::TRspResolveEndpoints& result) {
        class TDeadlineIO: public IInputStream, public IOutputStream {
        public:
            inline TDeadlineIO(SOCKET fd, TCont* cont, TInstant deadline)
                : Fd_(fd)
                , Cont_(cont)
                , Deadline_(deadline)
            {
            }

            void DoWrite(const void* buf, size_t len) override {
                auto r = NCoro::WriteD(Cont_, Fd_, buf, len, Deadline_);
                if (r.Status() == ETIMEDOUT) {
                    ythrow yexception() << "write timeout";
                }
                r.Checked();
            }

            size_t DoRead(void* buf, size_t len) override {
                auto r = NCoro::ReadD(Cont_, Fd_, buf, len, Deadline_);
                if (r.Status() == ETIMEDOUT) {
                    ythrow yexception() << "read timeout";
                }
                return r.Checked();
            }

            inline SOCKET Fd() const noexcept {
                return Fd_;
            }

        private:
            SOCKET Fd_;
            TCont* Cont_;
            TInstant Deadline_;
        };

        Y_ENSURE(NetworkAddress_);

        std::exception_ptr error;

        TString resultData;
        auto f = [this, &request, &resultData, &error](TCont* c) {
            try {
                TSocketHolder socket;
                TIOStatus(NCoro::ConnectT(c, socket, *NetworkAddress_, ConnectTimeout_)).Check();

                TDeadlineIO io(socket, c, Now() + RequestTimeout_);

                TBufferedOutput buffer(&io);
                THttpOutput httpO(&buffer);
                httpO.Write("POST /resolve_endpoints HTTP/1.1\r\n");

                TString tmp = request.SerializeAsStringOrThrow();
                httpO.Write("Content-Length: ");
                httpO.Write(ToString(tmp.size()));
                httpO.Write("\r\n\r\n");
                httpO.Write(tmp.data(), tmp.size());

                httpO.Finish();
                buffer.Finish();
                io.Finish();

                THttpInput httpI(&io);
                if (auto code = ParseHttpRetCode(httpI.FirstLine()); code != 200) {
                    ythrow yexception() << "non-200 http return code: " << code;
                }

                resultData = httpI.ReadAll();
            } catch (...) {
                error = std::current_exception();
            }
        };

        if (TCont* cont = RunningCont()) {
            f(cont);
        } else {
            TContExecutor e(128 * 1024);
            e.Create(f, "resolve_request");
            e.Execute();
        }

        if (error) {
            std::rethrow_exception(error);
        }

        result.ParseFromStringOrThrow(resultData);
    }

    void THttpResolver::DoResolve(const TResolveRequestBatch& request, TResolveResultBatch& result, TStatEnv& statEnv) {
        result.resize(request.size());
        for (size_t i = 0; i < request.size(); ++i) {
            if (auto* stat = statEnv.CommonStat) {
                ++stat->RemoteRequestsCounter;
            }

            try {
                ResolveOne(request[i], result[i].Result);
            } catch (...) {
                result[i].ErrorMessage = CurrentExceptionMessage();
                result[i].HasError = true;
            }
        }
    }
}
