#include "manager.h"

#include <util/random/random.h>
#include <util/system/thread.h>
#include <util/system/event.h>
#include <util/system/getpid.h>
#include <util/generic/algorithm.h>
#include <util/generic/vector.h>
#include <util/generic/scope.h>
#include <util/string/builder.h>
#include <util/folder/path.h>

namespace NYP::NServiceDiscovery {
    struct NPrivate::TSubscriber
        : IEndpointSetSubscriber
        , public TIntrusiveListItem<TSubscriber>
        , public IProviderHolder
    {
    public:
        size_t AppliedVersion = 0;

        TEndpointSetBundle* Bundle = 0;
    public:
        TSubscriber(TEndpointSetBundle& bundle, IProviderHolder p)
            : IProviderHolder(std::move(p))
            , Bundle(&bundle)
        {
        }

        ~TSubscriber();

        TEndpointSetEx GetEndpointSet() const override;
    };

    struct NPrivate::TEndpointSetBundle {
    public:
        TEndpointSetBundle(TEndpointSetManager& manager, const TString& dir, const TEndpointSetKey& key, const TEndpointSetOptions& opts, TCounterFactory f)
            : Key(key)
            , Options(opts)
            , Stat(key, f)
            , Cache(dir, key, manager.GetValidationOptions())
            , Manager_(manager)
        {
            SyncWithCache();
        }

        ~TEndpointSetBundle() {
            Y_UNUSED(Manager_);
            for (auto& s : Subscribers_) {
                s.Bundle = nullptr;
            }
        }
    public:

        const TEndpointSetKey Key;
        const TEndpointSetOptions Options;

        TEndpointSetManager::TEndpointSetStat Stat;

        TEndpointSetCache Cache;
        TEndpointSetEx Current;
        TMutex Lock;

        bool OnlyReadinessChangedSinceLastSync = false;

        IEndpointSetSubscriberRef Add(IProviderHolder p) {
            auto res = MakeHolder<TSubscriber>(*this, std::move(p));

            with_lock (Lock) {
                Subscribers_.PushBack(res.Get());
                LastSubscriberChange_ = TInstant::Now();
            }

            return res.Release();
        }

        void Remove(TSubscriber* removed) {
            with_lock (Lock) {
                removed->Unlink();
                LastSubscriberChange_ = TInstant::Now();
            }
        }

        template <typename T>
        void ForEach(T f) {
            with_lock (Lock) {
                for (auto& s : Subscribers_) {
                    f(s);
                }
            }
        }

        void SyncWithCache() {
            const TEndpointSetEx& cached = Cache.Get();
            if (cached.Info.Version == Current.Info.Version) {
                return;
            }

            const auto& skipEndpointSet = Options.GetSkipEndpoints();
            const i32 offsetPort = Options.GetOffsetPort();

            TEndpointSetEx resultEndpointSet = Cache.Get();
            if (!skipEndpointSet.empty() || offsetPort != 0) {
                TEndpointSetEx filteredEndpointSet = resultEndpointSet;
                filteredEndpointSet.clear_endpoints();

                for (int idx = 0; idx < resultEndpointSet.endpoints_size(); ++idx) {
                    NApi::TEndpoint endpoint = resultEndpointSet.endpoints(idx);
                    const TString endpointStr = TStringBuilder() << endpoint.fqdn() << ":" << endpoint.port();

                    if (::Find(skipEndpointSet.begin(), skipEndpointSet.end(), endpointStr) == skipEndpointSet.end()
                        && ::Find(skipEndpointSet.begin(), skipEndpointSet.end(), endpoint.fqdn()) == skipEndpointSet.end())
                    {
                        endpoint.set_port(endpoint.port() + offsetPort);
                        *(filteredEndpointSet.add_endpoints()) = endpoint;
                    }
                }

                resultEndpointSet = filteredEndpointSet;
            }

            OnlyReadinessChangedSinceLastSync = IsOnlyReadinessChanged(Current, resultEndpointSet);
            Current = std::move(resultEndpointSet);
        }

        bool IsNotUsedForAWhile(TInstant now, TDuration lifetime) const {
            with_lock (Lock) {
                return Subscribers_.Empty()
                    && (now - LastSubscriberChange_) > lifetime;
            }
        }

    private:
        TIntrusiveList<TSubscriber> Subscribers_;
        TEndpointSetManager& Manager_;
        TInstant LastSubscriberChange_;
    };

    using namespace NPrivate;

    TAtomicSharedPtr<IStatCounterImpl> MakeEndpointSetStatCounter(const TEndpointSetKey& key, TStringBuf name, TCounterFactory f) {
        if (key.FilePath) {
            return StandaloneCounterFactory("");
        } else {
            TString statName = "sd-" + key.cluster_name() + "-" + key.endpoint_set_id() + "-" + name;
            return f(statName);
        }
    }

    TEndpointSetManager::TEndpointSetStat::TEndpointSetStat(const TEndpointSetKey& key, TCounterFactory f)
        : RemoteErrors(MakeEndpointSetStatCounter(key, "remote_errors_summ", f))
        , CacheStoreErrors(MakeEndpointSetStatCounter(key, "cache_store_errors_summ", f))
        , CacheLoadErrors(MakeEndpointSetStatCounter(key, "cache_load_errors_summ", f))
        , ProviderUpdateErrors(MakeEndpointSetStatCounter(key, "update_errors_summ", f))
        , RequesterErrors(MakeEndpointSetStatCounter(key, "requester_errors_summ", f))
        , InvalidEndpointSetErrors(MakeEndpointSetStatCounter(key, "invalid_endpointset_errors_summ", f))
        , ObsoleteTimestamp(MakeEndpointSetStatCounter(key, "obsolete_timestamp_summ", f))
        , CacheStoreCounter(MakeEndpointSetStatCounter(key, "cache_store_summ", f))
        , ProviderUpdateCounter(MakeEndpointSetStatCounter(key, "update_summ", f))
        , YpTimestampMin(MakeEndpointSetStatCounter(key, "yp_timestamp_min_annn", f))
        , YpTimestampMax(MakeEndpointSetStatCounter(key, "yp_timestamp_max_axxx", f))
    {
    }

    TSubscriber::~TSubscriber() {
        if (Bundle) {
            Bundle->Remove(this);
        }
    }

    void CheckAvailability(TEndpointSetBundle& bundle) {
        Y_ENSURE(bundle.Current.Info.Version > 0, TStringBuilder() << "endpointset [" << bundle.Key.ToString() << "] unavailable");
    }

    TEndpointSetEx TSubscriber::GetEndpointSet() const {
        Y_ENSURE(Bundle, "try to access removed endpointset");

        with_lock (Bundle->Lock) {
            CheckAvailability(*Bundle);
            return Bundle->Current;
        }
    }

    TEndpointSetManager::TCommonStat::TCommonStat(TCounterFactory f)
        : UpdateCounter("sd-update_summ", f)
        , UpdateLoopCounter("sd-update_loop_summ", f)
        , UpdateSucceeded("sd-update_succ_summ", f)
        , UpdateLoopSucceeded("sd-update_loop_succ_summ", f)
        , UpdateFailed("sd-update_fail_summ", f)
        , UpdateLoopErrors("sd-update_loop_errors_summ", f)
        , RemoteErrors("sd-remote_errors_summ", f)
        , CacheStoreErrors("sd-cache_store_errors_summ", f)
        , CacheLoadErrors("sd-cache_load_errors_summ", f)
        , ProviderUpdateErrors("sd-update_errors_summ", f)
        , EndpointSetUpdateCounter("sd-endpointset_update_summ", f)
        , EndpointSetUpdateErrors("sd-endpointset_update_errors_summ", f)
        , RequesterErrors("sd-requester_errors_summ", f)
        , LogWriteErrors("sd-log_write_errors_summ", f)
        , InvalidEndpointSetErrors("sd-invalid_endpointset_errors_summ", f)
        , ObsoleteTimestamp("sd-obsolete_timestamp_summ", f)
        , UpdateLoopOverload("sd-update_loop_overload_summ", f)
        , RemoteRequestsCounter("sd-requests_summ", f)
    {
    }


    TEndpointSetManager::TUpdateMode::TUpdateMode() = default;

    struct TEndpointSetManager::TUpdater {
        THolder<ISimpleThread> Thread;
        TManualEvent Event;
        TAtomic Finished = 0;
    };

    TEndpointSetManager::TEndpointSetManager(const TString& cacheDir, IRemoteRequesterPtr rr, const TString& clientName, TCounterFactory counterFactory, TDuration endpointSetLifetime)
        : Updater_(MakeHolder<TUpdater>())
        , CacheDir_(cacheDir)
        , ClientName_(clientName)
        , Requester_(rr)
        , CounterFactory_(counterFactory)
        , Stat_(CounterFactory_)
        , EndpointSetLifetime_(endpointSetLifetime)
    {
        Y_ENSURE(clientName, "empty service discovery client name");

        if (CacheDir_) {
            TFsPath(CacheDir_).MkDir();
        }
    }

    TEndpointSetManager::~TEndpointSetManager() {
        Stop();
        Registered_.clear();
    }

    class TResolverThread : public ISimpleThread {
    public:
        TResolverThread(std::function<void()> f)
            : F_(std::move(f))
        {
        }

        void* ThreadProc() override {
            SetCurrentThreadName("sd_resolver");
            F_();
            return nullptr;
        }

    private:
        std::function<void()> F_;
    };

    void OnError(TLog& log, TStringBuf descr, const TEndpointSetKey& key, TStringBuf message) {
        log << TLOG_ERR << descr << " [" << key.ToString() << "] " << message;
    }

    void OnError(TLog& log, TStringBuf descr, const TEndpointSetKey& key) {
        OnError(log, descr, key, CurrentExceptionMessage());
    }

    void TEndpointSetManager::Start(TDuration updateInterval, TUpdateMode mode, TUpdateCallback callback) {
        Y_ENSURE(!AtomicGet(Updater_->Finished), "already finished");
        Y_ENSURE(!Updater_->Thread, "already started");

        if (Stat_.UpdateCounter == 0) {
            auto initMode = mode;
            initMode.Init = true;
            Update(initMode);
        }

        ForEach([](const TEndpointSetKey& key, TEndpointSetBundleRef bundle) {
            with_lock (bundle->Lock) {
                CheckAvailability(*bundle);
            }

            bundle->ForEach([&](const TSubscriber& s) {
                Y_ENSURE(s.Provider->GetActiveEndpointSetInfo().Version > 0 || s.AppliedVersion > 0, TStringBuilder() << "uninitialized provider for [" << key.ToString() << "] endpointset");
            });
        });

        Updater_->Thread = MakeHolder<TResolverThread>([this, updateInterval, mode, callback]() {
            Y_DEFER {
                Y_VERIFY(AtomicGet(Updater_->Finished));
            };

            Updater_->Event.WaitT(updateInterval * RandomNumber<double>());

            while (!AtomicGet(Updater_->Finished)) {
                const TInstant start = Now();
                const TInstant next = start + updateInterval;

                ++Stat_.UpdateCounter;
                ++Stat_.UpdateLoopCounter;
                try {
                    Update(mode);
                    ++Stat_.UpdateSucceeded;
                    ++Stat_.UpdateLoopSucceeded;
                } catch (...) {
                    ++Stat_.UpdateFailed;
                    ++Stat_.UpdateLoopErrors;
                    GetLog() << TLOG_ERR << "update loop error: " << CurrentExceptionMessage();
                }

                if (callback) {
                    try {
                        callback();
                    } catch (...) {
                        GetLog() << TLOG_ERR << "update loop callback failed: " << CurrentExceptionMessage();
                    }
                }

                if ((Now() - start) > updateInterval * 0.3) {
                    ++Stat_.UpdateLoopOverload;
                }

                Updater_->Event.WaitD(next);
            }
        });

        Updater_->Thread->Start();
    }

    namespace {
        void CheckSymbols(TStringBuf str) {
            for (const auto& c : str) {
                if (!IsAsciiAlpha(c) && !IsAsciiDigit(c) && !IsIn("-_.:", c)) {
                    Y_ENSURE(false, "unsupported symbol '" << c << "' in string " << str);
                }
            }
        }
    }

    void CheckKey(const TEndpointSetKey& key) {
        if (key.cluster_name() || key.endpoint_set_id()) {
            Y_ENSURE(key.FilePath.empty());
        } else {
            Y_ENSURE(!key.FilePath.empty());
        }

        CheckSymbols(key.cluster_name());
        CheckSymbols(key.endpoint_set_id());
    }

    IEndpointSetSubscriberRef TEndpointSetManager::SubscribeImpl(const TEndpointSetKey& key, IProviderHolder holder, const TEndpointSetOptions& opts) {
        CheckKey(key);

        auto& arena = EndpointSets_[ArenaHash(key)];

        with_lock (arena.Lock) {
            if (auto* p = arena.Map.FindPtr(key)) {
                // It's important to add a subscriber while holding the lock.
                // Otherwise this endpoint set bundle may get removed by periodic 'garbage collector'
                // (in case when its number of subscribers hits zero).
                return (*p)->Add(std::move(holder));
            }
        }

        TEndpointSetBundleRef bundleHolder = MakeAtomicShared<TEndpointSetBundle>(
            *this,
            opts.GetNoCacheOnDisk() ? TString{} : CacheDir_,
            key,
            opts,
            opts.GetUseUnistat() ? CounterFactory_ : StandaloneCounterFactory
        );

        with_lock (arena.Lock) {
            auto [it, _] = arena.Map.emplace(key, bundleHolder);
            // If `key` is already present, `it` points to the existing endpoint set bundle.
            return it->second->Add(std::move(holder));
        }
    }

    TEndpointSetBundleRef TEndpointSetManager::Find(const TEndpointSetKey& key) const {
        const auto& arena = EndpointSets_[ArenaHash(key)];
        with_lock (arena.Lock) {
            if (const auto* p = arena.Map.FindPtr(key)) {
                return *p;
            } else {
                return nullptr;
            }
        }
    }

    const TEndpointSetManager::TEndpointSetStat& TEndpointSetManager::GetStat(const TEndpointSetKey& key) const {
        TEndpointSetBundleRef bundle = Find(key);
        Y_ENSURE(bundle);
        return bundle->Stat;
    }

    void TEndpointSetManager::VerifyNotStarted() {
        Y_VERIFY(!Updater_->Thread);
    }

    void TEndpointSetManager::Stop() {
        if (!Updater_->Thread) {
            return;
        }

        AtomicSet(Updater_->Finished, 1);
        Updater_->Event.Signal();
        Updater_->Thread->Join();
    }

    void TEndpointSetManager::AssignLog(TLog* log) {
        Log_ = log;

        if (!Log_) {
            return;
        }

        class TWrapper: public TLogBackend {
        public:
            TWrapper(TCommonStat& stat, THolder<TLogBackend> backend)
                : Stat_(stat)
                , Backend_(std::move(backend))
            {
            }
        private:
            void WriteData(const TLogRecord& rec) override {
                try {
                    Backend_->WriteData(rec);
                } catch (...) {
                    ++Stat_.LogWriteErrors;
                }
            }
            void ReopenLog() override {
                Backend_->ReopenLog();
            }
            void ReopenLogNoFlush() override {
                Backend_->ReopenLogNoFlush();
            }
            ELogPriority FiltrationLevel() const override {
                return Backend_->FiltrationLevel();
            }
        private:
            TCommonStat& Stat_;
            THolder<TLogBackend> Backend_;
        };

        THolder<TLogBackend> wrapper = MakeHolder<TWrapper>(Stat_, Log_->ReleaseBackend());
        Log_->ResetBackend(std::move(wrapper));
        Log_->SetFormatter([](ELogPriority priority, TStringBuf data) {
            return TStringBuilder() << Now().ToStringLocal() << " " << ToString(priority) << " " << GetPID() << " " << data << Endl;
        });
    }

    TLog& TEndpointSetManager::GetLog() const {
        static TLog null;
        return Log_ ? *Log_ : null;
    }

    void TEndpointSetManager::Update(const TEndpointSetKey& key, TUpdateMode mode) {
        TEndpointSetBundleRef bundle = Find(key);

        Y_ENSURE(bundle);

        TVector<TEndpointSetBundleRef> endpointSets = {bundle};
        Update(endpointSets, mode);
    }

    void TEndpointSetManager::Update(TUpdateMode mode) {
        if (EndpointSetLifetime_ != TDuration::Max()) {
            RemoveUnusedBundles();
        }

        TVector<TEndpointSetBundleRef> endpointSets;
        ForEach([&](const TEndpointSetKey&, TEndpointSetBundleRef bundle) {
            endpointSets.push_back(bundle);
        });

        Update(endpointSets, mode);
    }

    void TEndpointSetManager::Update(const TVector<TEndpointSetBundleRef>& endpointSets, TUpdateMode mode) {
        if (!mode.ReadOnly && Requester_) {
            TVector<TEndpointSetBundle*> toResolve;
            for (auto& endpointSet : endpointSets) {
                if (endpointSet->Key.FilePath) {
                    continue;
                }

                if (mode.Init) {
                    with_lock (endpointSet->Lock) {
                        if (endpointSet->Cache.GetCurrentVersion()) {
                            continue;
                        }
                    }
                }

                toResolve.push_back(endpointSet.Get());
            }

            Resolve(toResolve);
        }

        Load(endpointSets, mode);

        for (auto& bundle : endpointSets) {
            ui64 minTs = Max<ui64>();
            ui64 maxTs = 0;

            size_t activeVersion = Max<size_t>();
            size_t appliedVersion = Max<size_t>();
            bundle->ForEach([&](const TSubscriber& s) {
                const auto active = s.Provider->GetActiveEndpointSetInfo();
                minTs = Min(minTs, active.yp_timestamp());
                maxTs = Max(maxTs, active.yp_timestamp());

                activeVersion = Min(activeVersion, active.Version);
                appliedVersion = Min(appliedVersion, s.AppliedVersion);
            });

            bundle->Stat.YpTimestampMin.Set(minTs);
            bundle->Stat.YpTimestampMax.Set(maxTs);

            const size_t last = activeVersion > 0 ? activeVersion : appliedVersion;
            try {
                with_lock (bundle->Lock) {
                    if (last == bundle->Cache.GetCurrentVersion()) {
                        bundle->Cache.SetLastApplied(bundle->Cache.Get());
                    }
                }
            } catch (...) {
                GetLog() << TLOG_ERR << "set last applied: " << CurrentExceptionMessage();
                ++bundle->Stat.CacheStoreErrors;
                ++Stat_.CacheStoreErrors;
            }
        }
    }

    static void SelectEndpointSetFields(NApi::TRspResolveEndpoints& rsp) {
        NApi::TEndpointSet eps;
        for (int i = 0; i < rsp.endpoint_set().endpoints_size(); ++i) {
            const NApi::TEndpoint& rhs = rsp.endpoint_set().endpoints(i);
            NApi::TEndpoint* lhs = eps.add_endpoints();
            lhs->set_id(rhs.id());
            lhs->set_protocol(rhs.protocol());
            lhs->set_fqdn(rhs.fqdn());
            lhs->set_ip4_address(rhs.ip4_address());
            lhs->set_ip6_address(rhs.ip6_address());
            lhs->set_port(rhs.port());
            lhs->set_ready(rhs.ready());
        }

        rsp.mutable_endpoint_set()->Swap(&eps);
    }

    static void ValidateResolvedEndpoints(const NApi::TRspResolveEndpoints& resolved, const TValidationOptions& options, const TEndpointSetInfo& currentInfo) {
        if (options.CheckForExistence) {
            Y_ENSURE(resolved.resolve_status() != NApi::EResolveStatus::NOT_EXISTS, "Endpoint set does not exist");
        }
        ValidateEndpointSet(resolved.endpoint_set(), options, currentInfo);
    }

    void TEndpointSetManager::Resolve(const TVector<TEndpointSetBundle*>& endpointSets) {
        if (endpointSets.empty()) {
            return;
        }

        GetLog() << TLOG_INFO << "resolve started";
        Y_ENSURE(Requester_);

        try {
            Requester_->Init();
        } catch (...) {
            GetLog() << TLOG_ERR << "requester init: " << CurrentExceptionMessage();
            for (auto* endpointSet : endpointSets) {
                ++endpointSet->Stat.RequesterErrors;
                ++Stat_.RequesterErrors;
            }

            return;
        }

        TResolveRequestBatch request;

        for (auto& b : endpointSets) {
            request.push_back(b->Key);
            request.back().set_client_name(ClientName_);
        }

        TResolveResultBatch result(request.size());

        TInstant now = Now();

        try {
            TStatEnv statEnv = {&Stat_, nullptr, Log_};
            Requester_->Resolve(request, result, statEnv);

            Y_ENSURE(result.size() == endpointSets.size());
        } catch (...) {
            GetLog() << TLOG_ERR << "requester: " << CurrentExceptionMessage();

            result.clear();
            for (auto* endpointSet : endpointSets) {
                ++endpointSet->Stat.RequesterErrors;
                ++Stat_.RequesterErrors;
            }
        }

        size_t stored = 0;
        for (size_t i = 0; i < result.size(); ++i) {
            if (!result[i].HasError) {
                auto& response = result[i].Result;


                try {
                    ValidateResolvedEndpoints(response, ValidationOptions_, endpointSets[i]->Current.Info);
                } catch (...) {
                    OnError(GetLog(), "invalid endpointset", endpointSets[i]->Key, CurrentExceptionMessage());
                    ++endpointSets[i]->Stat.InvalidEndpointSetErrors;
                    ++Stat_.InvalidEndpointSetErrors;

                    continue;
                }

                SelectEndpointSetFields(response);
                SortEndpointSet(response.mutable_endpoint_set());

                try {
                    TEndpointSetInfo info;
                    info.set_yp_timestamp(response.timestamp());
                    info.set_resolve_timestamp(now.MicroSeconds());

                    TStatEnv statEnv = {&Stat_, &endpointSets[i]->Stat, Log_};
                    with_lock (endpointSets[i]->Lock) {
                        if (endpointSets[i]->Cache.Store(response.endpoint_set(), info, statEnv)) {
                            ++stored;
                        }
                    }
                } catch (...) {
                    GetLog() << TLOG_INFO << "store cache [" << endpointSets[i]->Key.ToString() << "] " << response.ShortDebugString();
                    OnError(GetLog(), "cache store", endpointSets[i]->Key);
                    ++endpointSets[i]->Stat.CacheStoreErrors;
                    ++Stat_.CacheStoreErrors;
                }
            } else {
                OnError(GetLog(), "remote", endpointSets[i]->Key, result[i].ErrorMessage);
                ++endpointSets[i]->Stat.RemoteErrors;
                ++Stat_.RemoteErrors;
            }
        }
        GetLog() << TLOG_INFO << "resolve finished, " << endpointSets.size() << " " << result.size() << " " << stored;
    }

    bool TEndpointSetManager::ApplyCurrentEndpointSet(TSubscriber& s) {
        if (!s.Provider->AllowUpdate()) {
            return false;
        }

        TEndpointSetBundle& endpointSet = *s.Bundle;

        const auto currentVersion = endpointSet.Current.Info.Version;

        if (currentVersion == s.AppliedVersion || currentVersion == s.Provider->GetActiveEndpointSetInfo().Version) {
            return false;
        }

        ++endpointSet.Stat.ProviderUpdateCounter;
        ++Stat_.EndpointSetUpdateCounter;

        try {
            if (s.AppliedVersion + 1 == currentVersion && endpointSet.OnlyReadinessChangedSinceLastSync) {
                s.Provider->UpdateReadiness(endpointSet.Current);
            } else {
                s.Provider->Update(endpointSet.Current);
            }
            s.AppliedVersion = currentVersion;
            GetLog() << TLOG_INFO << "update endpoints [" << endpointSet.Key.ToString() << "] " << endpointSet.Current.ShortDebugString();
            return true;
        } catch (...) {
            OnError(GetLog(), "provider update", endpointSet.Key);
            ++endpointSet.Stat.ProviderUpdateErrors;
            ++Stat_.ProviderUpdateErrors;
            ++Stat_.EndpointSetUpdateErrors;
            return false;
        }
    }

    void TEndpointSetManager::Load(const TVector<TEndpointSetBundleRef>& endpointSets, TUpdateMode mode) {
        GetLog() << TLOG_INFO << "load started";
        size_t loaded = 0;
        for (auto& it : endpointSets) {
            auto& endpointSet = *it;

            with_lock (endpointSet.Lock) { //ensure consistent cache state across entire Load
                try {
                    if (!mode.Init || !endpointSet.Cache.GetCurrentVersion()) {
                        endpointSet.Cache.Load();
                    }
                } catch (...) {
                    OnError(GetLog(), "cache load", endpointSet.Key);
                    ++endpointSet.Stat.CacheLoadErrors;
                    ++Stat_.CacheLoadErrors;
                }

                endpointSet.SyncWithCache();

                endpointSet.ForEach([&](TSubscriber& s) {
                    if (ApplyCurrentEndpointSet(s)) {
                        ++loaded;
                    }
                });
            }
        }
        GetLog() << TLOG_INFO << "load finished, " << endpointSets.size() << " " << loaded;
    }

    size_t TEndpointSetManager::SubscriberCount() const {
        size_t res = 0;

        ForEach([&](const TEndpointSetKey&, TEndpointSetBundleRef b) {
            b->ForEach([&](TSubscriber&) {
                ++res;
            });
        });

        return res;
    }

    void TEndpointSetManager::RemoveUnusedBundles() {
        auto now = TInstant::Now();
        for (auto& arena : EndpointSets_) {
            with_lock (arena.Lock) {
                for (auto it = arena.Map.begin(); it != arena.Map.end(); ) {
                    if (it->second->IsNotUsedForAWhile(now, EndpointSetLifetime_)) {
                        arena.Map.erase(it++);
                    } else {
                        ++it;
                    }
                }
            }
        }
    }
}
