#pragma once

#include "cache.h"
#include "stat.h"
#include "subscribe.h"

#include <infra/yp_service_discovery/libs/sdlib/endpointset.pb.h>
#include <infra/yp_service_discovery/libs/sdlib/config/config.h>
#include <infra/yp_service_discovery/libs/sdlib/endpoint_set_options/endpoint_set_options.h>

#include <library/cpp/logger/log.h>

#include <util/generic/flags.h>
#include <util/generic/hash.h>
#include <util/datetime/base.h>
#include <util/system/mutex.h>
#include <util/generic/deque.h>

namespace NYP::NServiceDiscovery {
    struct TStatEnv;

    using TResolveRequestBatch = TVector<NApi::TReqResolveEndpoints>;

    struct TResolveResult {
        NApi::TRspResolveEndpoints Result;
        bool HasError = false;
        TString ErrorMessage;
    };

    using TResolveResultBatch = TVector<TResolveResult>;

    class IRemoteRequester {
    public:
        IRemoteRequester(TDuration initRetryDelay = TDuration::Minutes(5))
            : Delay_(initRetryDelay)
        {
        }

        virtual ~IRemoteRequester() {}

        void Init() {
            TInstant now = Now();
            with_lock (Lock_) {
                if (LastError_ && (now - PrevTry_) < Delay_) {
                    ythrow yexception() << "delay init, last error was " << LastError_;
                }

                if (!LastError_ && PrevTry_) {
                    return;
                }

                try {
                    PrevTry_ = now;
                    DoInit();
                    LastError_ = "";
                } catch (...) {
                    LastError_ = CurrentExceptionMessage();
                    throw;
                }
            }
        }

        void Resolve(const TResolveRequestBatch& request, TResolveResultBatch& result, TStatEnv& stat) {
            Y_ENSURE(!LastError_, "try to use uninitialized requester");
            DoResolve(request, result, stat);
        }
    private:
        virtual void DoInit() = 0;
        virtual void DoResolve(const TResolveRequestBatch& request, TResolveResultBatch& result, TStatEnv& stat) = 0;
    private:
        TMutex Lock_;
        TInstant PrevTry_;
        TString LastError_;
        const TDuration Delay_;
    };

    using IRemoteRequesterPtr = TAtomicSharedPtr<IRemoteRequester>;


    namespace NPrivate {
        struct TEndpointSetBundle;
        using TEndpointSetBundleRef = TAtomicSharedPtr<TEndpointSetBundle>;

        struct IProviderHolder {
        public:
            explicit IProviderHolder(THolder<IEndpointSetProvider> provider)
                : Holder_(std::move(provider))
                , Provider(Holder_.Get())
            {
            }

            explicit IProviderHolder(IEndpointSetProvider& provider)
                : Provider(&provider)
            {
            }
        private:
            THolder<IEndpointSetProvider> Holder_;
        public:
            IEndpointSetProvider* const Provider = 0;
        };

        struct TSubscriber;
    }



    class TEndpointSetManager {
    public:
        TEndpointSetManager(const TString& cacheDir, IRemoteRequesterPtr r, const TString& clientName, TCounterFactory f = StandaloneCounterFactory, TDuration endpointSetLifetime = TDuration::Max());

        TEndpointSetManager(IRemoteRequesterPtr r, const TString& clientName, TCounterFactory f = StandaloneCounterFactory, TDuration endpointSetLifetime = TDuration::Max())
            : TEndpointSetManager("", r, clientName, f, endpointSetLifetime)
        {
        }

        TEndpointSetManager()
            : TEndpointSetManager("", nullptr, "dummy_sd_client")
        {
        }

        TEndpointSetManager(const TSDConfig& config, IRemoteRequesterPtr r, TCounterFactory f = StandaloneCounterFactory)
            : TEndpointSetManager(config.GetCacheDir(), r, config.GetClientName(), f,
                                  !config.GetEndpointSetLifetime().empty() ? FromString<TDuration>(config.GetEndpointSetLifetime()) : TDuration::Max())
        {
            ValidationOptions_.AllowEmptyEndpointSetsOnStart = config.GetAllowEmptyEndpointSetsOnStart();
            ValidationOptions_.CheckForExistence = config.GetCheckForExistence();
        }

        virtual ~TEndpointSetManager();


        [[nodiscard]] IEndpointSetSubscriberRef Subscribe(const TEndpointSetKey& key, IEndpointSetProvider& provider, const TEndpointSetOptions& opts = {}) {
            NPrivate::IProviderHolder holder(provider);
            auto ref = SubscribeImpl(key, std::move(holder), opts);
            Init(key);

            ref->GetEndpointSet(); //ensure we have valid endpointset

            return ref;
        }

        template <typename T, typename... Args>
        T* RegisterEndpointSet(const TEndpointSetKey& key, Args&&... args) {
            return RegisterEndpointSetEx<T>(key, TEndpointSetOptions(), std::forward<Args>(args)...);
        }

        template <typename T, typename... Args>
        T* RegisterEndpointSetEx(const TEndpointSetKey& key, const TEndpointSetOptions& opts, Args&&... args) {
            VerifyNotStarted();

            auto provider = MakeHolder<T>(std::forward<Args>(args)...);
            T* res = provider.Get();

            NPrivate::IProviderHolder holder(std::move(provider));
            auto ref = SubscribeImpl(key, std::move(holder), opts);
            Registered_.push_back(ref);

            return res;
        }

        void Init(const TEndpointSetKey& key) {
            TUpdateMode mode;
            mode.Init = true;
            Update(key, mode);
        }


        struct TUpdateMode {
            TUpdateMode();
        public:
            bool ReadOnly = false;
            bool Init = false;
        };

        using TUpdateCallback = std::function<void()>;
        void Start(TDuration updateInterval, TUpdateMode mode = {}, TUpdateCallback callback = {});
        void Stop();


        void AssignLog(TLog* log);


        struct TCommonStat {
        public:
            explicit TCommonStat(TCounterFactory f);
        public:
            TStatCounter UpdateCounter;
            TStatCounter UpdateLoopCounter;
            TStatCounter UpdateSucceeded;
            TStatCounter UpdateLoopSucceeded;
            TStatCounter UpdateFailed;
            TStatCounter UpdateLoopErrors;

            TStatCounter RemoteErrors;
            TStatCounter CacheStoreErrors;
            TStatCounter CacheLoadErrors;
            TStatCounter ProviderUpdateErrors;
            TStatCounter EndpointSetUpdateCounter;
            TStatCounter EndpointSetUpdateErrors;
            TStatCounter RequesterErrors;
            TStatCounter LogWriteErrors;
            TStatCounter InvalidEndpointSetErrors;

            TStatCounter ObsoleteTimestamp;
            TStatCounter UpdateLoopOverload;
            TStatCounter RemoteRequestsCounter;

        };

        const TCommonStat& GetStat() const {
            return Stat_;
        }


        struct TEndpointSetStat {
        public:
            TEndpointSetStat(const TEndpointSetKey& key, TCounterFactory f);
        public:
            TStatCounter RemoteErrors;
            TStatCounter CacheStoreErrors;
            TStatCounter CacheLoadErrors;
            TStatCounter ProviderUpdateErrors;
            TStatCounter RequesterErrors;
            TStatCounter InvalidEndpointSetErrors;

            TStatCounter ObsoleteTimestamp;

            TStatCounter CacheStoreCounter;
            TStatCounter ProviderUpdateCounter;

            TStatCounter YpTimestampMin;
            TStatCounter YpTimestampMax;
        };

        const TEndpointSetStat& GetStat(const TEndpointSetKey& key) const;



        size_t SubscriberCount() const;

        void Update(TUpdateMode mode = {});
        void Update(const TEndpointSetKey& key, TUpdateMode mode = {});

        const TValidationOptions& GetValidationOptions() const {
            return ValidationOptions_;
        }

    private:
        NPrivate::TEndpointSetBundleRef Find(const TEndpointSetKey& key) const;

        template <typename T>
        void ForEach(T f) const {
            TVector<std::pair<const TEndpointSetKey, NPrivate::TEndpointSetBundleRef>> tmp;
            for (auto& arena : EndpointSets_) {
                tmp.clear();
                with_lock (arena.Lock) {
                    tmp.reserve(arena.Map.size());
                    for (auto it : arena.Map) {
                        tmp.push_back(it);
                    }
                }

                for (const auto& item : tmp) {
                    f(item.first, item.second);
                }
            }
        }

        void VerifyNotStarted();

        void Update(const TVector<NPrivate::TEndpointSetBundleRef>& endpointSets, TUpdateMode mode = {});
        void Resolve(const TVector<NPrivate::TEndpointSetBundle*>& endpointSets);
        void Load(const TVector<NPrivate::TEndpointSetBundleRef>& endpointSets, TUpdateMode mode);

        bool ApplyCurrentEndpointSet(NPrivate::TSubscriber& subscriber);

        IEndpointSetSubscriberRef SubscribeImpl(const TEndpointSetKey& key, NPrivate::IProviderHolder holder, const TEndpointSetOptions& opts);

        TLog& GetLog() const;

        void RemoveUnusedBundles();
    private:
        static const size_t ARENA_COUNT = 16;

        struct TArena {
            THashMap<TEndpointSetKey, NPrivate::TEndpointSetBundleRef, TEndpointSetKey::THash, TEndpointSetKey::TEqual> Map;
            TMutex Lock;
        } EndpointSets_[ARENA_COUNT];

        size_t ArenaHash(const TEndpointSetKey& key) const {
            return IntHash(TEndpointSetKey::THash()(key)) % ARENA_COUNT;
        }


        struct TUpdater;
        THolder<TUpdater> Updater_;

        const TString CacheDir_;
        const TString ClientName_;
        IRemoteRequesterPtr Requester_;

        TCounterFactory CounterFactory_;
        TCommonStat Stat_;

        TLog* Log_ = nullptr;

        TDeque<IEndpointSetSubscriberRef> Registered_;

        TValidationOptions ValidationOptions_;
        TDuration EndpointSetLifetime_;
    };

    struct TStatEnv {
        TEndpointSetManager::TCommonStat* CommonStat = nullptr;
        TEndpointSetManager::TEndpointSetStat* EndpointSetStat = nullptr;
        TLog* Log = nullptr;
    };
}
