#include "cache.h"
#include "manager.h"

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/env.h>

#include <util/system/fs.h>
#include <util/generic/algorithm.h>
#include <util/generic/scope.h>

using namespace NYP::NServiceDiscovery::NApi;
using namespace NYP::NServiceDiscovery;
using namespace NFs;

Y_UNIT_TEST_SUITE(TCache) {
    void CheckDirFiles(const TFsPath& path, const TVector<TString> fileNames) {
        UNIT_ASSERT(path.Exists() && path.IsDirectory());

        TVector<TString> expected(fileNames);
        Sort(expected.begin(), expected.end());

        TVector<TString> actual;
        path.ListNames(actual);
        Sort(actual.begin(), actual.end());

        UNIT_ASSERT_VALUES_EQUAL(actual, expected);

        for (const TString& f : actual) {
            UNIT_ASSERT_VALUES_EQUAL(TFileStat(path / f).Mode & MODE0777, TEndpointSetCache::DefaultFileMode());
        }
    }

    const TString dataPath = ArcadiaSourceRoot() + "/infra/yp_service_discovery/libs/sdlib/ut";

    const TFsPath cacheDir = "./cache";

    Y_UNIT_TEST(TestStoreAndLoad) {
        RemoveRecursive(cacheDir);

        UNIT_ASSERT(!Exists(cacheDir));
        cacheDir.MkDir();
        UNIT_ASSERT(Exists(cacheDir));

        TEndpointSetKey key("test-cluster", "test-service");
        TEndpointSetCache cache(cacheDir, key);

        TEndpointSetEx eps;
        ReadFile(dataPath + "/correct_endpointset", eps, eps.Info);
        cache.Store(eps, eps.Info);

        CheckDirFiles(cache.GetEndpointSetDir(), {"current"});

        CheckDirFiles(cache.GetEndpointSetDir(), {"current"});
        cache.Load();
        TEndpointSetEx eps2 = cache.Get();

        UNIT_ASSERT_VALUES_EQUAL(eps2.DebugString(), eps.DebugString());

        cache.SetLastApplied(eps2);

        CheckDirFiles(cache.GetEndpointSetDir(), {"applied", "current"});

        {
            TEndpointSetCache cache(cacheDir, key);
            CheckDirFiles(cache.GetEndpointSetDir(), {"applied", "current"});

            TEndpointSet eps3 = cache.Get();
            UNIT_ASSERT_VALUES_EQUAL(eps3.DebugString(), eps.DebugString());

            cache.Store(eps, eps.Info);
            CheckDirFiles(cache.GetEndpointSetDir(), {"applied", "current"});
        }
    }

    Y_UNIT_TEST(TestApplied) {
        RemoveRecursive(cacheDir);

        UNIT_ASSERT(!Exists(cacheDir));
        cacheDir.MkDir();
        UNIT_ASSERT(Exists(cacheDir));

        TEndpointSetKey key("test-cluster", "test-service");
        TEndpointSetCache cache(cacheDir, key);

        TEndpointSetEx eps;
        ReadFile(dataPath + "/correct_endpointset", eps, eps.Info);
        cache.Store(eps, eps.Info);
        cache.SetLastApplied(eps);
        CheckDirFiles(cache.GetEndpointSetDir(), {"applied", "current"});

        auto port1 = eps.endpoints(0).port();
        auto port2 = port1 + 1;

        eps.mutable_endpoints(0)->set_port(port2);
        cache.Store(eps, eps.Info);
        CheckDirFiles(cache.GetEndpointSetDir(), {"applied", "current", "prev"});

        {//restore cache from "applied" file first
            TEndpointSetCache cache(cacheDir, key);
            TEndpointSetEx eps = cache.Get();
            UNIT_ASSERT(eps.endpoints_size() > 0);
            UNIT_ASSERT_VALUES_EQUAL(eps.endpoints(0).port(), port1);
        }

        Remove(cache.GetEndpointSetDir() / "applied");
        CheckDirFiles(cache.GetEndpointSetDir(), {"current", "prev"});

        {//if there is no "applied" -- use "current"
            TEndpointSetCache cache(cacheDir, key);
            TEndpointSetEx eps = cache.Get();
            UNIT_ASSERT(eps.endpoints_size() > 0);
            UNIT_ASSERT_VALUES_EQUAL(eps.endpoints(0).port(), port2);
        }
    }

    Y_UNIT_TEST(TestStoreAndLoadInMem) {
        RemoveRecursive(cacheDir);

        TEndpointSetKey key("test-cluster", "test-service");
        TEndpointSetCache cache("", key);

        UNIT_ASSERT(!cache.GetEndpointSetDir().Exists());

        TEndpointSet eps;
        TEndpointSetInfo info;
        ReadFile(dataPath + "/correct_endpointset", eps, info);
        cache.Store(eps, info);

        cache.Load();
        TEndpointSet eps2 = cache.Get();

        UNIT_ASSERT(eps2.DebugString() == eps.DebugString());

        {
            TEndpointSetCache cache("", key);
            cache.Load();
            TEndpointSet eps3 = cache.Get();
            UNIT_ASSERT(eps3.endpoints_size() == 0);
        }
    }

    Y_UNIT_TEST(TestRewrite1) {
        RemoveRecursive(cacheDir);

        UNIT_ASSERT(!Exists(cacheDir));
        cacheDir.MkDir();
        UNIT_ASSERT(Exists(cacheDir));

        TEndpointSetKey key("test-cluster", "test-service");
        TEndpointSetCache cache(cacheDir, key);


        TEndpointSet eps1;
        TEndpointSetInfo info;
        ReadFile(dataPath + "/correct_endpointset", eps1, info);
        cache.Store(eps1, info);

        CheckDirFiles(cache.GetEndpointSetDir(), {"current"});

        {
            cache.Load();
            TEndpointSet eps = cache.Get();
            UNIT_ASSERT_VALUES_EQUAL(eps.DebugString(), eps1.DebugString());
        }

        cache.Store(eps1, info);
        CheckDirFiles(cache.GetEndpointSetDir(), {"current"});

        TEndpointSet eps2;
        auto* endpoint = eps2.add_endpoints();
        endpoint->set_fqdn("fqdn");
        endpoint->set_port(8080);
        endpoint->set_ip6_address("::1");


        cache.Store(eps2, info);
        CheckDirFiles(cache.GetEndpointSetDir(), {"current", "prev"});

        {
            cache.Load();
            TEndpointSet eps = cache.Get();
            UNIT_ASSERT_VALUES_EQUAL(eps.DebugString(), eps2.DebugString());
        }

        cache.Store(eps1, info);

        CheckDirFiles(cache.GetEndpointSetDir(), {"current", "prev"});
        {
            cache.Load();
            TEndpointSet eps = cache.Get();
            UNIT_ASSERT_VALUES_EQUAL(eps.DebugString(), eps1.DebugString());
        }
    }

    Y_UNIT_TEST(TestRewrite2) {
        RemoveRecursive(cacheDir);

        UNIT_ASSERT(!Exists(cacheDir));
        cacheDir.MkDir();
        UNIT_ASSERT(Exists(cacheDir));

        TEndpointSetKey key("test-cluster", "test-service");
        TEndpointSetCache cache(cacheDir, key);


        TEndpointSet eps1;
        TEndpointSetInfo info;
        ReadFile(dataPath + "/correct_endpointset", eps1, info);
        cache.Store(eps1, info);

        CheckDirFiles(cache.GetEndpointSetDir(), {"current"});
        {
            cache.Load();
            TEndpointSet eps = cache.Get();
            UNIT_ASSERT(eps.DebugString() == eps1.DebugString());
        }

        TEndpointSet eps2;
        auto* endpoint = eps2.add_endpoints();
        endpoint->set_fqdn("fqdn");
        endpoint->set_port(8080);
        endpoint->set_ip6_address("::1");

        WriteFile(cache.GetCacheFile(), eps2, info);

        {
            cache.Load();
            TEndpointSet eps = cache.Get();
            UNIT_ASSERT(eps.DebugString() == eps2.DebugString());
        }

        cache.Store(eps1, info);

        CheckDirFiles(cache.GetEndpointSetDir(), {"current", "prev"});
        {
            cache.Load();
            TEndpointSet eps = cache.Get();
            UNIT_ASSERT(eps.DebugString() == eps1.DebugString());
        }
    }

    Y_UNIT_TEST(TestEmptyForbid) {
        TEndpointSetKey key("test-cluster", "test-service");
        TEndpointSet eps;
        TEndpointSetInfo info;
        TEndpointSetCache cache("", key);
        UNIT_ASSERT_EXCEPTION(cache.Store(eps, info), yexception);
    }

    Y_UNIT_TEST(TestAllowEmptyOnStart) {
        RemoveRecursive(cacheDir);

        UNIT_ASSERT(!Exists(cacheDir));
        cacheDir.MkDir();
        UNIT_ASSERT(Exists(cacheDir));

        TEndpointSetKey key("test-cluster", "test-service");
        TEndpointSet eps1;
        TEndpointSetInfo info;
        TEndpointSetCache cache(cacheDir, key, {.AllowEmptyEndpointSetsOnStart = true});

        cache.Store(eps1, info);

        CheckDirFiles(cache.GetEndpointSetDir(), {"current"});
        {
            cache.Load();
            TEndpointSet eps = cache.Get();
            UNIT_ASSERT(eps.DebugString() == eps1.DebugString());
        }

        TEndpointSet eps2;
        auto* endpoint = eps2.add_endpoints();
        endpoint->set_fqdn("fqdn");
        endpoint->set_port(8080);
        endpoint->set_ip6_address("::1");

        cache.Store(eps2, info);

        {
            cache.Load();
            TEndpointSet eps = cache.Get();
            UNIT_ASSERT(eps.DebugString() == eps2.DebugString());
        }

        UNIT_ASSERT_EXCEPTION(cache.Store(eps1, info), yexception);

        {
            cache.Load();
            TEndpointSet eps = cache.Get();
            UNIT_ASSERT(eps.DebugString() == eps2.DebugString());
        }
    }

    class TTestStatEnv
        : public TStatEnv
    {
    public:
        TTestStatEnv(const TEndpointSetKey& key)
            : CommonStat_(StandaloneCounterFactory)
            , EndpointSetStat_(key, StandaloneCounterFactory)
        {
            CommonStat = &CommonStat_;
            EndpointSetStat = &EndpointSetStat_;
        }
    private:
        TEndpointSetManager::TCommonStat CommonStat_;
        TEndpointSetManager::TEndpointSetStat EndpointSetStat_;
    };

    Y_UNIT_TEST(TestExpiredTimestamp1) {
        TEndpointSetKey key("test-cluster", "test-service");
        TEndpointSet eps;
        TEndpointSetInfo info;

        info.set_yp_timestamp(123);

        auto* endpoint = eps.add_endpoints();
        endpoint->set_fqdn("fqdn");
        endpoint->set_port(8080);
        endpoint->set_ip6_address("::1");

        TEndpointSetCache cache("", key);
        cache.Store(eps, info);

        info.set_yp_timestamp(122);
        endpoint->set_port(8081);

        TTestStatEnv statEnv(key);
        cache.Store(eps, info, statEnv);
        UNIT_ASSERT(statEnv.CommonStat->ObsoleteTimestamp == 1);
        UNIT_ASSERT(statEnv.EndpointSetStat->CacheStoreCounter == 0);
        UNIT_ASSERT(statEnv.EndpointSetStat->ObsoleteTimestamp == 1);

        cache.Load();
        UNIT_ASSERT(cache.Get().endpoints(0).port() == 8080);
    }

    Y_UNIT_TEST(TestExpiredTimestamp2) {
        RemoveRecursive(cacheDir);

        UNIT_ASSERT(!Exists(cacheDir));
        cacheDir.MkDir();
        UNIT_ASSERT(Exists(cacheDir));

        TEndpointSetKey key("test-cluster", "test-service");
        TEndpointSet eps;
        TEndpointSetInfo info;

        info.set_yp_timestamp(123);

        auto* endpoint = eps.add_endpoints();
        endpoint->set_fqdn("fqdn");
        endpoint->set_port(8080);
        endpoint->set_ip6_address("::1");

        {
            TEndpointSetCache cache(cacheDir, key);
            cache.Store(eps, info);
        }

        TEndpointSetCache cache(cacheDir, key);

        info.set_yp_timestamp(122);
        endpoint->set_port(8081);

        TTestStatEnv statEnv(key);
        cache.Store(eps, info, statEnv);
        UNIT_ASSERT(cache.Get().endpoints(0).port() == 8080);
        UNIT_ASSERT(statEnv.CommonStat->ObsoleteTimestamp == 1);
        UNIT_ASSERT(statEnv.EndpointSetStat->CacheStoreCounter == 0);
        UNIT_ASSERT(statEnv.EndpointSetStat->ObsoleteTimestamp == 1);
    }

    Y_UNIT_TEST(TestCacheIgnoresReadinessInPrevState) {
        RemoveRecursive(cacheDir);
        cacheDir.MkDir();

        TEndpointSetKey key("test-cluster", "test-service");
        TEndpointSet eps;
        TEndpointSetInfo info;

        info.set_yp_timestamp(123);

        TEndpointSetCache cache{cacheDir, key};
        auto* endpoint = eps.add_endpoints();
        endpoint->set_fqdn("fqdn0");
        endpoint->set_port(8080);
        endpoint->set_ip6_address("::1");
        endpoint->set_ready(false);
        cache.Store(eps, info);

        for (int i = 0; i < 10; i++) {
            auto* endpoint = eps.add_endpoints();
            endpoint->set_fqdn(TStringBuilder{} << "fqdn" << i + 1);
            endpoint->set_port(8081 + i);
            endpoint->set_ip6_address(TStringBuilder{} << "::" << i + 2);
            endpoint->set_ready(false);
        }
        cache.Store(eps, info);

        endpoint = eps.add_endpoints();
        endpoint->set_fqdn("fqdn11");
        endpoint->set_port(8091);
        endpoint->set_ip6_address("::12");
        endpoint->set_ready(false);
        cache.Store(eps, info);

        CheckDirFiles(cache.GetEndpointSetDir(), {"current", "prev"});
        Rename(cache.GetEndpointSetDir() / "prev", cache.GetEndpointSetDir() / "current");

        cache.Load();
        UNIT_ASSERT(cache.Get().endpoints().size() == 11);
        for (int i = 0; i < 11; i++) {
            UNIT_ASSERT(cache.Get().endpoints(i).ready());
        }
    }
}
