#include "endpointset.h"

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/env.h>

#include <util/stream/file.h>
#include <util/system/fs.h>

using namespace NYP::NServiceDiscovery::NApi;
using namespace NYP::NServiceDiscovery;

Y_UNIT_TEST_SUITE(TEndpointset) {
    const TString dataPath = ArcadiaSourceRoot() + "/infra/yp_service_discovery/libs/sdlib/ut";

    Y_UNIT_TEST(TestRead) {
        TEndpointSet eps;
        ReadFile(dataPath + "/correct_endpointset", eps);
        UNIT_ASSERT_VALUES_EQUAL(eps.endpoints_size(), 2);
        UNIT_ASSERT_VALUES_EQUAL(eps.endpoints(0).fqdn(), "electra.search.yandex.net");
        UNIT_ASSERT_VALUES_EQUAL(eps.endpoints(0).port(), 25055);
        UNIT_ASSERT_VALUES_EQUAL(eps.endpoints(1).fqdn(), "minos.search.yandex.net");
        UNIT_ASSERT_VALUES_EQUAL(eps.endpoints(1).port(), 25055);
    }

    Y_UNIT_TEST(TestReadFail) {
        try {
            TEndpointSet eps;
            ReadFile(dataPath + "/broken_endpointset", eps);
            UNIT_ASSERT(false);
        } catch (...) {
        }
    }

    Y_UNIT_TEST(TestReadSortsEndpoint) {
        TEndpointSet eps;
        ReadFile(dataPath + "/unsorted_endpointset", eps);
        UNIT_ASSERT_VALUES_EQUAL(eps.endpoints_size(), 4);
        UNIT_ASSERT_VALUES_EQUAL(eps.endpoints(0).fqdn(), "electra.search.yandex.net");
        UNIT_ASSERT_VALUES_EQUAL(eps.endpoints(1).fqdn(), "irbis.search.yandex.net");
        UNIT_ASSERT_VALUES_EQUAL(eps.endpoints(2).fqdn(), "minos.search.yandex.net");
        UNIT_ASSERT_VALUES_EQUAL(eps.endpoints(3).fqdn(), "pushkin.sas.yp-c.yandex.net");
    }

    Y_UNIT_TEST(TestWrite) {
        const TString fileName = "f1";

        NFs::Remove(fileName);
        TEndpointSet eps;
        TEndpointSetInfo info;

        ReadFile(dataPath + "/correct_endpointset", eps, info);

        WriteFile(fileName, eps, info);

        UNIT_ASSERT_VALUES_EQUAL(TFileInput(fileName).ReadAll(), TFileInput(dataPath + "/correct_endpointset").ReadAll());
    }

    Y_UNIT_TEST(TestIsOnlyReadinessChanged) {
        NApi::TEndpointSet a;
        auto* ep = a.add_endpoints();
        ep->set_id("ep1");
        ep->set_fqdn("localhost");
        ep->set_ip4_address("127.0.0.1");
        ep->set_port(123);
        ep->set_ready(true);

        NApi::TEndpointSet b;
        ep = b.add_endpoints();
        ep->set_id("ep1");
        ep->set_fqdn("localhost");
        ep->set_ip4_address("127.0.0.1");
        ep->set_port(123);
        ep->set_ready(true);

        a.mutable_endpoints(0)->set_ready(false);
        b.mutable_endpoints(0)->set_ready(true);
        UNIT_ASSERT(IsOnlyReadinessChanged(a, b));

        a.mutable_endpoints(0)->set_ready(true);
        b.mutable_endpoints(0)->set_ready(false);
        UNIT_ASSERT(IsOnlyReadinessChanged(a, b));

        a.mutable_endpoints(0)->set_id("ep2");
        UNIT_ASSERT(!IsOnlyReadinessChanged(a, b));
        a.mutable_endpoints(0)->set_id("ep1");

        ep = b.add_endpoints();
        ep->set_id("ep2");
        ep->set_fqdn("localhost");
        ep->set_ip4_address("127.0.0.1");
        ep->set_port(1234);
        ep->set_ready(true);
        UNIT_ASSERT(!IsOnlyReadinessChanged(a, b));
    }
}
