#pragma once

#include "manager.h"

#include <library/cpp/testing/unittest/registar.h>
#include <util/system/mutex.h>

namespace NYP::NServiceDiscovery {
    const inline TString TestClientName = "sdlib/ut";

    template <typename F>
    void Wait(F f) {
        size_t limit = 10;
        while (!f() && --limit) {
            Sleep(TDuration::MilliSeconds(100));
        }

        UNIT_ASSERT(limit);
    }

    inline
    void WaitUpdate(const TEndpointSetManager::TCommonStat& stat) {
        const auto prev = stat.UpdateLoopSucceeded.Get();
        Wait([prev, &stat]() {
            return stat.UpdateLoopSucceeded > prev + 1;
        });
    }

    inline void WaitUpdate(TEndpointSetManager& manager) {
        return WaitUpdate(manager.GetStat());
    }

    inline NApi::TEndpoint& AddEndpoint(TResolveResult& r) {
        auto& endpoint = *r.Result.mutable_endpoint_set()->add_endpoints();
        endpoint.set_fqdn("dummy_fqdn");
        endpoint.set_port(80);
        endpoint.set_ip6_address("::1");
        return endpoint;
    }

    extern IRemoteRequesterPtr DummyRequester;

    class TProvider : public IEndpointSetProvider {
    public:
        TProvider(std::function<void(const NApi::TEndpointSet&)> update = {}, std::function<void(const NApi::TEndpointSet&)> updateReadiness = {})
            : Update_(std::move(update))
            , UpdateReadiness_(std::move(updateReadiness))
        {
        }

        void Update(const TEndpointSetEx& endpointSet) override {
            if (Update_) {
                Update_(endpointSet);
            }
            SetEndpointSet(endpointSet);
        };

        void UpdateReadiness(const TEndpointSetEx& endpointSet) override {
            if (UpdateReadiness_) {
                UpdateReadiness_(endpointSet);
            }
        }

        TEndpointSetEx GetEndpointSet() const {
            with_lock (Mutex_) {
                return EndpointSet_;
            }
        }

        void SetEndpointSet(const TEndpointSetEx& endpointSet) {
            with_lock (Mutex_) {
                EndpointSet_ = endpointSet;
                if (UpdateActiveEndpointSetInfo) {
                    SetActiveEndpointSetInfo(endpointSet.Info);
                }
            }
        }

        bool UpdateActiveEndpointSetInfo = false;
    private:
        std::function<void(const NApi::TEndpointSet&)> Update_;
        std::function<void(const NApi::TEndpointSet&)> UpdateReadiness_;
        TMutex Mutex_;
        TEndpointSetEx EndpointSet_;
    };
}
