#include "manager.h"
#include "helpers.h"
#include "resolver.h"

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/env.h>

#include <util/generic/algorithm.h>
#include <util/system/fs.h>

using namespace NYP::NServiceDiscovery;

Y_UNIT_TEST_SUITE(TEndpointSetManager) {
    Y_UNIT_TEST(TestRegister) {
        TEndpointSetManager manager(DummyRequester, TestClientName);

        TEndpointSetKey key1("test-cluster", "test-service");
        TEndpointSetKey key2("test-cluster", "test-service");

        UNIT_ASSERT_VALUES_EQUAL(manager.SubscriberCount(), 0);
        TProvider* p1 = manager.RegisterEndpointSet<TProvider>(key1);
        UNIT_ASSERT(p1);
        UNIT_ASSERT_VALUES_EQUAL(manager.SubscriberCount(), 1);
        TProvider* p2 = manager.RegisterEndpointSet<TProvider>(key2);
        UNIT_ASSERT(p2);
        UNIT_ASSERT_VALUES_EQUAL(manager.SubscriberCount(), 2);
        UNIT_ASSERT(p1 != p2);
        UNIT_ASSERT(&manager.GetStat(key1) == &manager.GetStat(key2));
    }

    Y_UNIT_TEST(TestNames) {
        TEndpointSetManager manager(DummyRequester, TestClientName);

        {
            TEndpointSetKey key("Aa-_:.", "123");
            UNIT_ASSERT(manager.RegisterEndpointSet<TProvider>(key));
        }

        {
            TEndpointSetKey key("a/b/c", "123");
            UNIT_ASSERT_EXCEPTION(manager.RegisterEndpointSet<TProvider>(key), yexception);
        }

        {
            TEndpointSetKey key("cluster", "!?*");
            UNIT_ASSERT_EXCEPTION(manager.RegisterEndpointSet<TProvider>(key), yexception);
        }
    }


    Y_UNIT_TEST(TestUpdate) {
        TEndpointSetKey key1("test-cluster", "test-service1");
        TEndpointSetKey key2("test-cluster", "test-service2");

        TAtomic requestCounter = 0;
        TAtomic updateCounter = 0;

        const auto requester = MakeRequester([&requestCounter, key2](const TResolveRequestBatch& request, TResolveResultBatch& result, TStatEnv&) {
            const auto rc = AtomicIncrement(requestCounter);

            for (size_t i = 0; i < request.size(); ++i) {
                auto& endpoint = AddEndpoint(result[i]);
                endpoint.set_fqdn(TString("fqdn") + (request[i].endpoint_set_id() == key2.endpoint_set_id() ? ToString(rc) : ""));
            }
        });

        const auto updater = [&updateCounter](const NApi::TEndpointSet&) {
            AtomicIncrement(updateCounter);
            return true;
        };

        TEndpointSetManager manager(requester, TestClientName);

        manager.RegisterEndpointSet<TProvider>(key1, updater);
        manager.RegisterEndpointSet<TProvider>(key2, updater);

        manager.Update();
        UNIT_ASSERT_VALUES_EQUAL(updateCounter, 2);

        manager.Start(TDuration::MilliSeconds(10));

        Wait([&requestCounter, &updateCounter]() {
            return AtomicGet(requestCounter) > 10 && AtomicGet(updateCounter) > 10;
        });

        const auto& stat1 = manager.GetStat(key1);
        const auto& stat2 = manager.GetStat(key2);

        manager.Stop();

        UNIT_ASSERT_VALUES_EQUAL(stat1.ProviderUpdateCounter.Get(), 1);
        UNIT_ASSERT_VALUES_EQUAL(stat1.CacheStoreCounter.Get(), 1);

        UNIT_ASSERT_VALUES_EQUAL(updateCounter, requestCounter + 1);

        UNIT_ASSERT_VALUES_EQUAL(stat2.ProviderUpdateCounter.Get(), updateCounter - 1);
        UNIT_ASSERT_VALUES_EQUAL(stat2.CacheStoreCounter.Get(), updateCounter - 1);

        UNIT_ASSERT_VALUES_EQUAL(manager.GetStat().EndpointSetUpdateCounter.Get(), updateCounter);
    }

    Y_UNIT_TEST(TestUnavailable) {
        TEndpointSetKey key1("test-cluster", "test-service1");
        TEndpointSetKey key2("test-cluster", "test-service2");

        TAtomic requestCounter = 0;

        const auto requester = MakeRequester([&requestCounter, key2](const TResolveRequestBatch& request, TResolveResultBatch& result, TStatEnv& stat) {
            AtomicIncrement(requestCounter);
            DummyRequester->Resolve(request, result, stat);
            for (size_t i = 0; i < request.size(); ++i) {
                if (request[i].endpoint_set_id() == key2.endpoint_set_id()) {
                    result[i].HasError = true;
                }
            }
        });

        TEndpointSetManager manager(requester, TestClientName);

        manager.RegisterEndpointSet<TProvider>(key1);
        manager.RegisterEndpointSet<TProvider>(key2);

        manager.Update();

        try {
            manager.Start(TDuration::MilliSeconds(10));
            UNIT_ASSERT(false);
        } catch (...) {
        }
    }

    Y_UNIT_TEST(TestUnavailableAndCache) {
        TFsPath cacheDir("./cache");
        NFs::RemoveRecursive(cacheDir.GetPath());

        TEndpointSetKey key("test-cluster", "test-service1");

        {
            TEndpointSetManager manager(cacheDir.GetPath(), DummyRequester, TestClientName);

            manager.RegisterEndpointSet<TProvider>(key);
            manager.Start(TDuration::MilliSeconds(10));
            manager.Stop();
        }
        {
            TAtomic requestCounter = 0;
            const auto brokenRequester = MakeRequester([&requestCounter](const TResolveRequestBatch&, TResolveResultBatch&, TStatEnv&) {
                AtomicIncrement(requestCounter);
                throw yexception();
            });

            TEndpointSetManager manager(cacheDir.GetPath(), brokenRequester, TestClientName);

            TProvider* p = manager.RegisterEndpointSet<TProvider>(key);
            manager.Start(TDuration::MilliSeconds(10));

            {
                auto eps = p->GetEndpointSet();
                UNIT_ASSERT(eps.endpoints_size() == 1);
                UNIT_ASSERT(eps.endpoints(0).fqdn() == "dummy_fqdn");
            }

            const auto& stat = manager.GetStat(key);

            Wait([&requestCounter]() {
                return AtomicGet(requestCounter) > 1;
            });

            {
                auto eps = p->GetEndpointSet();
                UNIT_ASSERT(eps.endpoints_size() == 1);
                UNIT_ASSERT(eps.endpoints(0).fqdn() == "dummy_fqdn");
            }

            manager.Stop();

            UNIT_ASSERT_VALUES_EQUAL(stat.RequesterErrors.Get(), requestCounter);
        }
    }

    Y_UNIT_TEST(TestDontResolveCached) {
        TFsPath cacheDir("./cache");
        NFs::RemoveRecursive(cacheDir.GetPath());

        TEndpointSetKey key("test-cluster", "test-service1");

        {
            TEndpointSetManager manager(cacheDir.GetPath(), DummyRequester, TestClientName);

            manager.RegisterEndpointSet<TProvider>(key);
            manager.Start(TDuration::MilliSeconds(10));
            manager.Stop();
        }

        TAtomic requestCounter = 0;;
        const auto requester = MakeRequester([&requestCounter](const TResolveRequestBatch&, TResolveResultBatch& result, TStatEnv&) {
            AtomicIncrement(requestCounter);
            for (auto& r : result) {
                auto& endpoint = AddEndpoint(r);
                endpoint.set_fqdn("new_fqdn");
            }
        });

        {
            TEndpointSetManager manager(cacheDir.GetPath(), requester, TestClientName);

            TProvider* p = manager.RegisterEndpointSet<TProvider>(key);
            manager.Start(TDuration::Max());

            {
                auto eps = p->GetEndpointSet();
                UNIT_ASSERT(eps.endpoints_size() == 1);
                UNIT_ASSERT_VALUES_EQUAL(eps.endpoints(0).fqdn(), "dummy_fqdn");
            }

            manager.Stop();

            UNIT_ASSERT_VALUES_EQUAL(AtomicGet(requestCounter), 0);
        }

        {
            TEndpointSetManager manager(cacheDir.GetPath(), requester, TestClientName);

            TProvider* p = manager.RegisterEndpointSet<TProvider>(key);
            manager.Init(key);
            manager.Start(TDuration::Max());

            {
                auto eps = p->GetEndpointSet();
                UNIT_ASSERT(eps.endpoints_size() == 1);
                UNIT_ASSERT_VALUES_EQUAL(eps.endpoints(0).fqdn(), "dummy_fqdn");
            }

            manager.Stop();

            UNIT_ASSERT_VALUES_EQUAL(AtomicGet(requestCounter), 0);
        }

        {
            TEndpointSetManager manager(cacheDir.GetPath(), requester, TestClientName);

            TProvider* p = manager.RegisterEndpointSet<TProvider>(key);
            manager.Update(key);
            manager.Start(TDuration::Max());

            {
                auto eps = p->GetEndpointSet();
                UNIT_ASSERT(eps.endpoints_size() == 1);
                UNIT_ASSERT_VALUES_EQUAL(eps.endpoints(0).fqdn(), "new_fqdn");
            }

            manager.Stop();

            UNIT_ASSERT(AtomicGet(requestCounter) > 0);
        }
    }

    Y_UNIT_TEST(TestBrokenInitAndCache) {
        TFsPath cacheDir("./cache");
        NFs::RemoveRecursive(cacheDir.GetPath());

        TEndpointSetKey key("test-cluster", "test-service1");

        {
            TEndpointSetManager manager(cacheDir.GetPath(), DummyRequester, TestClientName);

            manager.RegisterEndpointSet<TProvider>(key);
            manager.Start(TDuration::MilliSeconds(10));
            manager.Stop();
        }
        {
            TAtomic requestCounter = 0;
            TAtomic initCounter = 0;
            TAtomic brokenInit = 1;
            const auto brokenRequester = MakeRequester(
                [&requestCounter](const TResolveRequestBatch&, TResolveResultBatch& result, TStatEnv&) {
                    AtomicIncrement(requestCounter);
                    for (auto& r : result) {
                        auto& endpoint = AddEndpoint(r);
                        endpoint.set_fqdn("new_fqdn");
                    }
                },
                [&initCounter, &brokenInit]() {
                    AtomicIncrement(initCounter);
                    if (AtomicGet(brokenInit)) {
                        throw yexception();
                    }
                },
                TDuration::MilliSeconds(50)
            );

            TEndpointSetManager manager(cacheDir.GetPath(), brokenRequester, TestClientName);

            TProvider* p = manager.RegisterEndpointSet<TProvider>(key);
            manager.Start(TDuration::MilliSeconds(10));

            {
                auto eps = p->GetEndpointSet();
                UNIT_ASSERT(eps.endpoints_size() == 1);
                UNIT_ASSERT(eps.endpoints(0).fqdn() == "dummy_fqdn");
            }

            const auto& stat = manager.GetStat(key);

            Wait([&initCounter]() {
                return AtomicGet(initCounter) > 5;
            });

            UNIT_ASSERT_VALUES_EQUAL(AtomicGet(requestCounter), 0);

            {
                auto eps = p->GetEndpointSet();
                UNIT_ASSERT(eps.endpoints_size() == 1);
                UNIT_ASSERT(eps.endpoints(0).fqdn() == "dummy_fqdn");
            }

            UNIT_ASSERT(stat.RequesterErrors.Get() > 0);

            AtomicSet(brokenInit, 0);
            WaitUpdate(manager.GetStat());

            UNIT_ASSERT(AtomicGet(requestCounter) > 0);

            {
                auto eps = p->GetEndpointSet();
                UNIT_ASSERT(eps.endpoints_size() == 1);
                UNIT_ASSERT(eps.endpoints(0).fqdn() == "new_fqdn");
            }

            manager.Stop();
        }
    }

    Y_UNIT_TEST(TestYpTimetamp) {
        TFsPath cacheDir("./cache");
        NFs::RemoveRecursive(cacheDir.GetPath());

        TEndpointSetKey key("test-cluster", "test-service1");

        TAtomic ypTimestamp = 100;
        const auto requester = MakeRequester([&ypTimestamp](const TResolveRequestBatch&, TResolveResultBatch& result, TStatEnv&) {
            const size_t ts = AtomicGet(ypTimestamp);
            for (auto& r : result) {
                auto& endpoint = AddEndpoint(r);
                endpoint.set_port(ts);

                r.Result.set_timestamp(ts);
            }
        });

        {
            TEndpointSetManager manager(cacheDir.GetPath(), requester, TestClientName);

            manager.RegisterEndpointSet<TProvider>(key);
            manager.Start(TDuration::MilliSeconds(10));
            manager.Stop();
        }

        AtomicSet(ypTimestamp, 99);

        TEndpointSetManager manager(cacheDir.GetPath(), requester, TestClientName);

        TProvider* p = manager.RegisterEndpointSet<TProvider>(key);
        p->UpdateActiveEndpointSetInfo = true;

        manager.Start(TDuration::MilliSeconds(10));

        const auto& stat = manager.GetStat();
        const auto& epsStat = manager.GetStat(key);

        {
            auto eps = p->GetEndpointSet();
            UNIT_ASSERT(eps.endpoints_size() == 1);
            UNIT_ASSERT(eps.endpoints(0).port() == 100);
            UNIT_ASSERT_VALUES_EQUAL(epsStat.YpTimestampMin.Get(), 100);
            UNIT_ASSERT_VALUES_EQUAL(epsStat.YpTimestampMax.Get(), 100);
        }

        WaitUpdate(stat);

        {
            auto eps = p->GetEndpointSet();
            UNIT_ASSERT(eps.endpoints_size() == 1);
            UNIT_ASSERT(eps.endpoints(0).port() == 100);
            UNIT_ASSERT(stat.ObsoleteTimestamp.Get() > 0);
            UNIT_ASSERT(stat.CacheStoreErrors.Get() == 0);
            UNIT_ASSERT_VALUES_EQUAL(epsStat.YpTimestampMin.Get(), 100);
            UNIT_ASSERT_VALUES_EQUAL(epsStat.YpTimestampMax.Get(), 100);
        }

        AtomicSet(ypTimestamp, 101);

        WaitUpdate(stat);

        {
            auto eps = p->GetEndpointSet();
            UNIT_ASSERT(eps.endpoints_size() == 1);
            UNIT_ASSERT(eps.endpoints(0).port() == 101);
            UNIT_ASSERT(epsStat.YpTimestampMin.Get() == 101);
            UNIT_ASSERT(epsStat.YpTimestampMax.Get() == 101);
        }

        const auto prev = stat.ObsoleteTimestamp.Get();
        AtomicSet(ypTimestamp, 100);

        WaitUpdate(stat);

        {
            auto eps = p->GetEndpointSet();
            UNIT_ASSERT(eps.endpoints_size() == 1);
            UNIT_ASSERT(eps.endpoints(0).port() == 101);
            UNIT_ASSERT(stat.ObsoleteTimestamp.Get() > prev);
            UNIT_ASSERT(stat.CacheStoreErrors.Get() == 0);
            UNIT_ASSERT_VALUES_EQUAL(epsStat.YpTimestampMin.Get(), 101);
            UNIT_ASSERT_VALUES_EQUAL(epsStat.YpTimestampMax.Get(), 101);
        }

        manager.Stop();
    }

    Y_UNIT_TEST(TestYpTimetampMinMax) {
        TFsPath cacheDir("./cache");
        NFs::RemoveRecursive(cacheDir.GetPath());

        TAtomic ypTimestamp = 100;
        const auto requester = MakeRequester([&ypTimestamp](const TResolveRequestBatch&, TResolveResultBatch& result, TStatEnv&) {
            const auto ts = AtomicGet(ypTimestamp);

            for (auto& r : result) {
                auto& endpoint = AddEndpoint(r);
                endpoint.set_port(ts);

                r.Result.set_timestamp(ts);
            }
        });

        TEndpointSetManager manager(cacheDir.GetPath(), requester, TestClientName);

        TAtomic broken = 0;
        TEndpointSetKey key("test-cluster", "test-service1");

        auto* p1 = manager.RegisterEndpointSet<TProvider>(key);
        p1->UpdateActiveEndpointSetInfo = true;

        auto* p2 = manager.RegisterEndpointSet<TProvider>(key, [&](const NApi::TEndpointSet&) {
            if (AtomicGet(broken)) {
                throw yexception() << "broken";
            }
        });
        p2->UpdateActiveEndpointSetInfo = true;

        manager.Start(TDuration::MilliSeconds(10));

        const auto& stat = manager.GetStat();
        const auto& epsStat = manager.GetStat(key);

        UNIT_ASSERT_VALUES_EQUAL(epsStat.YpTimestampMin.Get(), 100);
        UNIT_ASSERT_VALUES_EQUAL(epsStat.YpTimestampMax.Get(), 100);

        AtomicSet(broken, 1);
        AtomicSet(ypTimestamp, 101);
        WaitUpdate(stat);

        UNIT_ASSERT_VALUES_EQUAL(epsStat.YpTimestampMin.Get(), 100);
        UNIT_ASSERT_VALUES_EQUAL(epsStat.YpTimestampMax.Get(), 101);
    }

    Y_UNIT_TEST(TestStartTwice) {
        TEndpointSetManager manager(DummyRequester, TestClientName);
        manager.Start(TDuration::MilliSeconds(10));

        try {
            manager.Start(TDuration::MilliSeconds(10));
            UNIT_ASSERT(false);
        } catch (...) {
            UNIT_ASSERT(CurrentExceptionMessage().Contains("already started"));
        }

        manager.Stop();

        try {
            manager.Start(TDuration::MilliSeconds(10));
            UNIT_ASSERT(false);
        } catch (...) {
            UNIT_ASSERT(CurrentExceptionMessage().Contains("already finished"));
        }
    }

    Y_UNIT_TEST(TestUpdateCallback) {
        TEndpointSetManager manager(DummyRequester, TestClientName);
        TAtomic updateCallbackCounter = 0;

        manager.Start(TDuration::MilliSeconds(10), {}, [&](){
            AtomicIncrement(updateCallbackCounter);
        });

        const auto& stat = manager.GetStat();

        for (size_t i = 0; i < 10; ++i) {
            WaitUpdate(stat);
        }

        manager.Stop();

        UNIT_ASSERT(updateCallbackCounter >= 10);
        UNIT_ASSERT_VALUES_EQUAL(stat.UpdateSucceeded.Get(), updateCallbackCounter);
        UNIT_ASSERT_VALUES_EQUAL(stat.UpdateLoopSucceeded.Get(), updateCallbackCounter);
        UNIT_ASSERT_VALUES_EQUAL(stat.UpdateCounter.Get(), updateCallbackCounter);
        UNIT_ASSERT_VALUES_EQUAL(stat.UpdateLoopCounter.Get(), updateCallbackCounter);
    }

    const TString dataPath = ArcadiaSourceRoot() + "/infra/yp_service_discovery/libs/sdlib/ut";

    Y_UNIT_TEST(TestRawFiles) {
        TEndpointSetManager manager(DummyRequester, TestClientName);

        TEndpointSetKey key("f");
        NFs::Copy(dataPath + "/correct_endpointset", key.FilePath);

        TVector<NApi::TEndpointSet> updateSequence;
        TAtomic updateCounter = 0;

        const auto updater = [&](const NApi::TEndpointSet& endpoints) {
            updateSequence.push_back(endpoints);
            AtomicIncrement(updateCounter);
            return true;
        };

        manager.RegisterEndpointSet<TProvider>(key, updater);

        manager.Start(TDuration::MilliSeconds(10));

        Wait([&]() {
            return AtomicGet(updateCounter) > 0;
        });

        NFs::Copy(dataPath + "/correct_endpointset2", key.FilePath);

        Wait([&]() {
            return AtomicGet(updateCounter) > 1;
        });

        manager.Stop();

        UNIT_ASSERT_VALUES_EQUAL(updateCounter, 2);

        NApi::TEndpointSet eps1;
        ReadFile(dataPath + "/correct_endpointset", eps1);

        UNIT_ASSERT_VALUES_EQUAL(eps1.DebugString(), updateSequence[0].DebugString());

        NApi::TEndpointSet eps2;
        ReadFile(dataPath + "/correct_endpointset", eps2);
        UNIT_ASSERT_VALUES_EQUAL(eps2.DebugString(), updateSequence[0].DebugString());
    }

    Y_UNIT_TEST(TestOverload) {
        const auto requester = MakeRequester([](const TResolveRequestBatch&, TResolveResultBatch& result, TStatEnv&) {
            Sleep(TDuration::MilliSeconds(20));
            for (auto& r : result) {
                AddEndpoint(r);
            }
        });

        TEndpointSetManager manager(requester, TestClientName);
        TEndpointSetKey key("test-cluster", "test-service");
        manager.RegisterEndpointSet<TProvider>(key);
        manager.Start(TDuration::MilliSeconds(50));

        const auto& stat = manager.GetStat();

        Wait([&]() {
            return stat.UpdateLoopOverload.Get() > 0;
        });
    }

    IRemoteRequesterPtr MakeFillRequester(const TVector<std::pair<TString, ui16>>& allEndpoints) {
        auto fillEndpointSet = [&allEndpoints](TResolveResult& r){
            for (const auto& [fqdn, port] : allEndpoints) {
                auto& endpoint = *r.Result.mutable_endpoint_set()->add_endpoints();
                endpoint.set_fqdn(fqdn);
                endpoint.set_port(port);
                endpoint.set_ip6_address("::1");
            }
        };

        return MakeRequester([fillEndpointSet](const TResolveRequestBatch&, TResolveResultBatch& result, TStatEnv&) {
            for (auto& r : result) {
                fillEndpointSet(r);
            }
        });
    }

    void CompareEndpointSet(const TEndpointSetEx& endpointSet, const TVector<TString>& expectedEndpoints) {
        UNIT_ASSERT_VALUES_EQUAL(endpointSet.endpoints_size(), expectedEndpoints.size());

        for (ssize_t i = 0; i < endpointSet.endpoints_size(); ++i) {
            const auto& endpoint = endpointSet.endpoints(i);
            const TString endpointStr = TStringBuilder() << endpoint.fqdn() << ":" << endpoint.port();

            UNIT_ASSERT_C(
                Find(expectedEndpoints.begin(), expectedEndpoints.end(), endpointStr) != expectedEndpoints.end(),
                TStringBuilder() << "Not found endpoint in expected endpoints: " << endpointStr
            );
        }
    }

    Y_UNIT_TEST(TestSkipEndpointsOption) {
        const TVector<std::pair<TString, ui16>> allEndpoints{
            {"dummy_fqdn1", 80},
            {"dummy_fqdn1", 8080},
            {"dummy_fqdn2", 80},
            {"dummy_fqdn2", 8080},
            {"dummy_fqdn3", 80},
            {"dummy_fqdn3", 8080}
        };

        const TVector<TString> skipEndpoints{
            "dummy_fqdn1",
            "dummy_fqdn2:80"
        };

        const TVector<TString> expectedEndpoints{
            "dummy_fqdn2:8080",
            "dummy_fqdn3:80",
            "dummy_fqdn3:8080"
        };

        const auto requester = MakeFillRequester(allEndpoints);

        TEndpointSetOptions endpointSetOptions;
        for (TString endpoint : skipEndpoints) {
            endpointSetOptions.MutableSkipEndpoints()->Add(std::move(endpoint));
        }

        TEndpointSetManager manager(requester, TestClientName);
        const TEndpointSetKey key("test-cluster", "test-service");

        const auto* provider = manager.RegisterEndpointSetEx<TProvider>(key, endpointSetOptions);
        manager.Start(TDuration::MilliSeconds(50));
        const auto endpointSet = provider->GetEndpointSet();

        CompareEndpointSet(endpointSet, expectedEndpoints);
    }

    Y_UNIT_TEST(TestOffsetPortOption) {
        const TVector<std::pair<TString, ui16>> allEndpoints{
            {"dummy_fqdn1", 80},
            {"dummy_fqdn1", 8080},
        };

        const i32 offsetPort = 3;

        const TVector<TString> expectedEndpoints{
            "dummy_fqdn1:83",
            "dummy_fqdn1:8083",
        };

        const auto requester = MakeFillRequester(allEndpoints);

        TEndpointSetOptions endpointSetOptions;
        endpointSetOptions.SetOffsetPort(offsetPort);

        TEndpointSetManager manager(requester, TestClientName);
        const TEndpointSetKey key1("test-cluster", "test-service");
        const TEndpointSetKey key2("test-cluster", "test-service2");

        {
            TProvider provider;
            auto ref = manager.Subscribe(key1, provider, endpointSetOptions);
            CompareEndpointSet(ref->GetEndpointSet(), expectedEndpoints);
        }

        {
            const auto* provider = manager.RegisterEndpointSetEx<TProvider>(key2, endpointSetOptions);
            manager.Start(TDuration::MilliSeconds(50));
            const auto endpointSet = provider->GetEndpointSet();
            CompareEndpointSet(endpointSet, expectedEndpoints);
        }
    }

    Y_UNIT_TEST(TestUnusedFields) {
        TAtomic requestCounter = 0;
        const auto requester = MakeRequester([&requestCounter](const TResolveRequestBatch&, TResolveResultBatch& result, TStatEnv&) {
            size_t rc = AtomicIncrement(requestCounter);
            for (auto& r : result) {
                auto& endpoint = AddEndpoint(r);
                endpoint.set_ready(rc % 2);
            }
        });

        TAtomic updateCounter = 0;
        const auto updater = [&updateCounter](const NApi::TEndpointSet&) {
            AtomicIncrement(updateCounter);
            return true;
        };

        TEndpointSetManager manager(requester, TestClientName);

        const TEndpointSetKey key1("test-cluster", "test-service");
        manager.RegisterEndpointSet<TProvider>(key1, updater);
        manager.Start(TDuration::MilliSeconds(10));

        Wait([&manager]() {
            return manager.GetStat().UpdateCounter > 10;
        });

        manager.Stop();

        UNIT_ASSERT(requestCounter > 10);
        UNIT_ASSERT(updateCounter == 1);

    }

    Y_UNIT_TEST(TestInvOffsetPortOption) {
        const TVector<std::pair<TString, ui16>> allEndpoints{
            {"dummy_fqdn1", 80},
            {"dummy_fqdn1", 8080},
        };

        const i32 offsetPort = -3;

        const TVector<TString> expectedEndpoints{
            "dummy_fqdn1:77",
            "dummy_fqdn1:8077",
        };

        const auto requester = MakeFillRequester(allEndpoints);

        TEndpointSetOptions endpointSetOptions;
        endpointSetOptions.SetOffsetPort(offsetPort);

        TEndpointSetManager manager(requester, TestClientName);
        const TEndpointSetKey key1("test-cluster", "test-service");
        const TEndpointSetKey key2("test-cluster", "test-service2");

        {
            TProvider provider;
            auto ref = manager.Subscribe(key1, provider, endpointSetOptions);
            CompareEndpointSet(ref->GetEndpointSet(), expectedEndpoints);
        }

        {
            const auto* provider = manager.RegisterEndpointSetEx<TProvider>(key2, endpointSetOptions);
            manager.Start(TDuration::MilliSeconds(50));
            const auto endpointSet = provider->GetEndpointSet();
            CompareEndpointSet(endpointSet, expectedEndpoints);
        }
    }

    Y_UNIT_TEST(BrokenUpdate) {
        TFsPath cacheDir("./cache");
        NFs::RemoveRecursive(cacheDir.GetPath());

        TEndpointSetKey key("test-cluster", "test-service");

        TAtomic port = 80;
        TAtomic requestCount = 0;
        auto requester = MakeRequester([&port, &requestCount](const TResolveRequestBatch&, TResolveResultBatch& result, TStatEnv&) {
            AtomicIncrement(requestCount);
            auto& endpoint = AddEndpoint(result[0]);
            endpoint.set_fqdn("fqdn");
            endpoint.set_ip4_address("127.0.0.1");
            endpoint.set_port(AtomicGet(port));
        });

        TAtomic errors = 0;
        auto onUpdate = [&errors](const NApi::TEndpointSet& endpointSet) {
            if (endpointSet.endpoints(0).port() == 81) {
                AtomicIncrement(errors);
                throw yexception();
            }
        };


        {
            TProvider p1(onUpdate);

            TEndpointSetManager manager(cacheDir.GetPath(), requester, TestClientName);
            auto s1 = manager.Subscribe(key, p1);
            p1.SetEndpointSet(s1->GetEndpointSet());

            auto* p2 = manager.RegisterEndpointSet<TProvider>(key, onUpdate);
            auto* p3 = manager.RegisterEndpointSet<TProvider>(key);

            manager.Start(TDuration::MilliSeconds(10));

            {
                const auto endpointSet = p1.GetEndpointSet();
                UNIT_ASSERT(endpointSet.endpoints_size() > 0);
                UNIT_ASSERT_VALUES_EQUAL(endpointSet.endpoints(0).port(), 80);
            }

            const auto& stat = manager.GetStat();
            AtomicSet(port, 81);
            WaitUpdate(stat);

            UNIT_ASSERT(AtomicGet(errors) > 0);

            {
                const auto endpointSet = p1.GetEndpointSet();
                UNIT_ASSERT(endpointSet.endpoints_size() > 0);
                UNIT_ASSERT_VALUES_EQUAL(endpointSet.endpoints(0).port(), 80);
            }
            {
                const auto endpointSet = p2->GetEndpointSet();
                UNIT_ASSERT(endpointSet.endpoints_size() > 0);
                UNIT_ASSERT_VALUES_EQUAL(endpointSet.endpoints(0).port(), 80);
            }
            {
                const auto endpointSet = p3->GetEndpointSet();
                UNIT_ASSERT(endpointSet.endpoints_size() > 0);
                UNIT_ASSERT_VALUES_EQUAL(endpointSet.endpoints(0).port(), 81);
            }
        }

        {
            errors = 0;
            const auto prevRequestCount = requestCount;

            TProvider p1(onUpdate);
            TEndpointSetManager manager(cacheDir.GetPath(), requester, TestClientName);
            auto s1 = manager.Subscribe(key, p1);
            p1.SetEndpointSet(s1->GetEndpointSet());


            auto* p2 = manager.RegisterEndpointSet<TProvider>(key, onUpdate);
            auto* p3 = manager.RegisterEndpointSet<TProvider>(key);
            manager.Init(key);

            UNIT_ASSERT_VALUES_EQUAL(prevRequestCount, requestCount);

            {
                const auto endpointSet = p1.GetEndpointSet();
                UNIT_ASSERT(endpointSet.endpoints_size() > 0);
                UNIT_ASSERT_VALUES_EQUAL(endpointSet.endpoints(0).port(), 80);
            }

            {
                const auto endpointSet = p2->GetEndpointSet();
                UNIT_ASSERT(endpointSet.endpoints_size() > 0);
                UNIT_ASSERT_VALUES_EQUAL(endpointSet.endpoints(0).port(), 80);
            }

            {
                const auto endpointSet = p3->GetEndpointSet();
                UNIT_ASSERT(endpointSet.endpoints_size() > 0);
                UNIT_ASSERT_VALUES_EQUAL(endpointSet.endpoints(0).port(), 80);
            }

            manager.Start(TDuration::MilliSeconds(10));

            const auto& stat = manager.GetStat();
            WaitUpdate(stat);

            UNIT_ASSERT(AtomicGet(errors) > 0);

            {
                const auto endpointSet = p1.GetEndpointSet();
                UNIT_ASSERT(endpointSet.endpoints_size() > 0);
                UNIT_ASSERT_VALUES_EQUAL(endpointSet.endpoints(0).port(), 80);
            }

            {
                const auto endpointSet = p2->GetEndpointSet();
                UNIT_ASSERT(endpointSet.endpoints_size() > 0);
                UNIT_ASSERT_VALUES_EQUAL(endpointSet.endpoints(0).port(), 80);
            }

            AtomicSet(port, 82);
            WaitUpdate(stat);

            {
                const auto endpointSet = p1.GetEndpointSet();
                UNIT_ASSERT(endpointSet.endpoints_size() > 0);
                UNIT_ASSERT_VALUES_EQUAL(endpointSet.endpoints(0).port(), 82);
            }

            {
                const auto endpointSet = p2->GetEndpointSet();
                UNIT_ASSERT(endpointSet.endpoints_size() > 0);
                UNIT_ASSERT_VALUES_EQUAL(endpointSet.endpoints(0).port(), 82);
            }

            {
                const auto endpointSet = p3->GetEndpointSet();
                UNIT_ASSERT(endpointSet.endpoints_size() > 0);
                UNIT_ASSERT_VALUES_EQUAL(endpointSet.endpoints(0).port(), 82);
            }
        }
    }

    Y_UNIT_TEST(AllowEmptyEndpointSetsOnStart) {
        TFsPath cacheDir("./cache");
        NFs::RemoveRecursive(cacheDir.GetPath());

        TSDConfig config;
        config.SetCacheDir(cacheDir.GetPath());
        config.SetAllowEmptyEndpointSetsOnStart(true);
        config.SetClientName(TestClientName);

        TEndpointSetKey key("test-cluster", "test-service");

        TAtomic empty = 1;
        TAtomic port = 80;
        auto requester = MakeRequester([&empty, &port](const TResolveRequestBatch&, TResolveResultBatch& result, TStatEnv&) {
            if (!AtomicGet(empty)) {
                auto& endpoint = AddEndpoint(result[0]);
                endpoint.set_fqdn("fqdn");
                endpoint.set_ip4_address("127.0.0.1");
                endpoint.set_port(AtomicGet(port));
            }
        });

        {
            TProvider p1;


            TEndpointSetManager manager(config, requester);
            auto s1 = manager.Subscribe(key, p1);
            UNIT_ASSERT(s1->GetEndpointSet().endpoints_size() == 0);

            manager.Start(TDuration::MilliSeconds(10));

            const auto& stat = manager.GetStat();
            AtomicSet(empty, 0);
            WaitUpdate(stat);

            {
                const auto endpointSet = p1.GetEndpointSet();
                UNIT_ASSERT(endpointSet.endpoints_size() > 0);
                UNIT_ASSERT_VALUES_EQUAL(endpointSet.endpoints(0).port(), 80);
                UNIT_ASSERT(stat.InvalidEndpointSetErrors.Get() == 0);
            }

            AtomicSet(empty, 1);
            WaitUpdate(stat);

            {
                const auto endpointSet = p1.GetEndpointSet();
                UNIT_ASSERT(endpointSet.endpoints_size() > 0);
                UNIT_ASSERT_VALUES_EQUAL(endpointSet.endpoints(0).port(), 80);
                UNIT_ASSERT(stat.InvalidEndpointSetErrors.Get() > 0);
            }

            AtomicSet(empty, 0);
            AtomicSet(port, 81);
            WaitUpdate(stat);

            {
                const auto endpointSet = p1.GetEndpointSet();
                UNIT_ASSERT(endpointSet.endpoints_size() > 0);
                UNIT_ASSERT_VALUES_EQUAL(endpointSet.endpoints(0).port(), 81);
                UNIT_ASSERT(stat.InvalidEndpointSetErrors.Get() > 0);
            }

            AtomicSet(empty, 1);
            WaitUpdate(stat);
        }

        {
            TProvider p1;

            TEndpointSetManager manager(config, requester);
            auto s1 = manager.Subscribe(key, p1);
            p1.SetEndpointSet(s1->GetEndpointSet());

            const auto& stat = manager.GetStat();

            {
                const auto endpointSet = p1.GetEndpointSet();
                UNIT_ASSERT(endpointSet.endpoints_size() > 0);
                UNIT_ASSERT_VALUES_EQUAL(endpointSet.endpoints(0).port(), 81);
                UNIT_ASSERT(stat.InvalidEndpointSetErrors.Get() == 0);
            }

            manager.Start(TDuration::MilliSeconds(10));

            WaitUpdate(stat);

            {
                const auto endpointSet = p1.GetEndpointSet();
                UNIT_ASSERT(endpointSet.endpoints_size() > 0);
                UNIT_ASSERT_VALUES_EQUAL(endpointSet.endpoints(0).port(), 81);
                UNIT_ASSERT(stat.InvalidEndpointSetErrors.Get() > 0);
            }


            AtomicSet(empty, 0);
            AtomicSet(port, 80);

            WaitUpdate(stat);

            {
                const auto endpointSet = p1.GetEndpointSet();
                UNIT_ASSERT(endpointSet.endpoints_size() > 0);
                UNIT_ASSERT_VALUES_EQUAL(endpointSet.endpoints(0).port(), 80);
            }

            AtomicSet(empty, 1);

            WaitUpdate(stat);

            {
                const auto endpointSet = p1.GetEndpointSet();
                UNIT_ASSERT(endpointSet.endpoints_size() > 0);
                UNIT_ASSERT_VALUES_EQUAL(endpointSet.endpoints(0).port(), 80);
            }
        }
    }

    Y_UNIT_TEST(CheckForExistence) {
        TFsPath cacheDir("./cache");
        TEndpointSetKey key("test-cluster", "test-service");
        TProvider p;

        auto okRequester = MakeRequester([](const TResolveRequestBatch&, TResolveResultBatch& result, TStatEnv&) {
                auto& endpoint = AddEndpoint(result[0]);
                endpoint.set_fqdn("fqdn");
                endpoint.set_ip4_address("127.0.0.1");
                endpoint.set_port(80);

                result[0].Result.set_resolve_status(NApi::EResolveStatus::OK);
        });
        auto emptyRequester = MakeRequester([](const TResolveRequestBatch&, TResolveResultBatch& result, TStatEnv&) {
            result[0].Result.set_resolve_status(NApi::EResolveStatus::EMPTY);
        });
        auto nonexistRequester = MakeRequester([](const TResolveRequestBatch&, TResolveResultBatch& result, TStatEnv&) {
            result[0].Result.set_resolve_status(NApi::EResolveStatus::NOT_EXISTS);
        });

        auto assertFails = [&p, &key](TSDConfig& config, IRemoteRequesterPtr& requester) {
            NFs::RemoveRecursive(config.GetCacheDir());
            TEndpointSetManager manager(config, requester);

            UNIT_ASSERT_EXCEPTION_CONTAINS(manager.Subscribe(key, p), yexception, "endpointset [test-cluster#test-service] unavailable");
            UNIT_ASSERT_EXCEPTION_CONTAINS(manager.Start(TDuration::MilliSeconds(10)), yexception, "endpointset [test-cluster#test-service] unavailable");
            UNIT_ASSERT_VALUES_EQUAL(manager.GetStat().InvalidEndpointSetErrors.Get(), 2);
        };
        auto assertOk = [&p, &key](TSDConfig& config, IRemoteRequesterPtr& requester) {
            NFs::RemoveRecursive(config.GetCacheDir());
            TEndpointSetManager manager(config, requester);

            UNIT_ASSERT_NO_EXCEPTION(manager.Subscribe(key, p));
            UNIT_ASSERT_NO_EXCEPTION(manager.Start(TDuration::MilliSeconds(10)));
            UNIT_ASSERT_VALUES_EQUAL(manager.GetStat().InvalidEndpointSetErrors.Get(), 0);
        };

        {
            TSDConfig config;
            config.SetCacheDir(cacheDir.GetPath());
            config.SetAllowEmptyEndpointSetsOnStart(true);
            config.SetCheckForExistence(true);
            config.SetClientName(TestClientName);

            assertOk(config, okRequester);
            assertOk(config, emptyRequester);
            assertFails(config, nonexistRequester);
        }

        {
            TSDConfig config;
            config.SetCacheDir(cacheDir.GetPath());
            config.SetAllowEmptyEndpointSetsOnStart(false);
            config.SetCheckForExistence(false);
            config.SetClientName(TestClientName);

            assertOk(config, okRequester);
            assertFails(config, emptyRequester);
            assertFails(config, nonexistRequester);
        }

        {
            TSDConfig config;
            config.SetCacheDir(cacheDir.GetPath());
            config.SetAllowEmptyEndpointSetsOnStart(true);
            config.SetCheckForExistence(false);
            config.SetClientName(TestClientName);

            assertOk(config, okRequester);
            assertOk(config, emptyRequester);
            assertOk(config, nonexistRequester);
        }

        {
            TSDConfig config;
            config.SetCacheDir(cacheDir.GetPath());
            config.SetAllowEmptyEndpointSetsOnStart(false);
            config.SetCheckForExistence(true);
            config.SetClientName(TestClientName);

            assertOk(config, okRequester);
            assertFails(config, emptyRequester);
            assertFails(config, nonexistRequester);
        }
    }

    Y_UNIT_TEST(RemoveUnusedEndpointSets) {
        TAtomic requestSize = 0;
        const auto requester = MakeRequester([&requestSize](const TResolveRequestBatch& request, TResolveResultBatch& result, TStatEnv& stat) {
            DummyRequester->Resolve(request, result, stat);
            AtomicSet(requestSize, request.size());
        });

        TEndpointSetManager manager(requester, TestClientName, StandaloneCounterFactory, TDuration::MilliSeconds(10));

        TEndpointSetKey key1("test-cluster", "test-service1");
        TEndpointSetKey key2("test-cluster", "test-service2");
        TProvider p1;
        TProvider p2;
        IEndpointSetSubscriberRef s1 = manager.Subscribe(key1, p1);
        IEndpointSetSubscriberRef s2 = manager.Subscribe(key2, p2);

        manager.Update();
        UNIT_ASSERT_VALUES_EQUAL(AtomicGet(requestSize), 2);
        AtomicSet(requestSize, 0);

        // Start the background thread
        manager.Start(TDuration::MilliSeconds(10));

        Wait([&requestSize]() {
            return AtomicGet(requestSize) == 2;
        });

        s1 = {}; // Unsubscribe

        // The manager will forget this endpoint set soon.
        Wait([&requestSize]() {
            return AtomicGet(requestSize) == 1;
        });
    }

    Y_UNIT_TEST(TestDontCreateDirForInvalidInput) {
        TFsPath cacheDir("./cache");
        NFs::RemoveRecursive(cacheDir.GetPath());

        TEndpointSetKey invalidKey("test-cluster", "invalid-service");
        TEndpointSetKey validKey("test-cluster", "valid-service");

        auto requester = MakeRequester([&validKey](const TResolveRequestBatch& request, TResolveResultBatch& result, TStatEnv&) {
            for (size_t i = 0; i < request.size(); ++i) {
                if (request[i].endpoint_set_id() == validKey.endpoint_set_id()) {
                    AddEndpoint(result[i]);
                } else {
                    // not resolved: no endpoints
                }
            }
        });

        TEndpointSetManager manager(cacheDir.GetPath(), requester, TestClientName);
        manager.RegisterEndpointSet<TProvider>(invalidKey);
        manager.RegisterEndpointSet<TProvider>(validKey);
        manager.Update();

        UNIT_ASSERT(!(cacheDir / EndpointSetDirName(invalidKey)).IsDirectory());
        UNIT_ASSERT((cacheDir / EndpointSetDirName(validKey)).IsDirectory());
    }

    Y_UNIT_TEST(TestDontCreateDirForSpecificEndpoints) {
        TFsPath cacheDir("./cache");
        NFs::RemoveRecursive(cacheDir.GetPath());

        TEndpointSetKey key1("test-cluster", "first-service");
        TEndpointSetKey key2("test-cluster", "second-service");

        TEndpointSetOptions opts1;
        TEndpointSetOptions opts2;
        opts2.SetNoCacheOnDisk(true);

        TEndpointSetManager manager(cacheDir.GetPath(), DummyRequester, TestClientName);
        manager.RegisterEndpointSetEx<TProvider>(key1, opts1);
        manager.RegisterEndpointSetEx<TProvider>(key2, opts2);
        manager.Update();

        UNIT_ASSERT((cacheDir / EndpointSetDirName(key1)).IsDirectory());
        UNIT_ASSERT(!(cacheDir / EndpointSetDirName(key2)).IsDirectory());
    }
}
