#include "service.h"

#include <infra/yp_service_discovery/libs/logger/events/events_decl.ev.pb.h>
#include <infra/yp_service_discovery/libs/router_api/router_api.h>

#include <infra/libs/memory_lock/memory_lock.h>
#include <infra/libs/sensors/macros.h>
#include <infra/libs/sensors/sensor_registry.h>
#include <infra/libs/yp_replica/yp_replica.h>
#include <yp/cpp/yp/token.h>

#include <library/cpp/protobuf/json/proto2json.h>
#include <library/cpp/watchdog/watchdog.h>

#include <util/stream/file.h>
#include <util/string/subst.h>
#include <util/system/mlock.h>
#include <util/system/hostname.h>
#include <util/datetime/base.h>

namespace {
    const TString ACCESS_DENIED_MESSAGE = "Access Denied. Required field \"client_name\" is missing";
    const TString EMPTY_CLUSTER_NAME_MESSAGE = "Resolve Endpoints failed. Required field \"cluster_name\" is missing";
    constexpr ui32 BACKUP_FORMAT_VERSION = 10;
    // TODO(ismagilas): remove after return of MAN
    constexpr ui32 MAN_BACKUP_FORMAT_VERSION = 9;

    ui32 GetBackupFormatVersionByYpCluster(const TStringBuf cluster) {
        return cluster == "man" ? MAN_BACKUP_FORMAT_VERSION : BACKUP_FORMAT_VERSION;
    }

    const THashSet<TString> SKIPPED_CLIENTS_FOR_EMPTY_RESPONSES_SENSOR = {{
        "scraperoveryt",
    }};

    template <typename TReq>
    TStringBuf GetYpClusterName(NInfra::TRequestPtr<TReq> request) {
        TStringBuf path = request->Path();
        path.SkipPrefix("/");
        return path.SplitOff('/').NextTok('/');
    }

    bool Equals(const google::protobuf::Message& lhs, const google::protobuf::Message& rhs) {
        return google::protobuf::util::MessageDifferencer::Equivalent(lhs, rhs);
    }
}

namespace NYP::NServiceDiscovery::NSensors {
    constexpr TStringBuf NAMESPACE = "service";
    constexpr TStringBuf RESOLVE_ENDPOINTS_NAMESPACE = "resolve_endpoints";
    constexpr TStringBuf RESOLVE_NODE_NAMESPACE = "resolve_node";
    constexpr TStringBuf RESOLVE_PODS_NAMESPACE = "resolve_pods";
    constexpr TStringBuf HTTP_SERVICE_NAMESPACE = "http_service";

    constexpr TStringBuf RESOLVE_REQUESTS = "requests";
    constexpr TStringBuf RESOLVE_SUCCESSES = "successes";
    constexpr TStringBuf RESOLVE_ACCESS_DENIED = "access_denied";
    constexpr TStringBuf RESOLVE_EMPTY_RESPONSES = "empty_responses";
    constexpr TStringBuf RESOLVE_NOT_CHANGED_RESPONSES = "not_changed";

    constexpr TStringBuf PING_RESPONSE_TIME = "ping_response_time";
    constexpr TStringBuf SHUTDOWN_RESPONSE_TIME = "shutdown_response_time";
    constexpr TStringBuf SENSORS_RESPONSE_TIME = "sensors_response_time";
    constexpr TStringBuf RESOLVE_RESPONSE_TIME = "resolve_response_time";
    constexpr TStringBuf REOPEN_LOG_RESPONSE_TIME = "reopen_log_response_time";

    constexpr TStringBuf RESOLVE_ENDPOINTS_STATUS = "endpoints_resolve_status";
    constexpr TStringBuf RESOLVE_NODE_STATUS = "node_resolve_status";
    constexpr TStringBuf RESOLVE_PODS_STATUS = "pods_resolve_status";
    constexpr TStringBuf RESOLVE_RESULT_SIZE = "resolve_result_size";

    constexpr TStringBuf QUEUE_OVERFLOW = "queue_overflow";
    constexpr TStringBuf FAILED_QUEUE_OVERFLOW = "failed_queue_overflow";
    constexpr TStringBuf EXCEPTIONS_OCCURED = "exceptions_occured";
    constexpr TStringBuf MAX_CONNECTIONS_REACHED = "max_connections_reached";
    constexpr TStringBuf REQUESTS_QUEUE_SIZE = "requests_queue_size";
    constexpr TStringBuf FAILED_REQUESTS_QUEUE_SIZE = "failed_requests_queue_size";

    constexpr TStringBuf YP_CLUSTER = "yp_cluster";
    constexpr TStringBuf ESID = "endpoint_set_id";
    constexpr TStringBuf NID = "node_id";
    constexpr TStringBuf PSID = "pod_set_id";
    constexpr double BASE_OF_HISTOGRAM = 1.5;
    constexpr double SCALE_OF_HISTORAM = 10.0;
    constexpr ui64 MAX_EMPTY_RESPONSES_SENSORS_NUMBER = 9500;
}


namespace {
    TString GenerateRUID() {
        return TStringBuilder() << Now().MicroSeconds() << RandomNumber<ui64>();
    }
}

namespace NYP::NServiceDiscovery {
    namespace {
        template <typename TRequest>
        NEventlog::TRequestStart GetEventRequestStart(NEventlog::ERequestType type, const TRequest& request) {
            NEventlog::TRequestStart ev;
            ev.SetRequestType(type);

            for (const auto& [key, value] : request.Attributes()) {
                auto* attr = ev.AddAttributes();
                attr->SetKey(key);
                attr->SetValue(value);
            }

            if (const NProtoBuf::FieldDescriptor* ruidField = request.Get().GetDescriptor()->FindFieldByName("ruid")) {
                ev.SetRUID(request.Get().GetReflection()->GetString(request.Get(), ruidField));
            }

            return ev;
        }

        NEventlog::TResolveEndpointsResult MakeResolveResultEvent(const NApi::TRspResolveEndpoints& response) {
            NEventlog::TResolveEndpointsResult result;
            result.SetYpTimestamp(response.timestamp());
            switch (response.resolve_status()) {
                case NApi::EResolveStatus::NOT_EXISTS:
                    result.SetResolveStatus(NEventlog::TResolveEndpointsResult::NOT_EXISTS);
                    break;
                case NApi::EResolveStatus::EMPTY:
                    result.SetResolveStatus(NEventlog::TResolveEndpointsResult::EMPTY);
                    break;
                case NApi::EResolveStatus::NOT_CHANGED:
                    result.SetResolveStatus(NEventlog::TResolveEndpointsResult::NOT_CHANGED);
                    break;
                case NApi::EResolveStatus::OK:
                    result.SetResolveStatus(NEventlog::TResolveEndpointsResult::OK);
                    break;
                default:
                    break;
            }
            result.SetEndpointSetSize(response.endpoint_set().endpoints().size());
            result.SetByteSize(response.ByteSizeLong());
            return result;
        }

        NEventlog::TResolvePodsResult MakeResolveResultEvent(const NApi::TRspResolvePods& response) {
            NEventlog::TResolvePodsResult result;
            result.SetYpTimestamp(response.timestamp());
            switch (response.resolve_status()) {
                case NApi::EResolveStatus::NOT_EXISTS:
                    result.SetResolveStatus(NEventlog::TResolvePodsResult::NOT_EXISTS);
                    break;
                case NApi::EResolveStatus::EMPTY:
                    result.SetResolveStatus(NEventlog::TResolvePodsResult::EMPTY);
                    break;
                case NApi::EResolveStatus::NOT_CHANGED:
                    result.SetResolveStatus(NEventlog::TResolvePodsResult::NOT_CHANGED);
                    break;
                case NApi::EResolveStatus::OK:
                    result.SetResolveStatus(NEventlog::TResolvePodsResult::OK);
                    break;
                default:
                    break;
            }
            result.SetPodSetSize(response.pod_set().pods().size());
            result.SetByteSize(response.ByteSizeLong());
            return result;
        }

        NEventlog::TResolveNodeResult MakeResolveResultEvent(const NApi::TRspResolveNode& response) {
            NEventlog::TResolveNodeResult result;
            result.SetYpTimestamp(response.timestamp());
            switch (response.resolve_status()) {
                case NApi::EResolveStatus::NOT_EXISTS:
                    result.SetResolveStatus(NEventlog::TResolveNodeResult::NOT_EXISTS);
                    break;
                case NApi::EResolveStatus::EMPTY:
                    result.SetResolveStatus(NEventlog::TResolveNodeResult::EMPTY);
                    break;
                case NApi::EResolveStatus::NOT_CHANGED:
                    result.SetResolveStatus(NEventlog::TResolveNodeResult::NOT_CHANGED);
                    break;
                case NApi::EResolveStatus::OK:
                    result.SetResolveStatus(NEventlog::TResolveNodeResult::OK);
                    break;
                default:
                    break;
            }
            result.SetNumberOfPods(response.pods().size());
            result.SetByteSize(response.ByteSizeLong());
            return result;
        }

        class TEmptyClusterNameError : public NInfra::TServiceError {
        public:
            TEmptyClusterNameError()
                : TServiceError(HttpCodes::HTTP_BAD_REQUEST, grpc::StatusCode::INVALID_ARGUMENT)
            {
            }
        };

        class THttpServiceCallbacks : public NInfra::IHttpServiceCallbacks {
        public:
            THttpServiceCallbacks(const NInfra::TSensorGroup* httpServiceSensorGroup, NInfra::TLogger* httpServiceLogger)
                : HttpServiceSensorGroup_(httpServiceSensorGroup)
                , HttpServiceLogger_(httpServiceLogger)
            {
                InitSensor();
            }

            void OnFailRequest(int failstate) override {
                if (failstate == 0) {
                    IncSensor(NSensors::QUEUE_OVERFLOW);
                } else if (failstate == -1) {
                    IncSensor(NSensors::FAILED_QUEUE_OVERFLOW);
                }
                LogEvent(ELogPriority::TLOG_ERR, NEventlog::TRequestFailed(failstate));
            }

            void OnException() override {
                IncSensor(NSensors::EXCEPTIONS_OCCURED);
                LogEvent(ELogPriority::TLOG_ERR, NEventlog::TException(CurrentExceptionMessage()));
            }

            void OnMaxConn() override {
                IncSensor(NSensors::MAX_CONNECTIONS_REACHED);
                LogEvent(ELogPriority::TLOG_ERR, NEventlog::TMaxConnReached());
            }

            void OnResponseBuild(const THttpResponse& response) override {
                TStringStream headers;
                headers << response.GetHeaders();
                headers.Finish();
                NEventlog::TResponseBuild ev(TString(HttpCodeStrEx(response.HttpCode())), headers.Str());
                LogEvent(ELogPriority::TLOG_INFO, ev);
            }

        private:
            const NInfra::TSensorGroup* HttpServiceSensorGroup_;
            NInfra::TLogger* HttpServiceLogger_;

            template <typename TEvent>
            void LogEvent(ELogPriority priority, const TEvent& event) {
                HttpServiceLogger_->SpawnFrame()->LogEvent(priority, event);
            }

            void IncSensor(const TStringBuf& sensor) {
                NInfra::TRateSensor(*HttpServiceSensorGroup_, sensor).Inc();
            }

            void InitSensor() {
                NInfra::TRateSensor(*HttpServiceSensorGroup_, NSensors::QUEUE_OVERFLOW);
                NInfra::TRateSensor(*HttpServiceSensorGroup_, NSensors::FAILED_QUEUE_OVERFLOW);
                NInfra::TRateSensor(*HttpServiceSensorGroup_, NSensors::EXCEPTIONS_OCCURED);
                NInfra::TRateSensor(*HttpServiceSensorGroup_, NSensors::MAX_CONNECTIONS_REACHED);
            }
        };
    } // anonymous namespace

    TIntrusivePtr<NInfra::THistogramRateSensor> GetHistogram(const TStringBuf& nameOfHistogram, NInfra::TSensorGroup sensorGroup, const TVector<std::pair<TStringBuf, TStringBuf>>& labels = {}) {
        for (const auto& [name, value] : labels) {
            sensorGroup.AddLabel(name, value);
        }
        return MakeIntrusive<NInfra::THistogramRateSensor>(
                sensorGroup, nameOfHistogram, NMonitoring::ExponentialHistogram(
                    NMonitoring::HISTOGRAM_MAX_BUCKETS_COUNT, NSensors::BASE_OF_HISTOGRAM, NSensors::SCALE_OF_HISTORAM));
    }

    template <class TReplicas>
    const auto& TService::GetClusterConfigs(const TConfig& config) {
        if constexpr (std::is_same<TReplicas, TYPEndpointReplicas>::value) {
            return config.GetYPEndpointClusterConfigs();
        } else if constexpr (std::is_same<TReplicas, TYPPodReplicas>::value) {
            return config.GetYPPodClusterConfigs();
        } else if constexpr (std::is_same<TReplicas, TYPNodeReplicas>::value) {
            return config.GetYPNodeClusterConfigs();
        } else {
            ythrow NInfra::TNotImplementedError() << "Unknown replicas holder type"; // Need more appropriate exception type
        }
    }

    TService::TService(const TConfig& config)
        : ConfigHolder_(config, config.GetWatchPatchConfig(), config.GetConfigUpdatesLoggerConfig())
        , Config_(ConfigHolder_.Accessor())
        , Logger_(CONFIG_SNAPSHOT_VALUE(Config_, GetLoggerConfig()))
        , EndpointsReplicaLogger_(CONFIG_SNAPSHOT_VALUE(Config_, GetEndpointsReplicaLoggerConfig()))
        , NodeReplicaLogger_(CONFIG_SNAPSHOT_VALUE(Config_, GetNodeReplicaLoggerConfig()))
        , PodsReplicaLogger_(CONFIG_SNAPSHOT_VALUE(Config_, GetPodsReplicaLoggerConfig()))
        , AdminHttpServiceLogger_(CONFIG_SNAPSHOT_VALUE(Config_, GetAdminHttpServiceLoggerConfig()))
        , DiscoveryHttpServiceLogger_(CONFIG_SNAPSHOT_VALUE(Config_, GetDiscoveryHttpServiceLoggerConfig()))
        , SensorGroup_(NSensors::NAMESPACE)
        , ResolveEndpointsSensorGroup_(SensorGroup_, NSensors::RESOLVE_ENDPOINTS_NAMESPACE)
        , ResolveNodeSensorGroup_(SensorGroup_, NSensors::RESOLVE_NODE_NAMESPACE)
        , ResolvePodsSensorGroup_(SensorGroup_, NSensors::RESOLVE_PODS_NAMESPACE)
        , AdminHttpServiceSensorGroup_(NInfra::TSensorGroup(SensorGroup_, NSensors::HTTP_SERVICE_NAMESPACE).AddLabels({{"type", "admin"}}))
        , DiscoveryHttpServiceSensorGroup_(NInfra::TSensorGroup(SensorGroup_, NSensors::HTTP_SERVICE_NAMESPACE).AddLabels({{"type", "discovery"}}))
        , AdminHttpService_(CONFIG_SNAPSHOT_VALUE(Config_, GetAdminHttpServiceConfig()), CreateAdminRouter(*this, *Config_), MakeHolder<NInfra::IProfileStatsCallbacks>(), MakeHolder<THttpServiceCallbacks>(&AdminHttpServiceSensorGroup_, &AdminHttpServiceLogger_))
        , DiscoveryHttpService_(CONFIG_SNAPSHOT_VALUE(Config_, GetDiscoveryHttpServiceConfig()), CreateDiscoveryRouter(*this), MakeHolder<NInfra::IProfileStatsCallbacks>(), MakeHolder<THttpServiceCallbacks>(&DiscoveryHttpServiceSensorGroup_, &DiscoveryHttpServiceLogger_))
        , GrpcServer_(CONFIG_SNAPSHOT_VALUE(Config_, GetGrpcServiceConfig()), *this)
        , YpToken_(NClient::FindToken())
        , ReplicasManagementPool_(CreateThreadPool(CONFIG_SNAPSHOT_VALUE(Config_, GetReplicasManagementThreadPoolSize())))
        , LimitedSensorsStorage_(NSensors::MAX_EMPTY_RESPONSES_SENSORS_NUMBER)
    {
        const auto initialConfig = Config_.Get();
        for (const auto& cluster : initialConfig->GetYPEndpointClusterConfigs()) {
            YPEndpointReplicas_.emplace(cluster.GetName(), new NYPReplica::TYPReplica<NYPReplica::TEndpointReplicaObject, NYPReplica::TEndpointSetReplicaObject>(
                Config_.Accessor<NYPReplica::TYPReplicaConfig>("YPEndpointReplicaConfig"),
                Config_.Accessor<NYPReplica::TYPClusterConfig>({"YPEndpointClusterConfigs", cluster.GetName()}),
                NYPReplica::TTableRulesHolder<NYPReplica::TEndpointReplicaObject, NYPReplica::TEndpointSetReplicaObject>(),
                YpToken_, EndpointsReplicaLogger_, GetBackupFormatVersionByYpCluster(cluster.GetName())));
        }
        for (const auto& cluster : initialConfig->GetYPNodeClusterConfigs()) {
            YPNodeReplicas_.emplace(cluster.GetName(), new NYPReplica::TYPReplica<NYPReplica::TPodWithNodeIdKeyReplicaObject>(
                Config_.Accessor<NYPReplica::TYPReplicaConfig>("YPNodeReplicaConfig"),
                Config_.Accessor<NYPReplica::TYPClusterConfig>({"YPNodeClusterConfigs", cluster.GetName()}),
                NYPReplica::TTableRulesHolder<NYPReplica::TPodWithNodeIdKeyReplicaObject>(),
                YpToken_, NodeReplicaLogger_, GetBackupFormatVersionByYpCluster(cluster.GetName())));
        }
        for (const auto& cluster : initialConfig->GetYPPodClusterConfigs()) {
            YPPodReplicas_.emplace(cluster.GetName(), new NYPReplica::TYPReplica<NYPReplica::TPodReplicaObject>(
                Config_.Accessor<NYPReplica::TYPReplicaConfig>("YPPodReplicaConfig"),
                Config_.Accessor<NYPReplica::TYPClusterConfig>({"YPPodClusterConfigs", cluster.GetName()}),
                NYPReplica::TTableRulesHolder<NYPReplica::TPodReplicaObject>(),
                YpToken_, PodsReplicaLogger_, GetBackupFormatVersionByYpCluster(cluster.GetName())));
        }

        NInfra::NMemoryLock::LockSelfMemory(initialConfig->GetMemoryLock(), Logger_.SpawnFrame(), SensorGroup_);

        ConfigHolder_.SetSwitchConfigsCallback([this](const TConfig& oldConfig, const TConfig& newConfig) {
            SwitchConfigsCallback<TYPEndpointReplicas>(oldConfig, newConfig);
            SwitchConfigsCallback<TYPNodeReplicas>(oldConfig, newConfig);
            SwitchConfigsCallback<TYPPodReplicas>(oldConfig, newConfig);
        });

        InitSensors(*Config_);
    }

    TService::~TService() {
        StopReplicas();
    }

    void TService::InitSensors(const TConfig& config) {
        LimitedSensorsStorage_.SetCommonSensorGroup(NInfra::TSensorGroup{"yp_service_discovery"});

        for (const NYP::NYPReplica::TYPClusterConfig& cluster : config.GetYPEndpointClusterConfigs()) {
            auto defaultEmptyResponsesSensor = LimitedSensorsStorage_.Rate(
                ResolveEndpointsSensorGroup_,
                NSensors::RESOLVE_EMPTY_RESPONSES,
                {{NSensors::YP_CLUSTER, cluster.GetName()},
                 {NSensors::ESID, "chaos-service-slave"}}
            );
            Y_ENSURE(defaultEmptyResponsesSensor->Init());
        }
        NInfra::TRateSensor(ResolveEndpointsSensorGroup_, NSensors::RESOLVE_ACCESS_DENIED);

        for (const NYP::NYPReplica::TYPClusterConfig& cluster : config.GetYPNodeClusterConfigs()) {
            auto defaultEmptyResponsesSensor = LimitedSensorsStorage_.Rate(
                ResolveNodeSensorGroup_,
                NSensors::RESOLVE_EMPTY_RESPONSES,
                {{NSensors::YP_CLUSTER, cluster.GetName()},
                 {NSensors::NID, "chaos-service-slave"}}
            );
            Y_ENSURE(defaultEmptyResponsesSensor->Init());
        }
        NInfra::TRateSensor(ResolveNodeSensorGroup_, NSensors::RESOLVE_ACCESS_DENIED);

        for (const NYP::NYPReplica::TYPClusterConfig& cluster : config.GetYPPodClusterConfigs()) {
            auto defaultEmptyResponsesSensor = LimitedSensorsStorage_.Rate(
                ResolvePodsSensorGroup_,
                NSensors::RESOLVE_EMPTY_RESPONSES,
                {{NSensors::YP_CLUSTER, cluster.GetName()},
                 {NSensors::PSID, "chaos-service-slave"}}
            );
            Y_ENSURE(defaultEmptyResponsesSensor->Init());
        }
        NInfra::TRateSensor(ResolvePodsSensorGroup_, NSensors::RESOLVE_ACCESS_DENIED);

        NInfra::TIntGaugeSensor(AdminHttpServiceSensorGroup_, NSensors::REQUESTS_QUEUE_SIZE);
        NInfra::TIntGaugeSensor(AdminHttpServiceSensorGroup_, NSensors::FAILED_REQUESTS_QUEUE_SIZE);
        NInfra::TIntGaugeSensor(DiscoveryHttpServiceSensorGroup_, NSensors::REQUESTS_QUEUE_SIZE);
        NInfra::TIntGaugeSensor(DiscoveryHttpServiceSensorGroup_, NSensors::FAILED_REQUESTS_QUEUE_SIZE);
    }

    void TService::PrintSensors(TStringOutput& stream) const {
        NMonitoring::IMetricEncoderPtr encoder = NInfra::CreateEncoder(stream, NInfra::ESensorsSerializationType::SPACK_V1);
        NInfra::SensorRegistry().Append(TInstant::Zero(), encoder.Get());
    }

    void TService::PrintDynamicSensors(TStringOutput& stream) const {
        NMonitoring::IMetricEncoderPtr encoder = NInfra::CreateEncoder(stream, NInfra::ESensorsSerializationType::SPACK_V1);
        LimitedSensorsStorage_.Append(TInstant::Zero(), encoder.Get());
    }

    void TService::LockSelfMemory() {
        try {
            LockAllMemory(LockCurrentMemory);
        } catch (const yexception& e) {
            NInfra::NLogEvent::TServiceMemoryLockError ev("Failed to lock memory: " + CurrentExceptionMessage());
            Logger_.SpawnFrame()->LogEvent(ELogPriority::TLOG_ERR, ev);
        }
    }

    template <class TReplicas>
    void TService::SwitchConfigsCallback(const TConfig& oldConfig, const TConfig& newConfig) {
        for (int i = 0; i < GetClusterConfigs<TReplicas>(oldConfig).size(); ++i) {
            Y_ENSURE(GetClusterConfigs<TReplicas>(oldConfig)[i].GetName() == GetClusterConfigs<TReplicas>(newConfig)[i].GetName());

            const auto& oldClusterConfig = GetClusterConfigs<TReplicas>(oldConfig)[i];
            const auto& newClusterConfig = GetClusterConfigs<TReplicas>(newConfig)[i];
            if (!Equals(oldClusterConfig.GetRollbackToBackup(), newClusterConfig.GetRollbackToBackup())) {
                if (newClusterConfig.GetRollbackToBackup().GetRollback()) {
                    NApi::TReqRollbackToBackup request;
                    *request.MutableRollbackConfig() = newClusterConfig.GetRollbackToBackup();
                    NApi::TRspRollbackToBackup response;
                    RollbackToBackup<TReplicas>(newClusterConfig.GetName(), request, response);
                }
            }

            if (oldClusterConfig.GetEnableUpdates() != newClusterConfig.GetEnableUpdates()) {
                if (newClusterConfig.GetEnableUpdates()) {
                    NApi::TRspStartUpdates response;
                    StartUpdates<TReplicas>(newClusterConfig.GetName(), NApi::TReqStartUpdates{}, response);
                } else {
                    NApi::TRspStopUpdates response;
                    StopUpdates<TReplicas>(newClusterConfig.GetName(), NApi::TReqStopUpdates{}, response);
                }
            }
        }
    }

    void TService::Start() {
        ConfigHolder_.StartWatchPatch();
        StartReplicas();
        AdminHttpService_.Start(Logger_.SpawnFrame());
        DiscoveryHttpService_.Start(Logger_.SpawnFrame());
        GrpcServer_.Start(Logger_.SpawnFrame());
    }

    void TService::Wait() {
        DiscoveryHttpService_.Wait(Logger_.SpawnFrame());
        AdminHttpService_.Wait(Logger_.SpawnFrame());
        GrpcServer_.Wait(Logger_.SpawnFrame());
    }

    void TService::StartReplicas() {
        auto startEndpointReplicas = [this] {
            TInstant startTime = Now();

            TVector<NThreading::TFuture<void>> replicasStartFutures;
            replicasStartFutures.reserve(YPEndpointReplicas_.size());
            for (const auto& [name, replica] : YPEndpointReplicas_) {
                replicasStartFutures.push_back(NThreading::Async(
                    [replicaPtr = replica.Get()] {
                        replicaPtr->Start();
                    },
                    *ReplicasManagementPool_
                ));
            }

            return NThreading::WaitAll(replicasStartFutures).Subscribe([this, startTime] (const auto&) {
                EndpointsReplicaLogger_.SpawnFrame()->LogEvent(NEventlog::TYPReplicasLoadEnd((Now() - startTime).MicroSeconds()));
            });
        };

        auto startPodReplicas = [this] {
            TInstant startTime = Now();

            TVector<NThreading::TFuture<void>> replicasStartFutures;
            replicasStartFutures.reserve(YPPodReplicas_.size());
            for (const auto& [name, replica] : YPPodReplicas_) {
                replicasStartFutures.push_back(NThreading::Async(
                    [replicaPtr = replica.Get()] {
                        replicaPtr->Start();
                    },
                    *ReplicasManagementPool_
                ));
            }

            return NThreading::WaitAll(replicasStartFutures).Subscribe([this, startTime] (const auto&) {
                PodsReplicaLogger_.SpawnFrame()->LogEvent(NEventlog::TYPReplicasLoadEnd((Now() - startTime).MicroSeconds()));
            });
        };

        auto startNodeReplicas = [this] {
            TInstant startTime = Now();

            TVector<NThreading::TFuture<void>> replicasStartFutures;
            replicasStartFutures.reserve(YPNodeReplicas_.size());
            for (const auto& [name, replica] : YPNodeReplicas_) {
                replicasStartFutures.push_back(NThreading::Async(
                    [replicaPtr = replica.Get()] {
                        replicaPtr->Start();
                    },
                    *ReplicasManagementPool_
                ));
            }

            return NThreading::WaitAll(replicasStartFutures).Subscribe([this, startTime] (const auto&) {
                NodeReplicaLogger_.SpawnFrame()->LogEvent(NEventlog::TYPReplicasLoadEnd((Now() - startTime).MicroSeconds()));
            });
        };

        TVector<NThreading::TFuture<void>> waitFutures = {
            startEndpointReplicas(),
            startNodeReplicas(),
            startPodReplicas(),
        };
        NThreading::WaitAll(waitFutures).GetValueSync();
    }

    void TService::StopReplicas() {
        TVector<NThreading::TFuture<void>> replicasStopFutures;
        replicasStopFutures.reserve(YPEndpointReplicas_.size() + YPPodReplicas_.size());

        for (const auto& [name, replica] : YPEndpointReplicas_) {
            replicasStopFutures.push_back(NThreading::Async(
                [replicaPtr = replica.Get()] {
                    replicaPtr->Stop();
                },
                *ReplicasManagementPool_
            ));
        }

        for (const auto& [name, replica] : YPNodeReplicas_) {
            replicasStopFutures.push_back(NThreading::Async(
                [replicaPtr = replica.Get()] {
                    replicaPtr->Stop();
                },
                *ReplicasManagementPool_
            ));
        }

        for (const auto& [name, replica] : YPPodReplicas_) {
            replicasStopFutures.push_back(NThreading::Async(
                [replicaPtr = replica.Get()] {
                    replicaPtr->Stop();
                },
                *ReplicasManagementPool_
            ));
        }

        NThreading::WaitAll(replicasStopFutures).GetValueSync();
    }

    void TService::Ping(NInfra::TRequestPtr<NApi::TReqPing> request, NInfra::TReplyPtr<NApi::TRspPing> reply) {
        auto frame = NInfra::CreateBiographedLoggerFrame(
            Logger_,
            GetEventRequestStart(NEventlog::ERequestType::PING, *request),
            NEventlog::TRequestStop(),
            GetHistogram(NSensors::PING_RESPONSE_TIME, SensorGroup_)
        );

        NApi::TRspPing result;
        result.set_data("pong");
        reply->Set(result);
    }

    void TService::Shutdown(NInfra::TRequestPtr<NApi::TReqShutdown> request, NInfra::TReplyPtr<NApi::TRspShutdown>) {
        auto frame = NInfra::CreateBiographedLoggerFrame(
            Logger_,
            GetEventRequestStart(NEventlog::ERequestType::SHUTDOWN, *request),
            NEventlog::TRequestStop(),
            GetHistogram(NSensors::SHUTDOWN_RESPONSE_TIME, SensorGroup_)
        );
        const auto config = Config_.Get();

        static THolder<IWatchDog> abortWatchDog = THolder<IWatchDog>(CreateAbortByTimeoutWatchDog(config->GetAbortWatchDogConfig(), "Ooops!"));
        DiscoveryHttpService_.ShutDown();
        AdminHttpService_.ShutDown();
        GrpcServer_.Shutdown();
    }

    void TService::ReopenLog(NInfra::TRequestPtr<NApi::TReqReopenLog> request, NInfra::TReplyPtr<NApi::TRspReopenLog> reply) {
        auto frame = NInfra::CreateBiographedLoggerFrame(
            Logger_,
            GetEventRequestStart(NEventlog::ERequestType::REOPEN_LOG, *request),
            NEventlog::TRequestStop(),
            GetHistogram(NSensors::REOPEN_LOG_RESPONSE_TIME, SensorGroup_)
        );

        NApi::TRspReopenLog result;
        result.set_data("ok");
        reply->Set(result);

        Logger_.ReopenLog();
        EndpointsReplicaLogger_.ReopenLog();
        NodeReplicaLogger_.ReopenLog();
        PodsReplicaLogger_.ReopenLog();
        ConfigHolder_.ReopenLog();
        AdminHttpServiceLogger_.ReopenLog();
        DiscoveryHttpServiceLogger_.ReopenLog();
    }

    void TService::Sensors(NInfra::TRequestPtr<NApi::TReqSensors> request, NInfra::TReplyPtr<NApi::TRspSensors> reply) {
        auto frame = NInfra::CreateBiographedLoggerFrame(
            Logger_,
            GetEventRequestStart(NEventlog::ERequestType::SENSORS, *request),
            NEventlog::TRequestStop(),
            GetHistogram(NSensors::SENSORS_RESPONSE_TIME, SensorGroup_)
        );

        for (const auto& [name, replica] : YPEndpointReplicas_) {
            Y_ENSURE(replica);
            replica->UpdateSensors();
        }
        for (const auto& [name, replica] : YPNodeReplicas_) {
            Y_ENSURE(replica);
            replica->UpdateSensors();
        }
        for (const auto& [name, replica] : YPPodReplicas_) {
            Y_ENSURE(replica);
            replica->UpdateSensors();
        }

        NInfra::TIntGaugeSensor(AdminHttpServiceSensorGroup_, NSensors::REQUESTS_QUEUE_SIZE).Set(AdminHttpService_.GetRequestQueueSize());
        NInfra::TIntGaugeSensor(DiscoveryHttpServiceSensorGroup_, NSensors::REQUESTS_QUEUE_SIZE).Set(DiscoveryHttpService_.GetRequestQueueSize());
        NInfra::TIntGaugeSensor(AdminHttpServiceSensorGroup_, NSensors::FAILED_REQUESTS_QUEUE_SIZE).Set(AdminHttpService_.GetFailQueueSize());
        NInfra::TIntGaugeSensor(DiscoveryHttpServiceSensorGroup_, NSensors::FAILED_REQUESTS_QUEUE_SIZE).Set(DiscoveryHttpService_.GetFailQueueSize());

        NApi::TRspSensors result;
        TStringOutput stream(*result.MutableData());
        PrintSensors(stream);

        reply->SetAttribute("Content-Type", "application/x-solomon-spack");

        reply->Set(result);
    }

    void TService::DynamicSensors(NInfra::TRequestPtr<NApi::TReqSensors> /* request */, NInfra::TReplyPtr<NApi::TRspSensors> reply) {
        NApi::TRspSensors result;
        TStringOutput stream(*result.MutableData());
        PrintDynamicSensors(stream);

        reply->SetAttribute("Content-Type", "application/x-solomon-spack");

        reply->Set(result);
    }

    void TService::ResolveEndpoints(NInfra::TRequestPtr<NApi::TReqResolveEndpoints> request, NInfra::TReplyPtr<NApi::TRspResolveEndpoints> reply) {
        const TString cluster = request->Get().cluster_name();
        TVector<std::pair<TStringBuf, TStringBuf>> labels;
        labels.emplace_back(NSensors::YP_CLUSTER, cluster);
        auto [logFrame, bio] = NInfra::CreateBiographedLoggerFrame(
            Logger_,
            GetEventRequestStart(NEventlog::ERequestType::RESOLVE_ENDPOINTS, *request),
            NEventlog::TRequestStop(),
            GetHistogram(NSensors::RESOLVE_RESPONSE_TIME, ResolveEndpointsSensorGroup_, labels)
        );

        NInfra::TRateSensor(ResolveEndpointsSensorGroup_, NSensors::RESOLVE_REQUESTS, labels).Inc();

        const TString& endpointSetId = request->Get().endpoint_set_id();
        const TString& clientName = request->Get().client_name();
        const auto& labelSelectors = request->Get().label_selectors();
        const auto& esLabelSelectors = request->Get().endpoint_set_label_selectors();
        TString ruid = request->Get().ruid();

        if (ruid.empty()) {
            ruid = GenerateRUID();
        }

        logFrame->LogEvent(NEventlog::TResolveEndpointsRequestData(
            cluster,
            endpointSetId,
            clientName,
            ruid
        ));

        if (cluster.empty()) {
            logFrame->LogEvent(NEventlog::TResolveEndpointsResult(
                /* YpTimestamp */ 0,
                NEventlog::TResolveEndpointsResult::ERROR,
                /* EndpointSetSize */ 0,
                EMPTY_CLUSTER_NAME_MESSAGE,
                /* ByteSize */ 0
            ));
            ythrow TEmptyClusterNameError() << EMPTY_CLUSTER_NAME_MESSAGE;
        }

        if (clientName.empty()) {
            NInfra::TRateSensor(ResolveEndpointsSensorGroup_, NSensors::RESOLVE_ACCESS_DENIED).Inc();
            logFrame->LogEvent(NEventlog::TResolveEndpointsResult(
                /* YpTimestamp */ 0,
                NEventlog::TResolveEndpointsResult::ERROR,
                /* EndpointSetSize */ 0,
                ACCESS_DENIED_MESSAGE,
                /* ByteSize */ 0
            ));
            ythrow NInfra::TAccessDeniedError() << ACCESS_DENIED_MESSAGE;
        }

        NApi::TRspResolveEndpoints result;
        result.set_host(HostName());
        result.set_ruid(ruid);
        if (auto it = YPEndpointReplicas_.find(cluster); it != YPEndpointReplicas_.end()) {
            const auto& replica = it->second;

            auto snapshot = replica->GetReplicaSnapshot();
            const auto selectionResult = replica->GetByKey<NYPReplica::TEndpointReplicaObject>(endpointSetId, snapshot);
            const auto esSelectionResult = replica->GetByKey<NYPReplica::TEndpointSetReplicaObject>(endpointSetId, snapshot);
            const ui64 ypTimestamp = *replica->GetYpTimestamp(snapshot);
            snapshot.Release();

            result.set_timestamp(ypTimestamp);

            if (selectionResult.Defined()) {
                const TString selectionResultHash = ToString(selectionResult->Hash);
                if (request->Get().watch_token() == selectionResultHash) {
                    result.set_resolve_status(NApi::EResolveStatus::NOT_CHANGED);
                    NInfra::TRateSensor(ResolveEndpointsSensorGroup_, NSensors::RESOLVE_NOT_CHANGED_RESPONSES).Inc();
                } else {
                    auto* endpointSet = result.mutable_endpoint_set();
                    result.set_resolve_status(NApi::EResolveStatus::OK);
                    result.set_watch_token(selectionResultHash);
                    for (const auto& item : selectionResult->Objects) {
                        auto* endpoint = endpointSet->add_endpoints();
                        if (item.Meta().id()) {
                            endpoint->set_id(item.Meta().id());
                        }
                        if (item.Spec().has_protocol()) {
                            endpoint->set_protocol(item.Spec().protocol());
                        }
                        if (item.Spec().has_fqdn()) {
                            endpoint->set_fqdn(item.Spec().fqdn());
                        }
                        if (item.Spec().has_ip4_address()) {
                            endpoint->set_ip4_address(item.Spec().ip4_address());
                        }
                        if (item.Spec().has_ip6_address()) {
                            endpoint->set_ip6_address(item.Spec().ip6_address());
                        }
                        if (item.Spec().has_port()) {
                            endpoint->set_port(item.Spec().port());
                        }
                        endpoint->set_ready(item.Status().ready());
                        for (const auto& pathToLabel : labelSelectors) {
                            auto* selectionResult = endpoint->add_label_selector_results();
                            if (item.HasLabels()) {
                                const auto* node = item.Labels().GetValueByPath(pathToLabel, '/');
                                if (node) {
                                    *selectionResult = node->GetStringRobust();
                                } else {
                                    *selectionResult = "null";
                                }
                            }
                        }
                    }
                }
            } else {
                if (esSelectionResult.Defined()) {
                    result.set_resolve_status(NApi::EResolveStatus::EMPTY);
                } else {
                    result.set_resolve_status(NApi::EResolveStatus::NOT_EXISTS);
                }
                if (!SKIPPED_CLIENTS_FOR_EMPTY_RESPONSES_SENSOR.contains(clientName)) {
                    LimitedSensorsStorage_.Rate(ResolveEndpointsSensorGroup_, NSensors::RESOLVE_EMPTY_RESPONSES, {{NSensors::YP_CLUSTER, cluster}, {NSensors::ESID, endpointSetId}})->Inc();
                }
            }

            if (esSelectionResult.Defined()) {
                auto* endpointSet = result.mutable_endpoint_set();
                endpointSet->set_endpoint_set_id(endpointSetId);

                for (const auto& pathToLabel : esLabelSelectors) {
                    auto* selectionResult = endpointSet->add_label_selector_results();
                    const auto& esItem = esSelectionResult->Objects.front();
                    if (esItem.HasLabels()) {
                        const auto* node = esItem.Labels().GetValueByPath(pathToLabel, '/');
                        if (node) {
                            *selectionResult = node->GetStringRobust();
                        } else {
                            *selectionResult = "null";
                        }
                    }
                }
            }
        }

        labels.emplace_back(NSensors::RESOLVE_ENDPOINTS_STATUS, NApi::EResolveStatus_Name(result.resolve_status()));
        NInfra::TRateSensor(ResolveEndpointsSensorGroup_, NSensors::RESOLVE_SUCCESSES, labels).Inc();
        GetHistogram(NSensors::RESOLVE_RESULT_SIZE, ResolveEndpointsSensorGroup_)->Add(result.ByteSizeLong());

        logFrame->LogEvent(MakeResolveResultEvent(result));

        reply->Set(result);
    }

    void TService::ResolveNode(NInfra::TRequestPtr<NApi::TReqResolveNode> request, NInfra::TReplyPtr<NApi::TRspResolveNode> reply) {
        const TString cluster = request->Get().cluster_name();
        TVector<std::pair<TStringBuf, TStringBuf>> labels;
        labels.emplace_back(NSensors::YP_CLUSTER, cluster);
        auto [logFrame, bio] = NInfra::CreateBiographedLoggerFrame(
            Logger_,
            GetEventRequestStart(NEventlog::ERequestType::RESOLVE_NODE, *request),
            NEventlog::TRequestStop(),
            GetHistogram(NSensors::RESOLVE_RESPONSE_TIME, ResolveNodeSensorGroup_, labels)
        );

        NInfra::TRateSensor(ResolveNodeSensorGroup_, NSensors::RESOLVE_REQUESTS, labels).Inc();

        const TString& nodeId = request->Get().node_id();
        const TString& clientName = request->Get().client_name();
        TString ruid = request->Get().ruid();
 
        if (ruid.empty()) {
            ruid = GenerateRUID();
        }

        logFrame->LogEvent(NEventlog::TResolveNodeRequestData(
            cluster,
            nodeId,
            clientName,
            ruid
        ));

        if (cluster.empty()) {
            logFrame->LogEvent(NEventlog::TResolveNodeResult(
                /* YpTimestamp */ 0,
                NEventlog::TResolveNodeResult::ERROR,
                /* NumberOfPods */ 0,
                EMPTY_CLUSTER_NAME_MESSAGE,
                /* ByteSize */ 0
            ));
            ythrow TEmptyClusterNameError() << EMPTY_CLUSTER_NAME_MESSAGE;
        }

        if (clientName.empty()) {
            NInfra::TRateSensor(ResolveNodeSensorGroup_, NSensors::RESOLVE_ACCESS_DENIED).Inc();
            logFrame->LogEvent(NEventlog::TResolveNodeResult(
                /* YpTimestamp */ 0,
                NEventlog::TResolveNodeResult::ERROR,
                /* NumberOfPods */ 0,
                ACCESS_DENIED_MESSAGE,
                /* ByteSize */ 0
            ));
            ythrow NInfra::TAccessDeniedError() << ACCESS_DENIED_MESSAGE;
        }

        NApi::TRspResolveNode result;
        result.set_host(HostName());
        result.set_ruid(ruid);
        if (auto it = YPNodeReplicas_.find(cluster); it != YPNodeReplicas_.end()) {
            const auto& replica = it->second;

            auto snapshot = replica->GetReplicaSnapshot();
            const auto selectionResult = replica->GetByKey<NYPReplica::TPodWithNodeIdKeyReplicaObject>(nodeId, snapshot);
            const ui64 ypTimestamp = *replica->GetYpTimestamp(snapshot);
            snapshot.Release();

            result.set_timestamp(ypTimestamp);

            if (selectionResult.Defined()) {
                result.set_resolve_status(NApi::EResolveStatus::OK);
                for (const auto& item : selectionResult->Objects) {
                    auto* pod = result.add_pods();

                    pod->set_id(item.Meta().id());

                    if (item.Status().has_dns()) {
                        auto resultDns = pod->mutable_dns();
                        if (item.Status().dns().has_persistent_fqdn()) {
                            resultDns->set_persistent_fqdn(item.Status().dns().persistent_fqdn());
                        }
                    }
                }
            } else {
                result.set_resolve_status(NApi::EResolveStatus::NOT_EXISTS);
                if (!SKIPPED_CLIENTS_FOR_EMPTY_RESPONSES_SENSOR.contains(clientName)) {
                    LimitedSensorsStorage_.Rate(ResolveNodeSensorGroup_, NSensors::RESOLVE_EMPTY_RESPONSES, {{NSensors::YP_CLUSTER, cluster}, {NSensors::NID, nodeId}})->Inc();
                }
            }
        }

        labels.emplace_back(NSensors::RESOLVE_NODE_STATUS, NApi::EResolveStatus_Name(result.resolve_status()));
        NInfra::TRateSensor(ResolveNodeSensorGroup_, NSensors::RESOLVE_SUCCESSES, labels).Inc();
        GetHistogram(NSensors::RESOLVE_RESULT_SIZE, ResolveNodeSensorGroup_)->Add(result.ByteSizeLong());

        logFrame->LogEvent(MakeResolveResultEvent(result));

        reply->Set(result);
    }

    void TService::ResolvePods(NInfra::TRequestPtr<NApi::TReqResolvePods> request, NInfra::TReplyPtr<NApi::TRspResolvePods> reply) {
        const TString cluster = request->Get().cluster_name();
        TVector<std::pair<TStringBuf, TStringBuf>> labels;
        labels.emplace_back(NSensors::YP_CLUSTER, cluster);
        auto [logFrame, bio] = NInfra::CreateBiographedLoggerFrame(
            Logger_,
            GetEventRequestStart(NEventlog::ERequestType::RESOLVE_PODS, *request),
            NEventlog::TRequestStop(),
            GetHistogram(NSensors::RESOLVE_RESPONSE_TIME, ResolvePodsSensorGroup_, labels)
        );

        NInfra::TRateSensor(ResolvePodsSensorGroup_, NSensors::RESOLVE_REQUESTS, labels).Inc();

        const TString& podSetId = request->Get().pod_set_id();
        const TString& clientName = request->Get().client_name();
        const auto& labelSelectors = request->Get().pod_label_selectors();
        TString ruid = request->Get().ruid();

        if (ruid.empty()) {
            ruid = GenerateRUID();
        }

        logFrame->LogEvent(NEventlog::TResolvePodsRequestData(
            cluster,
            podSetId,
            clientName,
            ruid
        ));

        if (cluster.empty()) {
            logFrame->LogEvent(NEventlog::TResolvePodsResult(
                /* YpTimestamp */ 0,
                NEventlog::TResolvePodsResult::ERROR,
                /* PodSetSize */ 0,
                EMPTY_CLUSTER_NAME_MESSAGE,
                /* ByteSize */ 0
            ));
            ythrow TEmptyClusterNameError() << EMPTY_CLUSTER_NAME_MESSAGE;
        }

        if (clientName.empty()) {
            NInfra::TRateSensor(ResolvePodsSensorGroup_, NSensors::RESOLVE_ACCESS_DENIED).Inc();
            logFrame->LogEvent(NEventlog::TResolvePodsResult(
                /* YpTimestamp */ 0,
                NEventlog::TResolvePodsResult::ERROR,
                /* PodSetSize */ 0,
                ACCESS_DENIED_MESSAGE,
                /* ByteSize */ 0
            ));
            ythrow NInfra::TAccessDeniedError() << ACCESS_DENIED_MESSAGE;
        }

        NApi::TRspResolvePods result;
        result.set_host(HostName());
        result.set_ruid(ruid);
        if (auto it = YPPodReplicas_.find(cluster); it != YPPodReplicas_.end()) {
            const auto& replica = it->second;

            auto snapshot = replica->GetReplicaSnapshot();
            const auto selectionResult = replica->GetByKey<NYPReplica::TPodReplicaObject>(podSetId, snapshot);
            const ui64 ypTimestamp = *replica->GetYpTimestamp(snapshot);
            snapshot.Release();

            result.set_timestamp(ypTimestamp);

            if (selectionResult.Defined()) {
                auto* podSet = result.mutable_pod_set();
                result.set_resolve_status(NApi::EResolveStatus::OK);
                for (const auto& item : selectionResult->Objects) {
                    auto* pod = podSet->add_pods();

                    pod->set_id(item.Meta().id());

                    for (const auto& pathToLabel : labelSelectors) {
                        auto* selectionResult = pod->add_pod_label_selector_results();
                        if (item.HasLabels()) {
                            const auto* node = item.Labels().GetValueByPath(pathToLabel, '/');
                            if (node) {
                                *selectionResult = node->GetStringRobust();
                            } else {
                                *selectionResult = "null";
                            }
                        } else {
                            *selectionResult = "null";
                        }
                    }

                    if (item.Spec().has_node_id()) {
                        pod->set_node_id(item.Spec().node_id());
                    }

                    for (const auto& addressAllocation : item.Status().ip6_address_allocations()) {
                        auto resultAddressAllocation = pod->mutable_ip6_address_allocations()->Add();
                        if (addressAllocation.has_address()) {
                            resultAddressAllocation->set_address(addressAllocation.address());
                        }
                        if (addressAllocation.has_vlan_id()) {
                            resultAddressAllocation->set_vlan_id(addressAllocation.vlan_id());
                        }
                        if (addressAllocation.has_persistent_fqdn()) {
                            resultAddressAllocation->set_persistent_fqdn(addressAllocation.persistent_fqdn());
                        }
                        if (addressAllocation.has_transient_fqdn()) {
                            resultAddressAllocation->set_transient_fqdn(addressAllocation.transient_fqdn());
                        }
                        if (addressAllocation.has_internet_address()) {
                            resultAddressAllocation->mutable_internet_address()->set_id(addressAllocation.internet_address().id());
                            resultAddressAllocation->mutable_internet_address()->set_ip4_address(addressAllocation.internet_address().ip4_address());
                        }
                        for (const auto& service : addressAllocation.virtual_services()) {
                            auto resultService = resultAddressAllocation->mutable_virtual_services()->Add();
                            resultService->mutable_ip6_addresses()->CopyFrom(service.ip6_addresses());
                            resultService->mutable_ip4_addresses()->CopyFrom(service.ip4_addresses());
                        }
                    }

                    for (const auto& subnet : item.Status().ip6_subnet_allocations()) {
                        auto resultSubnet = pod->mutable_ip6_subnet_allocations()->Add();
                        if (subnet.has_subnet()) {
                            resultSubnet->set_subnet(subnet.subnet());
                        }
                        if (subnet.has_vlan_id()) {
                            resultSubnet->set_vlan_id(subnet.vlan_id());
                        }
                    }

                    if (item.Status().has_dns()) {
                        auto resultDns = pod->mutable_dns();
                        if (item.Status().dns().has_persistent_fqdn()) {
                            resultDns->set_persistent_fqdn(item.Status().dns().persistent_fqdn());
                        }
                        if (item.Status().dns().has_transient_fqdn()) {
                            resultDns->set_transient_fqdn(item.Status().dns().transient_fqdn());
                        }
                    }

                    for (const auto& issConfSummary : item.Status().iss_conf_summaries()) {
                        (*pod->mutable_iss_conf_summaries())[issConfSummary.first].set_target_state(issConfSummary.second.target_state());
                    }

                    if (item.Status().has_agent() && item.Status().agent().has_iss_summary()) {
                        auto resultIssSummary = pod->mutable_agent()->mutable_iss_summary();
                        if (item.Status().agent().iss_summary().has_ready()) {
                            auto ready = resultIssSummary->mutable_ready();
                            ready->set_status(item.Status().agent().iss_summary().ready().status());
                            ready->set_reason(item.Status().agent().iss_summary().ready().reason());
                            ready->set_message(item.Status().agent().iss_summary().ready().message());
                            *ready->mutable_last_transition_time() = item.Status().agent().iss_summary().ready().last_transition_time();
                        }

                        for (const auto& stateSummary : item.Status().agent().iss_summary().state_summaries()) {
                            (*resultIssSummary->mutable_state_summaries())[stateSummary.first].set_current_state(stateSummary.second.current_state());

                            auto ready = (*resultIssSummary->mutable_state_summaries())[stateSummary.first].mutable_ready();
                            ready->set_status(stateSummary.second.ready().status());
                            ready->set_reason(stateSummary.second.ready().reason());
                            ready->set_message(stateSummary.second.ready().message());
                            *ready->mutable_last_transition_time() = stateSummary.second.ready().last_transition_time();

                            auto installed = (*resultIssSummary->mutable_state_summaries())[stateSummary.first].mutable_installed();
                            installed->set_status(stateSummary.second.installed().status());
                            installed->set_reason(stateSummary.second.installed().reason());
                            installed->set_message(stateSummary.second.installed().message());
                            *installed->mutable_last_transition_time() = stateSummary.second.installed().last_transition_time();
                        }
                    }

                    if (item.Status().has_agent() && item.Status().agent().has_pod_agent_payload()) {
                        auto resultPodAgentPayload = pod->mutable_agent()->mutable_pod_agent_payload();
                        if (item.Status().agent().pod_agent_payload().has_status()) {
                            auto resultPodAgentPayloadStatus = resultPodAgentPayload->mutable_status();
                            for (const auto& box : item.Status().agent().pod_agent_payload().status().boxes()) {
                                auto resultBox = resultPodAgentPayloadStatus->mutable_boxes()->Add();
                                resultBox->set_id(box.id());
                                resultBox->set_ip6_address(box.ip6_address());
                            }
                        }
                    }
                }
            } else {
                result.set_resolve_status(NApi::EResolveStatus::NOT_EXISTS);
                if (!SKIPPED_CLIENTS_FOR_EMPTY_RESPONSES_SENSOR.contains(clientName)) {
                    LimitedSensorsStorage_.Rate(ResolvePodsSensorGroup_, NSensors::RESOLVE_EMPTY_RESPONSES, {{NSensors::YP_CLUSTER, cluster}, {NSensors::PSID, podSetId}})->Inc();
                }
            }
        }

        labels.emplace_back(NSensors::RESOLVE_PODS_STATUS, NApi::EResolveStatus_Name(result.resolve_status()));
        NInfra::TRateSensor(ResolvePodsSensorGroup_, NSensors::RESOLVE_SUCCESSES, labels).Inc();
        GetHistogram(NSensors::RESOLVE_RESULT_SIZE, ResolvePodsSensorGroup_)->Add(result.ByteSizeLong());

        logFrame->LogEvent(MakeResolveResultEvent(result));

        reply->Set(result);
    }

    void TService::Config(NInfra::TRequestPtr<NApi::TReqConfig>, NInfra::TReplyPtr<NApi::TRspConfig> reply) {
        NApi::TRspConfig result;
        TStringOutput stream(*result.MutableData());
        NJson::TJsonWriter writer(&stream, /* formatOutput */ true);
        NProtobufJson::Proto2Json(*Config_, writer);

        reply->Set(result);
    }

    template <class TReplicas>
    void TService::ListBackups(NInfra::TRequestPtr<NApi::TReqListBackups> request, NInfra::TReplyPtr<NApi::TRspListBackups> reply) {
        const TStringBuf clusterName = GetYpClusterName(request);

        NApi::TRspListBackups result;
        if (const auto* replicaPtr = GetReplicas<TReplicas>().FindPtr(clusterName); replicaPtr && replicaPtr->Get()) {
            const TVector<NYPReplica::TBackupInfo> infos = replicaPtr->Get()->ListBackups();
            for (const NYPReplica::TBackupInfo& info : infos) {
                NApi::TBackupInfo& backupInfo = *result.AddBackupInfos();
                backupInfo.SetId(info.Id);
                backupInfo.SetTimestamp(info.Timestamp);
                backupInfo.SetSize(info.Size);
                backupInfo.SetNumberFiles(info.NumberFiles);
                *backupInfo.MutableMeta() = info.Meta;
            }
        }
        reply->Set(result);
    }

    void TService::ListEndpointBackups(NInfra::TRequestPtr<NApi::TReqListBackups> request, NInfra::TReplyPtr<NApi::TRspListBackups> reply) {
        ListBackups<TYPEndpointReplicas>(request, reply);
    }

    void TService::ListPodBackups(NInfra::TRequestPtr<NApi::TReqListBackups> request, NInfra::TReplyPtr<NApi::TRspListBackups> reply) {
        ListBackups<TYPPodReplicas>(request, reply);
    }

    void TService::ListNodeBackups(NInfra::TRequestPtr<NApi::TReqListBackups> request, NInfra::TReplyPtr<NApi::TRspListBackups> reply) {
        ListBackups<TYPNodeReplicas>(request, reply);
    }

    template <class TReplicas>
    void TService::RollbackToBackup(const TStringBuf clusterName, const NApi::TReqRollbackToBackup& request, NApi::TRspRollbackToBackup& response) {
        auto* replicaPtr = GetReplicas<TReplicas>().FindPtr(clusterName);

        if (!replicaPtr || !replicaPtr->Get()) {
            response.SetStatus(TStringBuilder() << "replica " << clusterName << " not found");
            return;
        }

        try {
            NYPReplica::TRollbackToBackupConfig rollbackConfig = request.GetRollbackConfig();
            rollbackConfig.SetRollback(true);

            if (replicaPtr->Get()->RollbackToBackup(rollbackConfig)) {
                response.SetStatus("ok");
            } else {
                response.SetStatus("failed");
            }
        } catch (...) {
            response.SetStatus(TStringBuilder() << "failed: " << CurrentExceptionMessage());
        }
    }

    void TService::RollbackToEndpointBackup(NInfra::TRequestPtr<NApi::TReqRollbackToBackup> request, NInfra::TReplyPtr<NApi::TRspRollbackToBackup> reply) {
        NApi::TRspRollbackToBackup response;
        RollbackToBackup<TYPEndpointReplicas>(GetYpClusterName(request), request->Get(), response);
        reply->Set(response);
    }

    void TService::RollbackToNodeBackup(NInfra::TRequestPtr<NApi::TReqRollbackToBackup> request, NInfra::TReplyPtr<NApi::TRspRollbackToBackup> reply) {
        NApi::TRspRollbackToBackup response;
        RollbackToBackup<TYPNodeReplicas>(GetYpClusterName(request), request->Get(), response);
        reply->Set(response);
    }

    void TService::RollbackToPodBackup(NInfra::TRequestPtr<NApi::TReqRollbackToBackup> request, NInfra::TReplyPtr<NApi::TRspRollbackToBackup> reply) {
        NApi::TRspRollbackToBackup response;
        RollbackToBackup<TYPPodReplicas>(GetYpClusterName(request), request->Get(), response);
        reply->Set(response);
    }

    template <class TReplicas>
    void TService::StartUpdates(const TStringBuf clusterName, const NApi::TReqStartUpdates&, NApi::TRspStartUpdates& response) {
        auto* replicaPtr = GetReplicas<TReplicas>().FindPtr(clusterName);

        if (!replicaPtr || !replicaPtr->Get()) {
            response.SetData(TStringBuilder() << "replica " << clusterName << " not found");
            return;
        }

        replicaPtr->Get()->EnableUpdates();
        response.SetData("ok");
    }

    void TService::StartEndpointUpdates(NInfra::TRequestPtr<NApi::TReqStartUpdates> request, NInfra::TReplyPtr<NApi::TRspStartUpdates> reply) {
        NApi::TRspStartUpdates response;
        StartUpdates<TYPEndpointReplicas>(GetYpClusterName(request), request->Get(), response);
        reply->Set(response);
    }

    void TService::StartNodeUpdates(NInfra::TRequestPtr<NApi::TReqStartUpdates> request, NInfra::TReplyPtr<NApi::TRspStartUpdates> reply) {
        NApi::TRspStartUpdates response;
        StartUpdates<TYPNodeReplicas>(GetYpClusterName(request), request->Get(), response);
        reply->Set(response);
    }

    void TService::StartPodUpdates(NInfra::TRequestPtr<NApi::TReqStartUpdates> request, NInfra::TReplyPtr<NApi::TRspStartUpdates> reply) {
        NApi::TRspStartUpdates response;
        StartUpdates<TYPPodReplicas>(GetYpClusterName(request), request->Get(), response);
        reply->Set(response);
    }

    template <class TReplicas>
    void TService::StopUpdates(const TStringBuf clusterName, const NApi::TReqStopUpdates&, NApi::TRspStopUpdates& response) {
        auto* replicaPtr = GetReplicas<TReplicas>().FindPtr(clusterName);

        if (!replicaPtr || !replicaPtr->Get()) {
            response.SetData(TStringBuilder() << "replica " << clusterName << " not found");
            return;
        }

        replicaPtr->Get()->DisableUpdates();
        response.SetData("ok");
    }

    NInfra::TLogger* TService::GetLogger() {
        return &Logger_;
    }

    void TService::StopEndpointUpdates(NInfra::TRequestPtr<NApi::TReqStopUpdates> request, NInfra::TReplyPtr<NApi::TRspStopUpdates> reply) {
        NApi::TRspStopUpdates response;
        StopUpdates<TYPEndpointReplicas>(GetYpClusterName(request), request->Get(), response);
        reply->Set(response);
    }

    void TService::StopNodeUpdates(NInfra::TRequestPtr<NApi::TReqStopUpdates> request, NInfra::TReplyPtr<NApi::TRspStopUpdates> reply) {
        NApi::TRspStopUpdates response;
        StopUpdates<TYPNodeReplicas>(GetYpClusterName(request), request->Get(), response);
        reply->Set(response);
    }

    void TService::StopPodUpdates(NInfra::TRequestPtr<NApi::TReqStopUpdates> request, NInfra::TReplyPtr<NApi::TRspStopUpdates> reply) {
        NApi::TRspStopUpdates response;
        StopUpdates<TYPPodReplicas>(GetYpClusterName(request), request->Get(), response);
        reply->Set(response);
    }

}
