import copy
import juggler_sdk

from infra.yp_service_discovery.monitoring.solomon.src.util import util

from infra.libs.monitoring_lib.docs_builder.alert.description import AlertDescription, JuggleCheckDescription
from infra.libs.monitoring_lib.juggler.check import JugglerCheck, JugglerCheckOptionalArgs
from infra.libs.monitoring_lib.juggler.meta import Url, Meta
from infra.libs.monitoring_lib.templates.discovery.discovery import AutoUrls


def get_access_denied_alert(service_id, geo, period_seconds=300):
    result = {
        'id': service_id,
        'periodMillis': period_seconds * 1000,
        'type': {
            'threshold': {
                'selectors': '{' + ', '.join([
                    'cluster="main"',
                    'service="production"',
                    'host="{}"'.format('|'.join(geo)),
                    'sensor="yp_service_discovery.service.resolve_endpoints.access_denied"',
                ]) + '}',
                'timeAggregation': 'SUM',
                'predicate': 'GT',
                'threshold': 20
            }
        },
        'name': 'Number of denied requests',
        'delaySeconds': 20,
        'notificationChannels': [
            'juggler',
        ],
        'annotations': {
            'service': util.get_juggler_service(service_id),
        },
        'groupByLabels': [
            'host',
        ]
    }
    return result


def get_empty_responses_alert(service_id, geo, masters, environment='Prod', period_seconds=300):
    result = {
        'id': service_id,
        'periodMillis': period_seconds * 1000,
        'type': {
            'threshold': {
                'selectors': '{' + ', '.join([
                    'cluster="main"',
                    'service="production|dynamic"',
                    'host="{}"'.format('|'.join(geo)),
                    'sensor="yp_service_discovery.service.resolve_endpoints.empty_responses"',
                    'yp_cluster="{}"'.format('|'.join(masters)),
                    'endpoint_set_id="any"',
                ]) + '}',
                'timeAggregation': 'SUM',
                'predicate': 'GT',
                'threshold': 6000
            }
        },
        'name': 'Number of empty responses. {} environment'.format(environment),
        'delaySeconds': 20,
        'notificationChannels': [
            'juggler',
        ],
        'annotations': {
            'service': util.get_juggler_service(service_id),
        },
        'groupByLabels': [
            'host',
            'yp_cluster',
        ]
    }
    return result


def get_empty_responses_meta():
    return Meta(
        urls=AutoUrls([
            Url(
                title="Production clusters",
                url="https://solomon.yandex-team.ru/admin/projects/service_discovery/alerts/sd-requests-empty-responses"
            ),
            Url(
                title="Testing clusters",
                url="https://solomon.yandex-team.ru/admin/projects/service_discovery/alerts/sd-requests-empty-responses-test"
            )
        ])
    )


def get_empty_responses_description(service_id):
    return JuggleCheckDescription(
        meaning="Общее количество ответов с пустым списком endpoint-ов на запросы резолва по endpoint set id превышает какой-то числовой лимит",
        cases=AlertDescription.Cases([
            AlertDescription.Case(
                cause="Какой-то из сервисов делает запросы на пустые/несуществующие endpoint set-ы",
                solution="Посмотреть в логи и понять из какого сервиса приходят запросы. Спросить ожидаемое ли поведение, попросить так не делать :D",
                example="[https://st.yandex-team.ru/SPI-35380](https://st.yandex-team.ru/SPI-35380)"
            ),
        ]),
        knowledgeable=AlertDescription.Knowledgeable(knowledgeable=["ismagilas", "azuremint"]),
    )


def get_http_service_errors_alert(service_id, geo, sensor, threshold, name, period_seconds=120):
    result = {
        'id': service_id,
        'periodMillis': period_seconds * 1000,
        'type': {
            'threshold': {
                'selectors': '{' + ', '.join([
                    'cluster="main"',
                    'service="production"',
                    'host="{}"'.format('|'.join(geo)),
                    'sensor="yp_service_discovery.service.http_service.' + sensor + '"',
                ]) + '}',
                'timeAggregation': 'SUM',
                'predicate': 'GT',
                'threshold': threshold
            }
        },
        'name': name,
        'delaySeconds': 20,
        'notificationChannels': [
            'juggler',
        ],
        'annotations': {
            'service': util.get_juggler_service(service_id),
        },
        'groupByLabels': [
            'host',
        ]
    }
    return result


def update_juggler_with_project(project, service_id, geo_list, alert_time_window, phone_escalation=True, start_flow=False, title=None, meta=None, description=None):
    for geo in geo_list:
        host = '{}.service_discovery'.format(geo)

        if not meta:
            meta = Meta()

        meta.other_field |= {'title': title}

        tags = []
        if start_flow:
            tags += [
                'warden_alert_create_spi',
                'warden_alert_start_flow',
                'warden_auto_source',
                'warden_alert_category_boolean',
                'warden_functionality_yp_yp-service-discovery_resolve',
            ]

        notifications=[
            util.notification(
                status={"from": "OK", "to": "CRIT"},
                logins=util.get_duty_logins() + util.get_system_logins(),
                notify_methods=[
                    'sms',
                    'telegram',
                ],
            ),
            util.notification(
                status={"from": "CRIT", "to": "OK"},
                logins=util.get_duty_logins() + util.get_system_logins(),
                notify_methods=[
                    'telegram',
                ],
            )
        ]
        if phone_escalation:
            notifications += [
                juggler_sdk.NotificationOptions(
                    template_name='phone_escalation',
                    template_kwargs={
                        'delay': 120,
                        'logins': util.get_duty_logins() + [
                            'elshiko',
                            'avitella',
                        ],
                    }
                )
            ]

        juggler_check = JugglerCheck(
            host=host,
            service=util.get_juggler_service(service_id),
            optional_args=JugglerCheckOptionalArgs(
                meta=copy.deepcopy(meta),
                other_args={
                    "namespace": 'yp.service_discovery',
                    "tags": tags,
                    "aggregator": 'logic_or',
                    "aggregator_kwargs": {
                        'unreach_checks': util.solomon_lag_checks(),
                    },
                    "flaps_config": juggler_sdk.FlapOptions(stable=alert_time_window + 30, critical=alert_time_window + 90, boost=0),
                    "notifications": notifications,
                }
            )
        )

        project.update_juggler_check(juggler_check, description)


def apply(geo, test_masters, prod_masters, juggler_api, solomon_api, project):
    service_id = 'sd-requests-access-denied'
    alert_time_window = 300  # seconds
    solomon_api.put_item('alerts', service_id, get_access_denied_alert(service_id, geo, alert_time_window))
    update_juggler_with_project(project, service_id, geo, alert_time_window)

    service_id = 'sd-requests-empty-responses-test'
    alert_time_window = 300  # seconds
    solomon_api.put_item('alerts', service_id, get_empty_responses_alert(service_id, geo, test_masters, 'Test', alert_time_window))
    update_juggler_with_project(project, service_id, geo, alert_time_window)

    service_id = 'sd-requests-empty-responses'
    alert_time_window = 300  # seconds

    solomon_api.put_item('alerts', service_id, get_empty_responses_alert(service_id, geo, prod_masters, period_seconds=alert_time_window))
    empty_responses_description = get_empty_responses_description(service_id)
    update_juggler_with_project(project, service_id, geo, alert_time_window, start_flow=False, title='Empty responses', meta=get_empty_responses_meta(), description=empty_responses_description)

    http_service_errors = {
        'sd-http-service-exceptions-occured': {
            'sensor': 'exceptions_occured',
            'threshold': 10000,
            'name': 'Number of exceptions occured',
            'start_flow': True,
            'phone_escalation': True,
        },
        'sd-http-service-queue-overflow': {
            'sensor': 'queue_overflow',
            'threshold': 1000,
            'name': 'Number of requests queue overflow events',
            'start_flow': True,
            'phone_escalation': True,
        },
        'sd-http-service-failed-queue-overflow': {
            'sensor': 'failed_queue_overflow',
            'threshold': 1000,
            'name': 'Number of failed requests queue overflow events',
            'start_flow': True,
            'phone_escalation': True,
        },
        'sd-http-service-max-connections-reached': {
            'sensor': 'max_connections_reached',
            'threshold': 5000,
            'name': 'Number of maximum connections reached events',
            'start_flow': False,
            'phone_escalation': False,
        }
    }
    alert_time_window = 120  # seconds
    for service_id, parameters in http_service_errors.items():
        solomon_api.put_item('alerts', service_id, get_http_service_errors_alert(service_id, geo, parameters['sensor'], parameters['threshold'], parameters['name'], alert_time_window))
        update_juggler_with_project(project, service_id, geo, alert_time_window, phone_escalation=parameters['phone_escalation'], start_flow=parameters['start_flow'], title=parameters['name'])
