import json
import juggler_sdk
import logging

from infra.yp_service_discovery.monitoring.solomon.src.util import util


def get_downtime_tags(cluster):
    return [
        "yp-downtime-9b8ee2a2b559b3afff751623e5e79546",
        "yp-{}-downtime-9b8ee2a2b559b3afff751623e5e79546".format(cluster)
    ]


def get_failures_alert(service_id, master, geo):
    result = {
        'id': service_id,
        'projectId': 'service_discovery',
        'name': 'Failures of requests to {} masters from ServiceDiscovery'.format(master),
        "notificationChannels": [
            "juggler",
        ],
        'type': {
            'threshold': {
                'selectors':
                    '{' + ', '.join([
                        'cluster="main"',
                        'service="production"',
                        'host="{}"'.format('|'.join(geo)),
                        'sensor="yp_service_discovery.yp_replica.endpoints.master.failures"',
                        'yp_cluster="{}"'.format(master),
                    ]) + '}',
                'timeAggregation': 'SUM',
                'predicate': 'GT',
                'threshold': 3,
            },
        },
        'periodMillis': 300000,
        'delaySeconds': 20,
        "groupByLabels": [
            "host",
        ],
        'annotations': {
            'service': util.get_juggler_service(service_id),
        },
    }
    return result


def get_backup_age_alert(service_id, master, geo):
    result = {
        'id': service_id,
        'projectId': 'service_discovery',
        'name': 'Indicator of backup age of {} master'.format(master),
        "notificationChannels": [
            "juggler",
        ],
        'type': {
            'threshold': {
                'selectors': '{' + ', '.join([
                    'cluster="main"',
                    'service="production"',
                    'host="{}"'.format('|'.join(geo)),
                    'sensor="yp_service_discovery.yp_replica.endpoints.backup.age_indicator"',
                    'yp_cluster="{}"'.format(master),
                ]) + '}',
                'timeAggregation': 'SUM',
                'predicate': 'GT',
                'threshold': 1,
            },
        },
        'periodMillis': 60000,
        'delaySeconds': 20,
        "groupByLabels": [
            "host",
        ],
        'annotations': {
            'service': util.get_juggler_service(service_id),
        },
    }
    return result


def update_failures_juggler(service_id, geo_list, master, juggler_api):
    for geo in geo_list:
        host = '{}.service_discovery'.format(geo)

        juggler_check = juggler_sdk.Check(
            host=host,
            service=util.get_juggler_service(service_id),
            namespace='yp.service_discovery',
            tags=get_downtime_tags(master),
            aggregator='logic_or',
            aggregator_kwargs={
                'unreach_checks': util.standard_unreach_checks(master),
            },
            flaps_config=juggler_sdk.FlapOptions(stable=30, critical=90, boost=0),
        )
        result = juggler_api.upsert_check(juggler_check).diff.to_dict()
        logging.info('Created juggler check: {} {}'.format(service_id, host))
        logging.debug('Diff:\n{}'.format(json.dumps(result, indent=4)))


def update_backup_age_juggler(service_id, geo_list, master, juggler_api, start_flow=False):
    for geo in geo_list:
        host = '{}.service_discovery'.format(geo)

        tags = []
        if start_flow:
            tags += [
                'warden_auto_source',
                'warden_alert_create_spi',
                'warden_alert_start_flow',
                'warden_alert_category_boolean',
                'warden_functionality_yp_yp-service-discovery_yp-to-sd-sync-lag',
            ]

        meta = {
            'title': 'YP {} to Service Discovery {} updates lag'.format(master.upper(), geo.upper()),
        }

        juggler_check = juggler_sdk.Check(
            host=host,
            service=util.get_juggler_service(service_id),
            namespace='yp.service_discovery',
            meta=meta,
            tags=get_downtime_tags(master) + tags,
            aggregator='logic_or',
            aggregator_kwargs={
                'unreach_checks': util.standard_unreach_checks(master),
            },
            flaps_config=juggler_sdk.FlapOptions(stable=30, critical=90, boost=0),
            notifications=[
                util.notification(
                    status={"from": "OK", "to": "CRIT"},
                    logins=util.get_duty_logins() + util.get_system_logins(),
                    notify_methods=[
                        'sms',
                        'telegram',
                    ],
                ),
                util.notification(
                    status={"from": "CRIT", "to": "OK"},
                    logins=util.get_duty_logins() + util.get_system_logins(),
                    notify_methods=[
                        'telegram',
                    ],
                ),
                juggler_sdk.NotificationOptions(
                    template_name='phone_escalation',
                    template_kwargs={
                        'delay': 120,
                        'logins': util.get_duty_logins() + [
                            'elshiko',
                            'avitella'
                        ],
                    }
                ),
            ]
        )
        result = juggler_api.upsert_check(juggler_check).diff.to_dict()
        logging.info('Created juggler check: {} {}'.format(service_id, host))
        logging.debug('Diff:\n{}'.format(json.dumps(result, indent=4)))


def apply(geo, masters, juggler_api, solomon_api):
    for master in masters:
        start_flow = master not in ['sas-test']

        service_id = 'yp-{}-replica-failures'.format(master)
        solomon_api.put_item('alerts', service_id, get_failures_alert(service_id, master, geo))
        update_failures_juggler(service_id, geo, master, juggler_api)

        service_id = 'yp-{}-replica-backup-age'.format(master)
        solomon_api.put_item('alerts', service_id, get_backup_age_alert(service_id, master, geo))
        update_backup_age_juggler(service_id, geo, master, juggler_api, start_flow=start_flow)
