import infra.yp_service_discovery.monitoring.solomon.src.dashboards.service_discovery as service_discovery
import infra.yp_service_discovery.monitoring.solomon.src.dashboards.yp_replica as yp_replica

import infra.yp_service_discovery.monitoring.solomon.src.alerts.service_discovery as service_discovery_alert
import infra.yp_service_discovery.monitoring.solomon.src.alerts.solomon as solomon_lag_alert
import infra.yp_service_discovery.monitoring.solomon.src.alerts.yp_replica as yp_replica_alert

from infra.yp_service_discovery.monitoring.solomon.src.util.solomon_api import SolomonApi as SolomonApiOld

from infra.libs.monitoring_lib.docs_builder.alert.builder import AlertDocsBuilder
from infra.libs.monitoring_lib.juggler.meta import Url
from infra.libs.monitoring_lib.project.project import AlertProject
from infra.libs.monitoring_lib.project.url_generators import UrlGenerator, JugglerUrlGenerator
from infra.libs.monitoring_lib.templates.discovery.discovery import AutoUrls
from infra.libs.monitoring_lib.util.api import Api, JugglerApi, SolomonApi

from argparse import ArgumentParser

import juggler_sdk
import logging


def parse_args(argv):
    parser = ArgumentParser()
    parser.add_argument('--solomon-token', required=True, help='solomon token. https://solomon.yandex-team.ru/api/internal/auth')
    parser.add_argument('--juggler-token', required=True, help='juggler token. https://juggler.yandex-team.ru/')
    parser.add_argument('--verbose', action="store_true", help='Enable verbose mode')
    return parser.parse_args(argv)


class NannyUrlGenerator(UrlGenerator):
    def __init__(self, geo):
        self.geo = geo

    def generate(self, project, juggler_check) -> AutoUrls:
        check_dc = juggler_check.host[:3]
        return AutoUrls([
            Url(
                title=f"{check_dc}_service_discovery",
                url=f"http://nanny.yandex-team.ru/ui/#/services/catalog/{check_dc}_yp_service_discovery",
                meta={"skip_docs": None}
            )
        ]) | AutoUrls([
            Url(
                title=f"{dc}_service_discovery",
                url=f"http://nanny.yandex-team.ru/ui/#/services/catalog/{dc}_yp_service_discovery",
                meta={"skip_juggler": None}
            )
            for dc in self.geo
        ])


def create_project(args, geo):
    api = Api(
        juggler_api_options=JugglerApi.Options(
            project_id="yp.service_discovery",
            token=args.juggler_token,
        ),
        solomon_api_options=SolomonApi.Options(
            project_id="service_discovery",
            token=args.solomon_token,
        )
    )
    docs_options=AlertDocsBuilder.Options(
        path_to_docs='infra/yp_service_discovery/docs/',
        docs_name='yp-service-discovery',
    )
    return AlertProject(
        name='yp_service_discovery',
        api=api,
        docs_options=docs_options,
        options=AlertProject.Options(
            url_generators_by_juggler_check=[JugglerUrlGenerator(), NannyUrlGenerator(geo)]
        )
    )


def upload(argv):
    args = parse_args(argv)
    logging.basicConfig(level=(logging.DEBUG if args.verbose else logging.INFO))

    geo = ['vla', 'sas', 'man', 'msk']
    project = create_project(args, geo)

    solomon_api = SolomonApiOld('service_discovery', args.solomon_token)
    juggler_api = juggler_sdk.JugglerApi('http://juggler-api.search.yandex.net', oauth_token=args.juggler_token)
    test_masters = ['sas-test']
    prod_masters = ['vla', 'sas', 'man', 'myt', 'man-pre', 'xdc', 'iva']
    masters = test_masters + prod_masters

    service_discovery.apply(geo, masters, solomon_api)
    yp_replica.apply(geo, masters, solomon_api)

    solomon_lag_alert.apply(juggler_api)
    yp_replica_alert.apply(geo, masters, juggler_api, solomon_api)
    service_discovery_alert.apply(geo, test_masters, prod_masters, juggler_api, solomon_api, project)

    project.apply()
