import grpc

from infra.yp_service_discovery.api import api_pb2_grpc


SD_GRPC_ADDRESS = 'sd.yandex.net:8081'


class Resolver(object):
    def __init__(self, client_name, grpc_address=SD_GRPC_ADDRESS, timeout=None, credentials=None, options=None):
        """A client for resolving YP endpoints.
        :see: README.md

        :attention: YP SD responds with empty endpoint set
            if client name is empty or not specified.
        :attention: Different instances of YP SD may respond with endpoints
            corresponding different points in time
            since all replicas of YP SD are independent.
            Take timestamp from response into account to grab only new data.

        :param client_name: A name to identify you on YP Service Discovery side,
            it is desirable to specify hostname.
        :param grpc_address: gRPC address of YP Service Discovery.
        :param timeout: An optional duration of time in seconds to allow for the RPC.
        :param credentials: An optional CallCredentials for the RPC.
        :param options: An optional list of key-value pairs (channel args in gRPC Core runtime)
            to configure the channel
        """
        self._client_name = client_name
        self._timeout = timeout

        if credentials is None:
            self._channel = grpc.insecure_channel(grpc_address, options)
        else:
            self._channel = grpc.secure_channel(grpc_address, credentials, options)
        self._stub = api_pb2_grpc.TServiceDiscoveryServiceStub(self._channel)

    def resolve_endpoints(self, request):
        """
        :param api_pb2.TReqResolveEndpoints request:
        :rtype: api_pb2.TReqResolveEndpoints
        :raises: RpcError: Indicating that the RPC terminated with non-OK status. The
            raised RpcError will also be a Call for the RPC affording the RPC's
            metadata, status code, and details.
        """

        if not request.client_name:
            request.client_name = self._client_name

        response = self._stub.ResolveEndpoints.future(request, timeout=self._timeout)
        return response.result()

    def resolve_pods(self, request):
        """
        :param api_pb2.TReqResolvePods request:
        :rtype: api_pb2.TRspResolvePods
        :raises: RpcError: Indicating that the RPC terminated with non-OK status. The
            raised RpcError will also be a Call for the RPC affording the RPC's
            metadata, status code, and details.
        """

        if not request.client_name:
            request.client_name = self._client_name

        response = self._stub.ResolvePods.future(request, timeout=self._timeout)
        return response.result()
