#include "resolver.h"

#include <infra/libs/service_iface/str_iface.h>
#include <infra/yp_service_discovery/libs/client/client.h>

namespace NServiceDiscovery {

const TString SERVICE_DISCOVERY_GRPC_ADDRESS = "sd.yandex.net:8081";

using namespace NYP::NServiceDiscovery;

class TResolver::TImpl {
public:
    TImpl(TString clientName, TString address, TDuration timeout)
        : ClientName_(std::move(clientName))
        , SDClient_(std::move(address), std::move(timeout))
    {
    }

    NApi::TRspResolveEndpoints Resolve(NApi::TReqResolveEndpoints request) {
        if (request.client_name().empty()) {
            request.set_client_name(ClientName_);
        }

        NApi::TRspResolveEndpoints response;
        NInfra::TAttributes responseAttributes;
        SDClient_.ResolveEndpoints(
            NInfra::RequestPtr<NInfra::TProtoRequest<NApi::TReqResolveEndpoints>>("", std::move(request), NInfra::TAttributes()),
            NInfra::ReplyPtr<NInfra::TProtoReply<NApi::TRspResolveEndpoints>>(response, responseAttributes)
        );
        return response;
    }

    TVector<NApi::TRspResolveEndpoints> Resolve(TVector<NApi::TReqResolveEndpoints> requests) {
        TVector<NApi::TRspResolveEndpoints> result;
        result.reserve(requests.size());

        for (NApi::TReqResolveEndpoints& request : requests) {
            result.push_back(Resolve(std::move(request)));
        }

        return result;
    }

    NApi::TRspResolveEndpoints Resolve(const TStringBuf cluster, const TStringBuf endpointSetId) {
        NApi::TReqResolveEndpoints request;
        request.set_cluster_name(TString{cluster});
        request.set_endpoint_set_id(TString{endpointSetId});
        request.set_client_name(ClientName_);

        return Resolve(std::move(request));
    }

    TVector<NApi::TRspResolveEndpoints> Resolve(const TStringBuf cluster, const TVector<TStringBuf>& endpointSetIds) {
        TVector<NApi::TRspResolveEndpoints> result;
        result.reserve(endpointSetIds.size());

        for (const TStringBuf endpointSetId : endpointSetIds) {
            result.push_back(Resolve(cluster, endpointSetId));
        }

        return result;
    }
    
    void Ping(NYP::NServiceDiscovery::NApi::TRspPing& result) {
        NInfra::TAttributes attributes;

        SDClient_.Ping(NInfra::RequestPtr<NInfra::TEmptyRequest<NYP::NServiceDiscovery::NApi::TReqPing>>("/ping", TString(), attributes),
                       NInfra::ReplyPtr<NInfra::TProtoReply<NYP::NServiceDiscovery::NApi::TRspPing>>(result, attributes ));
    }

private:
    const TString ClientName_;
    TClient SDClient_;
};

TResolver::TResolver(TString clientName, TString address, TDuration timeout)
    : Impl_(new TImpl(std::move(clientName), std::move(address), std::move(timeout)))
{
}

TResolver::~TResolver() = default;

NApi::TRspResolveEndpoints TResolver::Resolve(NApi::TReqResolveEndpoints request) const {
    return Impl_->Resolve(std::move(request));
}

TVector<NApi::TRspResolveEndpoints> TResolver::Resolve(TVector<NApi::TReqResolveEndpoints> requests) const {
    return Impl_->Resolve(std::move(requests));
}

NApi::TRspResolveEndpoints TResolver::Resolve(const TStringBuf cluster, const TStringBuf endpointSetId) const {
    return Impl_->Resolve(cluster, endpointSetId);
}

TVector<NApi::TRspResolveEndpoints> TResolver::Resolve(const TStringBuf cluster, const TVector<TStringBuf>& endpointSetIds) const {
    return Impl_->Resolve(cluster, endpointSetIds);
}

void TResolver::Ping(NYP::NServiceDiscovery::NApi::TRspPing& result) {
    Impl_->Ping(result);
}
} // namespace NServiceDiscovery
