#pragma once

#include <infra/yp_service_discovery/api/api.pb.h>

#include <util/datetime/base.h>
#include <util/generic/ptr.h>
#include <util/generic/string.h>
#include <util/generic/vector.h>

namespace NServiceDiscovery {

/// sd.yandex.net:8081 - YP Service Discovery gRPC address
extern const TString SERVICE_DISCOVERY_GRPC_ADDRESS;

/**
 * A client for resolving YP endpoints
 * @see README.md
 *
 * @attention YP SD responds with empty endpoint set if client name is empty or is not specified.
 * @attention Different instances of YP SD may respond with endpoints corresponding different points in time
 *            since all replicas in YP SD are independent.
 *            Take timestamp from response into account to grab only new data.
 */
class TResolver {
public:
    /**
     * @param clientName      name to identify you on YP Service Discovery side, it is desirable to specify hostname
     * @param address         gRPC address of YP Service Discovery
     * @param timeout         timeout value for each request
     */
    TResolver(TString clientName, TString address = SERVICE_DISCOVERY_GRPC_ADDRESS, const TDuration timeout = TDuration::Seconds(1));

    /**
     * @param request         NYP::NServiceDiscovery::NApi::TReqResolveEndpoints proto message
     * @return                NYP::NServiceDiscovery::NApi::TRspResolveEndpoints proto message
     *
     * @throws yexception     if query timed out
     */
    NYP::NServiceDiscovery::NApi::TRspResolveEndpoints Resolve(NYP::NServiceDiscovery::NApi::TReqResolveEndpoints request) const;

    /**
     * @param requests        list of NYP::NServiceDiscovery::NApi::TReqResolveEndpoints proto messages
     * @return                list NYP::NServiceDiscovery::NApi::TRspResolveEndpoints proto messages
     *
     * @throws yexception     if any query timed out
     */
    TVector<NYP::NServiceDiscovery::NApi::TRspResolveEndpoints> Resolve(TVector<NYP::NServiceDiscovery::NApi::TReqResolveEndpoints> requests) const;

    /**
     * @param cluster         YP cluster name
     * @param endpointSetId   id of endpoint set for which endpoints must be resolved
     * @return                NYP::NServiceDiscovery::NApi::TRspResolveEndpoints proto message
     *
     * @throws yexception     if query timed out
     */
    NYP::NServiceDiscovery::NApi::TRspResolveEndpoints Resolve(const TStringBuf cluster, const TStringBuf endpointSetId) const;

    /**
     * @param cluster         YP cluster name
     * @param endpointSetIds  list of endpoint set ids for which endpoints must be resolved
     * @return                list of NYP::NServiceDiscovery::NApi::TRspResolveEndpoints proto messages
     *
     * @throws yexception     if any query timed out
     */
    TVector<NYP::NServiceDiscovery::NApi::TRspResolveEndpoints> Resolve(const TStringBuf cluster, const TVector<TStringBuf>& endpointSetIds) const;


    /**
     * @param result          contains result of "/ping"
     */
    void Ping(NYP::NServiceDiscovery::NApi::TRspPing& result);

    ~TResolver();

private:
    class TImpl;
    THolder<TImpl> Impl_;
};

} // namespace NServiceDiscovery
