import argparse
import coloredlogs
import getpass
import logging
import os
import requests
import sys
import time

from yt.wrapper import YtClient
from yp.client import YpClient
from base64 import b64encode

import library.python.resource as resource
import yt.yson as yson


logging.getLogger("requests").setLevel(logging.ERROR)
logging.getLogger("urllib3.connectionpool").setLevel(logging.ERROR)
logging.getLogger("yt.packages.urllib3.connectionpool").setLevel(logging.ERROR)
logging.getLogger("yt.packages.requests.packages.urllib3.connectionpool").setLevel(logging.ERROR)
coloredlogs.install("DEBUG")

log = logging.getLogger("chaos_service")


def too_old(props):
    return time.time() - props["timestamp"] > 60 * 10


def too_old_to_live(props, pod_max_lifetime):
    return time.time() - props["timestamp"] > pod_max_lifetime


def add_common_attributes(attributes, owner=getpass.getuser()):
    if "labels" not in attributes:
        attributes["labels"] = {}
    attributes["labels"]["chaos_service"] = {
        "owner": owner,
        "timestamp": time.time(),
    }
    return attributes


def create_schedulable_pod_attributes(pod_set_id, pod_id, timestamp):
    spec = resource.find("/spec.yson")
    spec = spec.replace("${POD_ID}", pod_id)
    spec = spec.replace("${COMMAND}", "data:;base64,{}".format(b64encode("./server {}".format(timestamp))))

    attributes = {
        "meta": {
            "id": pod_id,
            "pod_set_id": pod_set_id,
        },
        "spec": yson.loads(spec)
    }
    return attributes


def parse_args(argv):
    parser = argparse.ArgumentParser()
    parser.add_argument("--cluster", required=True,
                        help="YP cluster name.")
    parser.add_argument("--pod-set-id", required=True,
                        help="Pod set id.")
    parser.add_argument("--yp-token",
                        help="YP token.")
    parser.add_argument("--yt-token",
                        help="YT token.")
    parser.add_argument("--yt-lock",
                        help="YT lock path.")
    parser.add_argument("--pod-count", type=int, default=2,
                        help="Pod count.")
    parser.add_argument("--pod-max-lifetime", type=int, default=20 * 60,
                        help="Time in seconds after which pod will be deleted.")
    parser.add_argument("--max-pods-on-node", type=int, default=1,
                        help="Max pods on one node.")
    parser.add_argument("--force-cleanup", action="store_true",
                        help="Cleanup env.")
    parser.add_argument("--account-id", default='tmp',
                        help="Account id.")
    args = parser.parse_args(argv)

    if args.yp_token is None:
        args.yp_token = os.getenv("YP_TOKEN")

    if args.yt_token is None:
        args.yt_token = os.getenv("YT_TOKEN")

    if args.yt_lock is None:
        args.yt_lock = "//home/infra/chaos_controller/{}_lock".format(args.cluster)

    return args


def is_active(iss_status):
    if "currentStates" not in iss_status:
        return False
    for current_state in iss_status["currentStates"]:
        if current_state["currentState"] == "ACTIVE":
            return True
    return False


def create_pod(yp_client, pod_set_id):
    timestamp = yp_client.generate_timestamp()
    attrs = create_schedulable_pod_attributes(pod_set_id, "chaos-{}".format(timestamp), timestamp)

    log.info("Creating pod: {}".format(attrs["meta"]["id"]))
    yp_client.create_object("pod", attributes=add_common_attributes(attrs))


def create_environment(yp_client, pod_set_id, force, account_id, max_pods_on_node):
    pod_set_exists = False
    endpoint_set_exists = False

    for pod in yp_client.select_objects("pod", selectors=["/meta/id"], filter="[/meta/pod_set_id] = \"{}\"".format(pod_set_id)):
        if force:
            log.warning("Removing pod: {}".format(pod[0]))
            yp_client.remove_object("pod", pod[0])

    for pod_set in yp_client.select_objects("pod_set", selectors=["/meta/id"], filter="[/meta/id] = \"{}\"".format(pod_set_id)):
        if force:
            log.warning("Removing pod_set: {}".format(pod_set[0]))
            yp_client.remove_object("pod_set", pod_set[0])
        else:
            pod_set_exists = True

    for endpoint_set in yp_client.select_objects("endpoint_set", selectors=["/meta/id"], filter="[/meta/id] = \"{}\"".format(pod_set_id)):
        if force:
            log.warning("Removing endpoint_set: {}".format(endpoint_set[0]))
            yp_client.remove_object("endpoint_set", endpoint_set[0])
        else:
            endpoint_set_exists = True

    if not pod_set_exists:
        log.info("Creating pod_set: {}".format(pod_set_id))
        pod_set_id = yp_client.create_object("pod_set", attributes=add_common_attributes({
            "meta": {
                "id": pod_set_id,
                "acl": [
                    {
                        "action": "allow",
                        "subjects": [
                            "robot-chaos-srv",
                            "avitella",
                            "dima-zakharov",
                            "elshiko",
                        ],
                        "permissions": [
                            "read",
                            "write",
                            "ssh_access",
                            "root_ssh_access",
                            "read_secrets",
                        ],
                    },
                ],
            },
            "spec": {
                "antiaffinity_constraints": [
                    {
                        "key": "node",
                        "max_pods": max_pods_on_node,
                    },
                ],
                "account_id": account_id,
            },
        }))
    else:
        log.info("pod_set {} already exists".format(pod_set_id))

    if not endpoint_set_exists:
        log.info("Creating endpoint_set: {}".format(pod_set_id))
        yp_client.create_object("endpoint_set", attributes=add_common_attributes({
            "meta": {"id": pod_set_id},
            "spec": {
                "pod_filter": "[/meta/pod_set_id] = \"{}\"".format(pod_set_id),
                "protocol": "TCP",
                "port": 3388,
            }
        }))
    else:
        log.info("endpoint_set {} already exists".format(pod_set_id))


def main(argv):
    args = parse_args(argv)

    yp_client = YpClient("{}.yp.yandex-team.ru:8090".format(args.cluster), config={"token": args.yp_token})
    yt_client = YtClient("locke", token=args.yt_token)

    with yt_client.Transaction(ping=True, interrupt_on_failed=True):
        yt_client.lock(args.yt_lock, waitable=True, wait_for=24 * 60 * 60 * 1000)

        create_environment(yp_client, args.pod_set_id, force=args.force_cleanup, account_id=args.account_id, max_pods_on_node=args.max_pods_on_node)

        while True:
            def iteration():
                pods = yp_client.select_objects(
                    "pod",
                    selectors=["/meta/id", "/labels/chaos_service", "/status/agent/iss"],
                    filter="[/meta/pod_set_id] = \"{}\"".format(args.pod_set_id))
                pods.sort(key=lambda pod: pod[1]["timestamp"])

                endpoints = yp_client.select_objects(
                    "endpoint",
                    selectors=["/meta/id", "/spec"],
                    filter="[/meta/endpoint_set_id] = \"{}\"".format(args.pod_set_id))

                active_endpoint_count = 0
                for meta_id, spec in endpoints:
                    try:
                        url = "http://[{}]:{}/version".format(spec["ip6_address"], spec["port"])
                        log.debug("Ping {}".format(url))
                        log.debug(requests.get(url, timeout=2))
                        active_endpoint_count += 1
                    except Exception as e:
                        log.debug(e)
                        pass

                pod_count = len(pods)
                active_pod_count = 0

                for pod_id, props, iss_status in pods:
                    if is_active(iss_status):
                        active_pod_count += 1
                        log.debug("Pod {} is alive (created {} seconds ago)".format(pod_id, time.time() - props["timestamp"]))
                    else:
                        log.debug("Pod {} is dead (created {} seconds ago)".format(pod_id, time.time() - props["timestamp"]))

                log.debug("Pod count: {}, Active pod count: {}, Endpoint count: {}, Active endpoint count: {}"
                          .format(pod_count, active_pod_count, len(endpoints), active_endpoint_count))

                if pod_count <= args.pod_count:
                    create_pod(yp_client, args.pod_set_id)

                elif active_pod_count <= args.pod_count:
                    if too_old(pods[0][1]):
                        log.error("Pods too old!")
                        debug_pods = yp_client.select_objects(
                            "pod",
                            selectors=["/meta/id", "/status"],
                            filter="[/meta/pod_set_id] = \"{}\"".format(args.pod_set_id))
                        for debug_pod in debug_pods:
                            log.error("pod_id: {}, pod_status: {}".format(debug_pod[0], yson.dumps(debug_pod[1], yson_format="pretty")))

                    for pod_id, props, iss_status in pods:
                        if too_old_to_live(props, args.pod_max_lifetime):
                            log.warning("Removing pod: {}".format(pod_id))
                            yp_client.remove_object("pod", pod_id)

                    log.info("Waiting 2 seconds")
                    time.sleep(2)

                elif active_endpoint_count != active_pod_count:
                    log.warning("Some alive pods is not really alive!")

                else:
                    log.info("Waiting 1 minute")
                    time.sleep(60)
                    log.info("Removing pod: {}".format(pods[0][0]))
                    yp_client.remove_object("pod", pods[0][0])

            try:
                iteration()
            except Exception as e:
                log.error(e)

                log.info("Waiting 10 seconds")
                time.sleep(10)


if __name__ == "__main__":
    main(sys.argv[1:])
