# -*- coding: utf-8 -*-

import yp_util.account_explain as account_explain
import yp_util.nodes_resources as nodes_resources
from yp_util.cluster_model import (
    ClusterModel,
    Size,
)
from yp_util.common import (
    build_yp_client,
    create_filter,
    create_node_filter,
    print_pandas_table,
    register_common_args,
    register_common_node_segment_args,
    register_common_nodes_args,
    register_nodes_status_args,
    register_yp_token_args,
)

from infra.yp import account_estimation

from yp.client import BatchingOptions

from yp.common import YpNoSuchObjectError

import yt.yson as yson
from yt.yson import YsonEntity

from color import colored
import pandas as pd

import argparse
import datetime
import logging
import os
import requests
import six
import sys
import time
import warnings


warnings.simplefilter(action='ignore', category=FutureWarning)

YP_DCs = ["vla", "man", "sas", "iva", "myt", "xdc"]


def register_nanny_args(parser):
    parser.add_argument('--nanny_token', required=False, help='Nanny OAuth token')


def register_common_selectors_args(parser):
    parser.add_argument('--selectors')


def register_endpointset_resolve_args(parser):
    parser.add_argument('endpointset', help='endpoint_set id {nanny-api-v1}')


def register_common_pod_args(parser):
    parser.add_argument('pod_id', help='pod_id {hamster-tunneller-yp-man-web}')


def register_account_mode_args(parser):
    parser.add_argument("mode", dest="account_mode")


def register_estimate_account_args(parser):
    parser.add_argument('--nanny_services')
    parser.add_argument('--gencfg_groups')


def add_pod_pod_set_filter(args, filters):
    if args.pod_set_id is not None:
        filters.append('([/meta/pod_set_id] = "{}")'.format(args.pod_set_id))


def add_pod_filter(args, filters):
    if args.pod_id is not None:
        filters.append('([/meta/id] = "{}")'.format(args.pod_id))


def add_endpoint_set_filter(args, filters):
    if args.endpointset is not None:
        filters.append('([/meta/endpoint_set_id] = "{}")'.format(args.endpointset))


def human_readable_timestamp(timestamp):
    tm = datetime.datetime.fromtimestamp(timestamp / 1000000.)
    return tm.strftime('%Y-%m-%d %H:%M:%S')


def nodes_status(argv):
    def parse_args(status_argv):
        parser = argparse.ArgumentParser(add_help=True)

        register_common_args(parser)
        register_common_nodes_args(parser)
        register_nodes_status_args(parser)

        return parser.parse_args(status_argv)

    args = parse_args(argv)

    client = build_yp_client(args)

    nodes = client.select_objects(
        "node",
        filter=create_node_filter(args, client),
        selectors=[
            "/meta/id",
            "/labels/segment",
            "/status/hfsm/state",
            "/status/last_seen_time",
            "/status/epoch_id",
            "/status/heartbeat_sequence_number",
        ],
        batching_options=BatchingOptions(),
    )

    rows = []
    for node in nodes:
        node_id, segment, status, last_seen_time, epoch_id, heartbeat = node
        rows.append([node_id, segment, status, human_readable_timestamp(last_seen_time), epoch_id, heartbeat])

    table = pd.DataFrame(rows, columns=['node_id', 'segment', 'status', 'last_seen_time', 'epoch_id', 'heartbeat'])

    print_pandas_table(table, args)


def bad_pods_status(argv):
    def parse_args(pods_argv):
        parser = argparse.ArgumentParser(add_help=True)

        register_common_args(parser)
        parser.add_argument('--pod-set-id', help='pod_set_id {hamster-tunneller-yp-man-web}', required=True)
        parser.add_argument('--min-down-time', type=int, help='min down time for pod\'s node in seconds {120}')

        return parser.parse_args(pods_argv)

    args = parse_args(argv)

    client = build_yp_client(args)

    filters = []
    add_pod_pod_set_filter(args, filters)

    pods_info = find_pods(
        client,
        selectors=[
            '/meta/id',
            '/meta/pod_set_id',
            '/meta/creation_time',
            '/status/scheduling'
        ],
        filter=create_filter(filters),
        limit=args.output_limit)

    current_time = time.time()

    rows = []

    for pod_id, pod_set_id, creation_time, scheduling in pods_info:
        message = None

        if scheduling['state'] == 'assigned':
            host = scheduling['node_id']

            host_info = client.select_objects(
                'node',
                selectors=[
                    '/meta/id',
                    '/status/hfsm/state', '/status/hfsm/last_updated'
                ],
                filter='[/meta/id] = "{}"'.format(host)
            )

            if host_info[0][1] == 'down':
                last_updated = host_info[0][2] / 1000000.

                down_time = current_time - last_updated
                if down_time > args.min_down_time:
                    message = '{} down for {}'.format(host, down_time)
        else:
            message = 'Pod is not assigned'

        if message is not None:
            rows.append([pod_id, pod_set_id, human_readable_timestamp(creation_time), message])

    table = pd.DataFrame(rows, columns=['pod_id', 'pod_set_id', 'creation_time', 'message'])

    print_pandas_table(table, args)


def account_estimate(argv):
    def parse_args(pods_argv):
        parser = argparse.ArgumentParser(add_help=True)

        register_yp_token_args(parser)
        register_nanny_args(parser)
        register_estimate_account_args(parser)

        return parser.parse_args(pods_argv)

    args = parse_args(argv)

    nanny_token = None
    if args.nanny_services is not None:
        nanny_token = get_token(args.nanny_token, "NANNY_TOKEN")
        if nanny_token is None:
            logging.error(
                colored(
                    'Please add --nanny_token=<Nanny oauth token> or pass it through $NANNY_TOKEN, please. '
                    'https://nanny.yandex-team.ru/ui/#/oauth/',
                    'red'))
            exit(-1)

    yp_clients = {}
    for dc in YP_DCs:
        yp_clients[dc] = build_yp_client(args, dc)

    nanny_session = requests.Session()
    nanny_session.headers['Authorization'] = 'OAuth {}'.format(nanny_token)

    account_per_dc = account_estimation.estimate_account(
        [] if args.nanny_services is None else args.nanny_services.split(","),
        [] if args.gencfg_groups is None else args.gencfg_groups.split(","),
        nanny_session,
        yp_clients,
    )

    rows = []
    for dc, account in six.iteritems(account_per_dc):
        if account.cpu > 0 or account.memory > 0 or account.hdd > 0 or account.ssd > 0 or account.ipv4 > 0:
            rows.append(
                [dc.upper(),
                 account.cpu / 1000.0,
                 "{}Gb".format(account.memory),
                 "{}Tb".format(account.hdd),
                 "{}Tb".format(account.ssd),
                 account.ipv4
                 ])

    table = pd.DataFrame(rows, columns=["DC", 'Cpu', 'Memory', 'Hdd', 'Ssd', 'ipv4'])
    print_pandas_table(table, args=None)


def query_pods(client, pod_id, selectors, filter_):
    limit = 200

    if pod_id[0] is None:
        filter = filter_
    else:
        filter = "([/meta/pod_set_id],[/meta/id])>('{}','{}')".format(pod_id[0], pod_id[1])
        if filter_ is not None:
            filter += " AND " + filter_

    return client.select_objects("pod", selectors=selectors, limit=limit, filter=filter)


def find_pods(client, selectors, filter, limit):
    selectors_ = ["/meta/id", "/meta/pod_set_id"]
    selectors_.extend(selectors)
    pods = query_pods(client, (None, None), selectors_, filter)
    while len(pods) > 0:
        for pod in pods:
            limit = limit - 1
            if limit < 0:
                return

            yield tuple(pod[2:])

        pods = query_pods(client, (pods[-1][1], pods[-1][0]), selectors_, filter)


def pods_list(argv):
    def parse_args(pods_argv):
        parser = argparse.ArgumentParser(add_help=True)

        register_common_args(parser)
        parser.add_argument("--pod-set-id", "--pod-set", help="pod_set_id {hamster-tunneller-yp-man-web}")
        parser.add_argument("--node-segment-id", "--segment", help="node_segment_id {default}")

        return parser.parse_args(pods_argv)

    args = parse_args(argv)

    client = build_yp_client(args)

    filters = []
    add_pod_pod_set_filter(args, filters)

    pods_info = find_pods(client,
                          selectors=[
                              '/meta/id',
                              '/meta/pod_set_id',
                              '/status/scheduling/node_id',
                              '/spec/resource_requests'
                          ],
                          filter=create_filter(filters),
                          limit=args.output_limit)

    pod_set_to_segment = dict(client.select_objects("pod_set", selectors=["/meta/id", "/spec/node_segment_id"]))

    rows = []

    for pod_id, pod_set_id, node_id, resources in pods_info:
        node_segment = pod_set_to_segment.get(pod_set_id, "")
        if args.node_segment_id and node_segment != args.node_segment_id:
            continue

        if isinstance(resources, YsonEntity):
            rows.append(
                [pod_id, pod_set_id, node_id, node_segment, 0, 0])
        else:
            rows.append(
                [pod_id, pod_set_id, node_id, node_segment,
                 resources.get('vcpu_guarantee', 0),
                 resources.get('memory_guarantee', 0)])

    table = pd.DataFrame(rows, columns=[
        'pod_id',
        'pod_set_id',
        'node_id',
        'node_segment_id',
        'vcpu_guarantee',
        'memory_guarantee',
    ])

    print_pandas_table(table, args)


def get_service_url(cluster, pod_id, labels):
    if labels.get("deploy_engine", "") == "QYP":
        return "https://qyp.nanny.yandex-team.ru/vm/{}/{}/".format(cluster, pod_id)
    elif labels.get("deploy_engine", "") == "YP_LITE":
        return "https://nanny.yandex-team.ru/ui/#/services/catalog/{}/".format(labels["nanny_service_id"])
    else:
        return pod_id


def pod_neighbours(argv):
    def parse_args(argv):
        parser = argparse.ArgumentParser(add_help=True)

        register_common_args(parser)
        register_common_pod_args(parser)

        return parser.parse_args(argv)

    args = parse_args(argv)

    client = build_yp_client(args)

    filters = []
    add_pod_filter(args, filters)

    pod_info = None
    pod_id = args.pod_id

    try:
        pod_info = client.get_object(
            'pod',
            args.pod_id,
            selectors=["/status"]
        )
    except YpNoSuchObjectError:
        print("Pod '{}' is not found in {}".format(pod_id, args.cluster))
        return

    status = pod_info[0]
    if status.get("scheduling", {}).get("node_id", None) is None:
        print("Pod is not scheduled anywhere, so no neighbours found")
        return

    node_id = status["scheduling"]["node_id"]

    node_labels = client.get_object("node", node_id, selectors=["/labels"])[0]
    migration_source = node_labels.get("extras", {}).get("migration", {}).get("source", "")
    is_coexistence = migration_source == "gencfg"

    pods = client.select_objects("pod",
                                 filter='[/status/scheduling/node_id]="{}"'.format(node_id),
                                 selectors=["/meta", "/labels"])

    rows = []
    for pod_info in pods:
        labels = pod_info[1]

        if not isinstance(pod_info, YsonEntity) and pod_info is not None:
            pod_id = pod_info[0]["id"]
            rows.append(
                [pod_id, get_service_url(args.cluster, pod_id, labels)])

    table = pd.DataFrame(rows,
                         columns=['Neighbours on node {}'.format(node_id),
                                  "On host with not isolated Gencfg containers={}".format(is_coexistence)])

    print_pandas_table(table, args)


def pod_raw(argv):
    def parse_args(argv):
        parser = argparse.ArgumentParser(add_help=True)

        register_common_args(parser)
        register_common_pod_args(parser)
        register_common_selectors_args(parser)

        return parser.parse_args(argv)

    args = parse_args(argv)

    client = build_yp_client(args)

    filters = []
    add_pod_filter(args, filters)

    selectors = ["/meta"]
    if args.selectors is not None:
        selectors = args.selectors.split(",")

    pod_info = client.get_object(
        'pod',
        args.pod_id,
        selectors=selectors
    )
    p = {}
    for selector_idx, selector in enumerate(selectors):
        p[selector] = pod_info[selector_idx]

    print(yson._dumps_to_native_str(p, yson_format="pretty"))


def pod_get(argv):
    def parse_args(argv):
        parser = argparse.ArgumentParser(add_help=True)

        register_common_args(parser)
        register_common_pod_args(parser)

        return parser.parse_args(argv)

    args = parse_args(argv)

    client = build_yp_client(args)

    filters = []
    add_pod_filter(args, filters)

    selectors = ["/meta", "/spec", "/status"]

    pod_id = args.pod_id
    pod_info = None

    try:
        pod_info = client.get_object(
            'pod',
            pod_id,
            selectors=selectors
        )
    except YpNoSuchObjectError:
        print("Pod '{}' is not found in {}".format(pod_id, args.cluster))
        return

    meta = pod_info[0]
    spec = pod_info[1]
    status = pod_info[2]

    node_id = status["scheduling"]["node_id"]
    pod_set_id = meta["pod_set_id"]

    vcpu_guarantee = spec["resource_requests"]["vcpu_guarantee"]
    vcpu_limit = spec["resource_requests"]["vcpu_limit"]

    memory = spec["resource_requests"]["memory_guarantee"]

    disk_requests = {}
    for disk_request in spec["disk_volume_requests"]:
        storage_class = disk_request["storage_class"]
        size = disk_request["quota_policy"]["capacity"]

        if storage_class not in disk_requests:
            disk_requests[storage_class] = 0

        disk_requests[storage_class] += size

    row = [pod_id, pod_set_id, node_id, vcpu_guarantee, vcpu_limit, memory]
    columns = ["pod_id", "pod_set_id", "node", "vcpu_guarantee", "vcpu_limit", "memory"]

    if "hdd" in disk_requests:
        row.append(disk_requests["hdd"])
        columns.append("hdd")

    if "ssd" in disk_requests:
        row.append(disk_requests["ssd"])
        columns.append("ssd")

    table = pd.DataFrame([row], columns=columns)

    print_pandas_table(table, args)


def endpointset_resolve(argv):
    def parse_args(eps_argv):
        parser = argparse.ArgumentParser(add_help=True)

        register_common_args(parser)
        register_endpointset_resolve_args(parser)

        return parser.parse_args(eps_argv)

    args = parse_args(argv)

    client = build_yp_client(args)

    filters = []
    add_endpoint_set_filter(args, filters)

    endpoints_info = client.select_objects(
        'endpoint',
        selectors=[
            '/spec/fqdn',
            '/spec/ip6_address',
            '/spec/port',
            '/spec/protocol',
        ],
        filter=create_filter(filters)
    )

    rows = []

    for (fqdn, ip6_address, port, protocol) in endpoints_info:
        rows.append([fqdn, ip6_address, port, protocol])

    table = pd.DataFrame(rows, columns=['fqdn', 'ip6_address', 'port', 'protocol'])

    print_pandas_table(table, args)


def can_find_place(argv):
    def parse_args(argv):
        parser = argparse.ArgumentParser(add_help=True)

        register_common_args(parser)
        register_common_node_segment_args(parser)
        parser.add_argument("--gencfg_groups", required=True)

        return parser.parse_args(argv)

    args = parse_args(argv)

    print("Loading nodes list from YP. It takes a lot of time...")
    client = build_yp_client(args)
    cluster_model = ClusterModel(client, None, None)
    group_fit_probability = dict()

    for group in args.gencfg_groups.split(","):
        print("\nLoading group info for {} from Gencfg...".format(group))
        max_instance_size, instance_count = account_estimation.get_resources_by_instances(args.cluster, group, "trunk")
        if instance_count == 0:
            print("No instances found for group {} in {}".format(group, args.cluster))
            continue

        suitable_nodes = list()

        pod_size = Size(max_instance_size.cpu, max_instance_size.memory, max_instance_size.hdd, max_instance_size.ssd)
        for node in cluster_model.node_cache.get_all(sorted_by=None):
            if node.has_free_space_for(pod_size):
                suitable_nodes.append(node)

        print("Looking for {} pods with {}".format(instance_count, pod_size))
        fit_probability = min(int((len(suitable_nodes) * 100) / (2 * instance_count)), 99)

        print("Found {} suitable nodes for group {}, fit probability {}%".format(
            len(suitable_nodes), group, fit_probability
        ))

        group_fit_probability[group] = (fit_probability, len(suitable_nodes))

    print("\n-----------------------------Total---------------------------")
    for group in group_fit_probability:
        fit_probability, suitable_nodes = group_fit_probability[group]
        print("Found {} suitable nodes for group {}, fit probability {}%".format(
            suitable_nodes, group, fit_probability
        ))


def get_token(token, token_name):
    if token is not None:
        return token

    if token_name in os.environ:
        return os.environ[token_name]
    else:
        return None


def account_mode(argv):
    mode = argv[0] if len(argv) > 0 else None
    if mode == "explain":
        return account_explain.main(argv[1:])
    elif mode == "estimate":
        return account_estimate(argv[1:])
    else:
        print_help_message()


def pod_mode(argv):
    mode = argv[0] if len(argv) > 0 else None
    if mode == "raw":
        return pod_raw(argv[1:])
    if mode == "get":
        return pod_get(argv[1:])
    elif mode == "neighbours":
        return pod_neighbours(argv[1:])
    else:
        print_help_message()


MODES = {
    'nodes-status': nodes_status,
    'nodes-resources': nodes_resources.main,
    'bad-pods-status': bad_pods_status,
    'pods-list': pods_list,
    'account': account_mode,
    'endpointset-resolve': endpointset_resolve,
    'pod': pod_mode,
    'has-place-for': can_find_place,
}


def print_help_message():
    clusters = 'sas,man,vla,iva,myt,xdc'
    print(colored('Full description https://wiki.yandex-team.ru/yp/yp-cli/', attrs=['bold']))
    print(colored('Avaliable modes:', attrs=['bold']))
    print(colored('    %s' % (", ".join(sorted(MODES.keys()))), 'white'))
    print(colored('    use ./yp-util mode --help for additional information', 'white'))
    print(colored('    use --query {pandas query} for table output filtering {cpu in millicores, memory and disk in bytes}', 'white'))
    print(colored('    use --cluster {%s} to specify YP cluster' % clusters, 'white'))
    print(colored('    YP_TOKEN could be passed through command line --token <YP_TOKEN> or throught environment variable $YP_TOKEN', 'white'))
    print(colored('    You can find YP_TOKEN value at https://oauth.yandex-team.ru/authorize?response_type=token&client_id=f8446f826a6f4fd581bf0636849fdcd7', 'white'))
    print(colored('Examples:', attrs=['bold']))
    print(colored('    ./yp-util nodes-status --segment dev', 'green'))
    print(colored('        Print status, last_seen_time, epoch_id, heartbeat for all nodes in dev segment', 'white'))
    print(colored('    ./yp-util nodes-resources --segment default --output-limit 100 --sort-by total_memory --node-status {up,down} '
                  '--show-free {Show free resources instead of occupied}', 'green'))
    print(colored('        Print top 100 nodes in default segment sorted by total memory', 'white'))
    print(colored('    ./yp-util bad-pods-status --output-limit 100 --min-down-time 120', 'green'))
    print(colored('        Print 100 pods which are not assigned to any node or it\'s node is down for more than 120 seconds', 'white'))
    print(colored('    ./yp-util pods-list --output-limit 100 --sort-by memory_guarantee,vcpu_guarantee --sort-order asc', 'green'))
    print(colored('        Print top 100 pods sorted by tuple (memory_guarantee,vcpu_guarantee) in ascending order', 'white'))
    print(colored('    ./yp-util endpointset-resolve <endpointset_id> --output-limit 100', 'green'))
    print(colored('        Print top 100 pods sorted by tuple (memory_guarantee,vcpu_guarantee) in ascending order', 'white'))
    print(colored('    ./yp-util account explain <account_id> --output-limit 100 --sort-by total_memory', 'green'))
    print(colored('        Print account information sorted by tuple (memory_guarantee,vcpu_guarantee) in ascending order', 'white'))
    print(colored('    ./yp-util account estimate --gencfg_groups <comma separated Gencfg groups> --nanny_services <comma separated services list> --nanny_token <NANNY_TOKEN> ',
                  'green'))
    print(colored('        Print account information sorted by tuple (memory_guarantee,vcpu_guarantee) in ascending order', 'white'))
    print(colored('    ./yp-util pod get <pod_id> ', 'green'))
    print(colored('        Pretty print pod information', 'white'))
    print(colored('    ./yp-util pod raw <pod_id> --selectors {/meta,/spec,/status or more specific, comma separated}', 'green'))
    print(colored('        Print raw pod information', 'white'))
    print(colored('    ./yp-util pod neighbours <pod_id>', 'green'))
    print(colored('        Print pod neighbours information on the same node', 'white'))
    print(colored('    ./yp-util has-place-for --gencfg_groups <comma separated Gencfg groups>', 'green'))
    print(colored('        Checks place in YP for specified Gencfg groups', 'white'))


def configure_logger():
    FORMAT = "%(asctime)s\t%(levelname)s\t%(message)s"
    handler = logging.StreamHandler()
    handler.setFormatter(logging.Formatter(FORMAT))
    root_logger = logging.getLogger()
    root_logger.setLevel(logging.INFO)
    root_logger.handlers = [handler]


if __name__ == '__main__':
    configure_logger()
    if len(sys.argv) <= 1 or sys.argv[1] == '-h' or sys.argv[1] == '--help':
        print_help_message()
    elif sys.argv[1] in MODES:
        MODES[sys.argv[1]](sys.argv[2:])
    else:
        print('no such mode {}'.format(sys.argv[1]))
