# -*- coding: utf-8 -*-

from infra.yp.lib.retrier import retry_me

import requests


class AbcInaccessible(Exception):
    pass


class AbcClient(object):
    FQDN = "abc-back.yandex-team.ru"

    API_V4_SERVICES = "/api/v4/services/"

    def __init__(self, token):
        self._token = token
        # Validate the connection.
        self._get_json_with_retries("")

    def _get_json_with_retries(self, cgi_params):
        def impl():
            url = "https://{fqdn}{api}?{cgi_params}".format(
                fqdn=self.FQDN,
                api=self.API_V4_SERVICES,
                cgi_params=cgi_params,
            )
            headers = {
                "Authorization": "OAuth {}".format(self._token)
            }
            response = requests.get(url, headers=headers, timeout=(0.1, 5.0))
            response.raise_for_status()
            return response.json()

        try:
            return retry_me(impl, retry_count=5, sleep_time=0.05, silent=True)
        except (requests.ConnectionError, requests.HTTPError):  # ConnectTimeout included.
            raise AbcInaccessible

    def list_service_children(self, service_id):
        response = self._get_json_with_retries(
            "parent__with_descendants={service_id}&fields=id,slug".format(service_id=service_id),
        )
        results = response["results"]
        return {result["id"]: result for result in results}

    def get_service_slug_by_id(self, service_id):
        """
        Obtain service slug (string) via service id (number)
        :return service slug or None if no such service
        """
        response = self._get_json_with_retries(
            "id={service_id}&fields=id,slug".format(service_id=service_id),
        )
        if "results" not in response or not response["results"]:
            return None

        service_info = response["results"][0]
        assert service_info["id"] == service_id, "Logic error, service match is ambiguous"
        return service_info["slug"]

    def get_service_id_by_slug(self, service_slug):
        """
        Obtain service id (number) via service slug (string)
        :return service id or None if no such service
        """
        response = self._get_json_with_retries(
            "slug={service_slug}&fields=id,slug".format(service_slug=service_slug),
        )
        if "results" not in response or not response["results"]:
            return None

        service_info = response["results"][0]
        assert service_info["slug"] == service_slug, "Logic error, service match is ambiguous"
        return service_info["id"]
