# -*- coding: utf-8 -*-

import six

from yt.wrapper.yson import YsonEntity

import pandas as pd

from datetime import datetime, timedelta
from requests.packages.urllib3.util.retry import Retry
import dateutil.parser
import logging
import requests
import time


NANNY_URL = 'http://nanny.yandex-team.ru'
LOGGER = None


def create_session(oauth=None):
    session = requests.Session()
    session.verify = False

    retry = Retry(
        backoff_factor=0.3,
        total=10
    )
    session.mount(
        'http://',
        requests.adapters.HTTPAdapter(max_retries=retry),
    )
    session.mount(
        'https://',
        requests.adapters.HTTPAdapter(max_retries=retry),
    )
    if oauth is not None:
        session.headers['Authorization'] = 'OAuth {}'.format(oauth)

    return session


def timing(f):
    def wrap(*args):
        time1 = time.time()
        ret = f(*args)
        time2 = time.time()
        print('%s function took %0.3f ms' % (f.func_name, (time2 - time1) * 1000.0))
        return ret
    return wrap


class PodSet:
    def __init__(self, id, client, nanny_services_cache, node_cache, pod_cache):
        self.id = id
        self.client = client
        self._labels = None
        self.meta = None
        self.nanny_services_cache = nanny_services_cache
        self.node_cache = node_cache
        self.pod_cache = pod_cache
        self.pod_ids = list()
        self._spec = None
        self.cached_suitable_nodes = None
        self.cached_podset_nodes = None
        for pod in client.select_objects(
            "pod",
            filter="[/meta/pod_set_id]=\"{}\"".format(self.id), selectors=["/meta/id"]
        ):
            self.pod_ids.append(pod[0])

    @staticmethod
    def create(client, pod_set_id, nanny_services_cache, node_cache, pod_cache):
        return PodSet(pod_set_id, client, nanny_services_cache, node_cache, pod_cache)

    def size(self):
        return len(self.pod_ids)

    def labels(self):
        if self._labels is None:
            self.load()
        return self._labels

    def load(self):
        if self._labels is None:
            self._labels, self._spec = self.client.get_object("pod_set", self.id, selectors=["/labels", "/spec"])

    def url(self):
        if "yp.deploy_engine_url" in self.labels():
            return self.labels()["yp.deploy_engine_url"]
        else:
            return "None"

    def get_deploy_engine(self):
        _labels = self.labels()
        if "deploy_engine" in _labels:
            return _labels["deploy_engine"]
        else:
            return None

    def get_nanny_service(self):
        deploy_engine = self.get_deploy_engine()
        if deploy_engine == "YP_LITE":
            return self.nanny_services_cache.get_service(self.labels()["nanny_service_id"])

        return None

    def can_evict(self):
        nanny_service = self.get_nanny_service()
        if nanny_service is None:
            return False
        else:
            if not nanny_service.can_be_evicted():
                return False

            return all([self.pod_cache.get(pod_id).not_in_replication_state() for pod_id in self.pod_ids])

    def get_nodes(self):
        if self.cached_podset_nodes is not None:
            return self.cached_podset_nodes

        self.cached_podset_nodes = self.node_cache.get_pod_set_nodes(self.id)
        return self.cached_podset_nodes

    def get_suitable_nodes(self):
        if self.cached_suitable_nodes is not None:
            return self.cached_suitable_nodes

        node_filter = self.get_node_filter()

        if node_filter is None or len(node_filter) == 0:
            self.cached_suitable_nodes = self.node_cache.nodes()
        else:
            self.cached_suitable_nodes = self.node_cache.get_filtered_nodes(node_filter)

        return self.cached_suitable_nodes

    def spec(self):
        if self._spec is None:
            self.load()
        return self._spec

    def get_node_segment(self):
        return self.spec()["node_segment_id"]

    def get_node_filter(self):
        spec = self.spec()
        if "node_filter" in spec:
            node_filter = spec["node_filter"]
            if isinstance(node_filter, YsonEntity):
                return None
            if len(node_filter) > 0:
                return node_filter

        return None


class PodSetCache:
    def __init__(self):
        self.client = None
        self.cache = None
        self.nanny_services_cache = None
        self.node_cache = None
        self.pod_cache = None

    def init(self, client, nanny_services_cache, node_cache):
        self.client = client
        self.cache = dict()
        self.nanny_services_cache = nanny_services_cache
        self.node_cache = node_cache

    def get(self, pod_set_id):
        if pod_set_id not in self.cache:
            self.cache[pod_set_id] = PodSet.create(
                self.client, pod_set_id, self.nanny_services_cache, self.node_cache, self.pod_cache
            )

        return self.cache[pod_set_id]


class UnsatisfiedResource:
    def __init__(self, name, value):
        self.type = name
        self.value = int(value)


class UnsatisfiedResources:
    def __init__(self):
        self.unsatisfied_resources = dict()

    def add_resource(self, resource):
        self.unsatisfied_resources[resource.type] = resource

    def dominating_resource(self, pod_to_find_place):
        max_resource = self.unsatisfied_resources.values()[0]

        for resource_type, resource_value in six.iteritems(self.unsatisfied_resources):
            if resource_value.value > max_resource.value:
                max_resource = resource_value

        if max_resource.type == "disk":
            if pod_to_find_place.get_size().ssd > 0:
                max_resource.type = "ssd"
            else:
                max_resource.type = "hdd"

        return max_resource


class Node:
    def __init__(self, client, node_id, used_size, total_size, pod_cache, hfsm_state):
        self.hfsm_state = hfsm_state
        self.pod_cache = pod_cache
        self.client = client
        self.pods = None
        self.id = node_id
        self.used_size = used_size
        self.total_size = total_size
        self.free_size = self.total_size - self.used_size

    def __repr__(self):
        return self.id

    def get_pods(self):
        if self.pods is None:
            self.pods = list()
            for pod in self.client.select_objects(
                "pod",
                filter="[/spec/node_id]=\"{}\"".format(self.id), selectors=["/meta/id"]
            ):
                pod_id = pod[0]
                self.pods.append(self.pod_cache.get(pod_id))

        return self.pods

    def has_free_space_for(self, pod_or_size):
        if isinstance(pod_or_size, Pod):
            return self.free_size.bigger_than(pod_or_size.get_size())
        else:
            return self.free_size.bigger_than(pod_or_size)

    def has_free_space_for_size(self, size):
        return self.free_size.bigger_than(size)

    def has_total_space_for(self, pod):
        return self.total_size.bigger_than(pod.get_size())

    def is_up(self):
        return self.hfsm_state == "up"

    @staticmethod
    def from_yp(client, node_id, host_resources, pod_cache, hfsm_state):

        def default_resources_dict():
            return {"cpu": 0, "memory": 0, "hdd": 0, "ssd": 0}

        total_volumes = default_resources_dict()
        total_pods_gurantee_actual = default_resources_dict()
        total_pods_gurantee_scheduled = default_resources_dict()

        for resource in host_resources:
            spec = resource[0]
            actual_allocations = resource[1]
            kind = resource[2]
            scheduled_allocations = resource[3]

            if kind not in ["cpu", "memory", "disk"]:
                continue

            if kind == 'disk':
                total_volumes[spec[kind]['storage_class']] += spec[kind]['total_capacity']
            else:
                total_volumes[kind] += spec[kind]['total_capacity']

            if actual_allocations:
                for allocation in actual_allocations:
                    if kind == 'disk':
                        total_pods_gurantee_actual[spec[kind]['storage_class']] += allocation[kind]['capacity']
                    else:
                        total_pods_gurantee_actual[kind] += allocation[kind]['capacity']

            if scheduled_allocations:
                for allocation in scheduled_allocations:
                    if kind == 'disk':
                        total_pods_gurantee_scheduled[spec[kind]['storage_class']] += allocation[kind]['capacity']
                    else:
                        total_pods_gurantee_scheduled[kind] += allocation[kind]['capacity']

        used_cpu = max(total_pods_gurantee_actual['cpu'], total_pods_gurantee_scheduled['cpu'])
        used_memory = max(total_pods_gurantee_actual['memory'], total_pods_gurantee_scheduled['memory'])
        used_hdd = max(total_pods_gurantee_actual['hdd'], total_pods_gurantee_scheduled['hdd'])
        used_ssd = max(total_pods_gurantee_actual['ssd'], total_pods_gurantee_scheduled['ssd'])

        return Node(client,
                    node_id,
                    used_size=Size(used_cpu, used_memory, used_hdd, used_ssd),
                    total_size=Size(
                        total_volumes['cpu'],
                        total_volumes['memory'],
                        total_volumes['hdd'],
                        total_volumes['ssd'],
                    ),
                    pod_cache=pod_cache,
                    hfsm_state=hfsm_state)


class NodeCache:
    def __init__(self):
        self.cache = None
        self.client = None
        self.pod_cache = None
        self.segments_in = None
        self.pod_node_cache = dict()

    def init(self, client, pod_cache, segments_in=['default']):
        self.cache = dict()
        self.client = client
        self.pod_cache = pod_cache
        self.segments_in = segments_in
        self._load(client, segments_in=segments_in)

    def get(self, node_id):
        return self.cache[node_id]

    def nodes(self):
        return self.cache

    @staticmethod
    def _query_nodes(client, node_id, filter):
        limit = 400

        if node_id is not None:
            filter = "[/meta/id]>'{}' {}".format(node_id, "" if filter is None else " AND " + filter)

        return client.select_objects("node", selectors=["/meta/id", "/status/hfsm/state"], filter=filter, limit=limit)

    @staticmethod
    def _query_node_resources(client, resource_id):
        limit = 1000

        if resource_id[0] is None:
            filter = None
        else:
            filter = "([/meta/node_id],[/meta/id])>('{}','{}')".format(resource_id[0], resource_id[1])

        return client.select_objects("resource",
                                     selectors=[
                                         '/meta/node_id',
                                         '/meta/id',
                                         '/spec',
                                         '/status/actual_allocations',
                                         '/meta/kind',
                                         '/status/scheduled_allocations'],
                                     limit=limit, filter=filter)

    def get_node_segment_filter(self, segment_name):
        return self.client.get_object("node_segment", segment_name, selectors=["/spec/node_filter"])[0]

    def _load(self, client, segments_in=['default']):
        LOGGER.debug("Loading nodes and resources")
        _node_cache = dict()
        _node_info = dict()

        segment_filters = []
        for segment in segments_in:
            segment_filters.append(self.get_node_segment_filter(segment))

        if len(segment_filters) > 0:
            segment_filter = " AND ".join(segment_filters)
        else:
            segment_filter = None

        nodes = NodeCache._query_nodes(client, None, segment_filter)
        while len(nodes) > 0:
            for node in nodes:
                node_id = node[0]
                _node_cache[node_id] = None
                _node_info[node_id] = (node[1])  # status/hfsm

            nodes = NodeCache._query_nodes(client, nodes[-1][0], segment_filter)

        node_resources_cache = dict()
        node_resources = NodeCache._query_node_resources(client, (None, None))
        while len(node_resources) > 0:
            for node_resource in node_resources:
                node_id = node_resource[0]
                spec = node_resource[2]
                actual_allocations = node_resource[3]
                kind = node_resource[4]
                scheduled_allocations = node_resource[5]

                if node_id in _node_cache:
                    if node_id not in node_resources_cache:
                        node_resources_cache[node_id] = list()

                    node_resources_cache[node_id].append((spec, actual_allocations, kind, scheduled_allocations))

            node_resources = NodeCache._query_node_resources(client, (node_resources[-1][0], node_resources[-1][1]))

        for node_id in node_resources_cache:
            self.cache[node_id] = Node.from_yp(
                client, node_id, node_resources_cache[node_id], self.pod_cache, _node_info[node_id]
            )

    def get_filtered_nodes(self, pod_node_filter):
        if pod_node_filter in self.pod_node_cache:
            return self.pod_node_cache[pod_node_filter]

        filters = [pod_node_filter]
        if self.segments_in is not None:
            for segment in self.segments_in:
                filters.append(self.get_node_segment_filter(segment))

        filter = " AND ".join(filters)

        nodes = dict()
        all_nodes = self.client.select_objects("node", filter=filter, selectors=["/meta/id"])
        for node_id in all_nodes:
            node_id = node_id[0]
            nodes[node_id] = self.cache[node_id]

        self.pod_node_cache[pod_node_filter] = nodes
        return self.pod_node_cache[pod_node_filter]

    def get_pod_set_nodes(self, pod_set_id):
        occupied_nodes = dict()

        all_nodes = self.client.select_objects("pod", filter="[/meta/pod_set_id]=\"{}\"".format(pod_set_id), selectors=["/spec/node_id"])
        for node in all_nodes:
            node = node[0]
            if len(node.strip()) > 0:
                occupied_nodes[node] = self.cache[node]

        return occupied_nodes

    def find_nodes_suitable_for_pod_right_now(self, pod, sort_by_resource=None):
        return self._find_suitable_nodes(pod, sort_by_resource, immediate=True)

    def find_nodes_suitable_for_pod_potentially(self, pod, sort_by_resource=None):
        return self._find_suitable_nodes(pod, sort_by_resource, immediate=False)

    def _find_suitable_nodes(self, pod, sort_by_resource=None, immediate=False):
        rows = []

        suitable_nodes = dict()
        already_occupied_nodes = pod.get_pod_set().get_nodes()
        podset_suitable_nodes = pod.get_pod_set().get_suitable_nodes()

        for node_id, node in six.iteritems(self.cache):
            if node_id not in already_occupied_nodes and node_id in podset_suitable_nodes:
                suitable_nodes[node_id] = node

        for node_id in suitable_nodes:
            node = suitable_nodes[node_id]
            if not node.is_up():
                continue

            result = node.has_free_space_for(pod) if immediate else node.has_total_space_for(pod)
            if result:
                rows.append([
                    node_id,
                    node.used_size.cpu, node.total_size.cpu,
                    node.used_size.memory, node.total_size.memory,
                    node.used_size.hdd, node.total_size.hdd,
                    node.used_size.ssd, node.total_size.ssd,
                ])

        table = pd.DataFrame(rows, columns=[
            'host',
            'used_cpu', 'total_cpu',
            'used_memory', 'total_memory',
            'used_disk /place', 'total_disk /place',
            'used_disk /ssd', 'total_disk /ssd'
        ])

        if sort_by_resource is not None:
            table.eval("free_%(name)s = total_%(name)s - used_%(name)s" % {"name": sort_by_resource.type}, inplace=True)
            table.sort_values('free_{}'.format(sort_by_resource.type), inplace=True, ascending=False)

        nodes_list = []
        for index, row in table.iterrows():
            if sort_by_resource is not None:
                if 'free_{}'.format(sort_by_resource.type) <= 0:
                    continue

            nodes_list.append(self.cache[row["host"]])

        return nodes_list

    def get_all(self, sorted_by):
        if sorted_by is None:
            return self.cache.values()

        all_nodes = list()
        for node in self.cache.values():
            all_nodes.append([node.id, node.free_size.cpu, node.free_size.memory, node.free_size.hdd, node.free_size.ssd])

        table = pd.DataFrame(all_nodes, columns=[
            "node_id",
            'cpu',
            'memory',
            'hdd',
            'ssd'
        ])

        table.sort_values(sorted_by.type, inplace=True, ascending=False)

        all_nodes = list()
        for index, row in table.iterrows():
            node_id = row["node_id"]
            all_nodes.append(self.get(node_id))

        return all_nodes


class Pod:
    def __init__(self, client, pod_id, pod_set_cache):
        self.client = client
        self.id = pod_id
        self.size = None
        self.pod_set_id = None
        self.pod_set_cache = pod_set_cache
        self.node_id = None
        self.eviction_state = None
        self.spec = None

    def __repr__(self):
        return self.id

    @staticmethod
    def create(client, pod_id, pod_set_cache):
        return Pod(client, pod_id, pod_set_cache)

    def _fill_basic_info(self):
        pod_info = self.client.get_object("pod", self.id, selectors=["/spec", "/meta/pod_set_id", "/spec/node_id", "/status/eviction", "/status/scheduling"])
        self.spec = pod_info[0]
        pod_set_id = pod_info[1]
        self.node_id = pod_info[2]
        self.size = Pod._size_from_pod_info(self.spec)
        self.pod_set_id = pod_set_id
        self.eviction_state = pod_info[3]
        self.scheduling_state = pod_info[4]

    def not_in_replication_state(self):
        if self.eviction_state is None:
            self._fill_basic_info()

        return self.eviction_state["state"] == "none" \
               and "state" in self.scheduling_state \
               and self.scheduling_state["state"] == "assigned"

    def get_size(self):
        if self.size is None:
            self._fill_basic_info()

        return self.size

    def get_node_id(self):
        return self.node_id

    def get_pod_set(self):
        if self.pod_set_id is None:
            self._fill_basic_info()

        return self.pod_set_cache.get(self.pod_set_id)

    @staticmethod
    def _size_from_pod_info(pod_info):
        def find_disk_size(pod_info, disk_type):
            return sum(
                [volume_request["quota_policy"]["capacity"] for volume_request in pod_info["disk_volume_requests"]
                 if "capacity" in volume_request["quota_policy"] and
                 volume_request["storage_class"] == disk_type])

        cpu = pod_info["resource_requests"]["vcpu_guarantee"]
        if "memory_limit" in pod_info["resource_requests"]:
            memory = pod_info["resource_requests"]["memory_limit"]
        else:
            memory = pod_info["resource_requests"]["memory_guarantee"]

        hdd = find_disk_size(pod_info, "hdd")
        ssd = find_disk_size(pod_info, "ssd")

        return Size(cpu, memory, hdd, ssd)


class PodCache:
    def __init__(self):
        self.client = None
        self.cache = None
        self.pod_set_cache = None

    def init(self, client, pod_set_cache):
        self.client = client
        self.cache = dict()
        self.pod_set_cache = pod_set_cache

    def get(self, pod_id):
        if pod_id not in self.cache:
            self.cache[pod_id] = Pod.create(self.client, pod_id, self.pod_set_cache)

        return self.cache[pod_id]

    def get_all(self, sorted_by):
        all_pods = list()
        for pod in self.client.select_objects("pod", selectors=["/meta/id"]):
            pod_id = pod[0]
            pod = self.get(pod_id)
            all_pods.append(pod)

        return sort_pods(pods=all_pods, sort_by=sorted_by, ascending=False)


def find_domitating_resource(message, pod_to_find_place):
    unsatisfied_resources_result = UnsatisfiedResources()

    unsatisfied_resources = message.split("{")[1].split("}")[0].split(",")
    for unsatisfied_resource in unsatisfied_resources:
        resource_name = unsatisfied_resource.split(":")[0].strip(' ').lower()
        resource_value = unsatisfied_resource.split(":")[1].strip(' ')

        if resource_name in ["cpu", "disk", "memory"]:
            unsatisfied_resources_result.add_resource(UnsatisfiedResource(resource_name, resource_value))

    return unsatisfied_resources_result.dominating_resource(pod_to_find_place)


def sort_pods(pods, sort_by, ascending):
    pods_dict = dict()
    pod_list = list()
    for pod in pods:
        pods_dict[pod.id] = pod
        pod_size = pod.get_size()
        pod_list.append([pod.id, pod_size.cpu, pod_size.memory, pod_size.hdd, pod_size.ssd])

    table = pd.DataFrame(pod_list, columns=[
        "pod_id",
        'cpu',
        'memory',
        'hdd',
        'ssd'
    ])

    table.sort_values(sort_by.type, inplace=True, ascending=ascending)

    pod_list = list()
    for index, row in table.iterrows():
        pod_id = row["pod_id"]
        pod = pods_dict[pod_id]
        pod_list.append(pod)

    return pod_list


class Size:
    def __init__(self, cpu, memory, hdd, ssd):
        self.cpu = cpu
        self.memory = memory
        self.hdd = hdd
        self.ssd = ssd

    @staticmethod
    def human_readable(memory, digits_after_comma=3):
        for unit in ('b', 'Kb', 'Mb', 'Gb', 'Tb'):
            if memory < 1024:
                return str(round(memory, digits_after_comma)) + unit

            memory /= 1024.

        return str(round(memory, digits_after_comma)) + 'Pb'

    def bigger_than(self, other):
        return \
            self.cpu >= other.cpu and self.memory >= other.memory and \
            self.hdd >= other.hdd and self.ssd >= other.ssd

    def __sub__(self, other):
        return Size(self.cpu - other.cpu, self.memory - other.memory, self.hdd - other.hdd, self.ssd - other.ssd)

    def __add__(self, other):
        return Size(self.cpu + other.cpu, self.memory + other.memory, self.hdd + other.hdd, self.ssd + other.ssd)

    def __repr__(self):
        return "cpu={} cores memory={} hdd={} ssd={}".format(round(self.cpu/1000, 2),
                                                             Size.human_readable(self.memory),
                                                             Size.human_readable(self.hdd),
                                                             Size.human_readable(self.ssd))


class NannyService:
    def __init__(self, id, service_info, replication_policy):
        self.id = str(id)
        self.service_info = service_info
        self.replication_policy = replication_policy
        self.pod_sets = list()

    def can_be_evicted(self):
        is_copy_and_replace = self.replication_policy is not None and self.replication_policy["policy"]["spec"]["replicationMethod"] == "REPLACE"
        # self.replication_policy["policy"]["spec"]["involuntaryReplicationChoice"] != 'DISABLED'

        if not is_copy_and_replace:
            return False

        for recipe_parameter in self.replication_policy["policy"]["spec"].get("recipeExecutionParameters", {}).get("recipeParameters", []):
            if recipe_parameter.get("value", True):
                return False

        return True

    def url(self):
        return "https://nanny.yandex-team.ru/ui/#/services/catalog/{}".format(self.id)

    def add_pod_set(self, pod_set):
        self.pod_sets.append(pod_set)

    @staticmethod
    def create(nanny_session, service_id):
        response = nanny_session.get('{}/v2/services/{}'.format(NANNY_URL, service_id), timeout=60)
        if response.status_code == 404:
            return NannyService(service_id, None, None)

        response.raise_for_status()

        service_data = response.json()
        service_id = service_data['_id']
        replication_policy = None

        deploy_engine = service_data['runtime_attrs'].get("content", {}).get("engines", {}).get("engine_type", "")
        if deploy_engine == "YP_LITE":
            response_replication_policy = nanny_session.get(
                '{}/api/repo/GetReplicationPolicy/?policyId={}'.format(NANNY_URL, service_id))
            replication_policy = None
            if response_replication_policy.status_code != 404:
                replication_policy = response_replication_policy.json()

        return NannyService(service_id, service_data, replication_policy)

    def is_online(self):
        content = self.service_info["current_state"].get("content", {})

        if content.get("is_paused", {}).get("value", False):
            return False

        if not content.get("summary", {}).get("value", "") == "ONLINE":
            return False

        active_snapshots = content.get("active_snapshots", [])
        if len(active_snapshots) == 0:
            return False

        active_snapshot = active_snapshots[0]
        is_active = active_snapshot.get("state", "") == "ACTIVE"
        return is_active

    def is_replicated_recently(self):
        # https://st.yandex-team.ru/SWAT-5953
        return False
        last_transition_time_str = self.replication_policy["policy"]["status"]["lastTransitionTime"]
        last_transition_time = dateutil.parser.parse(last_transition_time_str).replace(tzinfo=None)

        min_safe_hours = 0
        replication_policy_involuntary_timeout = self.replication_policy["policy"]["spec"].get("involuntary", {}).get("disruptionTimeoutSeconds", 3600*min_safe_hours)
        next_safe_transition_time = last_transition_time + timedelta(seconds=max(min_safe_hours*3600, replication_policy_involuntary_timeout))

        is_replicated_recently_result = next_safe_transition_time > datetime.utcnow()

        if is_replicated_recently_result:
            LOGGER.debug("Service {} last transition time={}, safe transition time={}".format(self.id,
                                                                                              last_transition_time,
                                                                                              next_safe_transition_time))

        return is_replicated_recently_result


class NannyServiceCache:
    def __init__(self, session):
        self.session = session
        self.cache = dict()

    def get_service(self, nanny_service_id):
        if nanny_service_id not in self.cache:
            self.cache[nanny_service_id] = NannyService.create(self.session, nanny_service_id)

        return self.cache[nanny_service_id]


def find_pods_can_be_evicted(pod_to_find_place, node, dominating_resource, node_cache):
    pods = sort_pods(pods=node.get_pods(), sort_by=dominating_resource, ascending=False)

    for pod in pods:
        if not pod.get_pod_set().can_evict():
            continue

        pod_set = pod.get_pod_set()
        if pod_set.size() <= 5:
            continue

        nanny_service = pod_set.get_nanny_service()
        if nanny_service is None:
            continue

        if not nanny_service.is_online() or nanny_service.is_replicated_recently():
            continue

        free_size_after = node.free_size + pod.get_size()

        if free_size_after.bigger_than(pod_to_find_place.get_size()):
            nodes = node_cache.find_nodes_suitable_for_pod_right_now(pod)

            if len(nodes) >= 3:
                return pod, nodes

    return None, None


class ClusterModel:
    @staticmethod
    def errors_only_logger(name):
        logging.basicConfig(level=logging.ERROR)
        return logging.getLogger(name)

    def __init__(self, client, nanny_token, logger, segments_in=["default"]):
        global LOGGER

        if logger is None:
            LOGGER = ClusterModel.errors_only_logger("")
        else:
            LOGGER = logger

        self.client = client
        self.nanny_service_cache = NannyServiceCache(create_session(nanny_token))
        self.podset_cache = PodSetCache()
        self.pod_cache = PodCache()
        self.node_cache = NodeCache()

        self.node_cache.init(client, self.pod_cache, segments_in=segments_in)
        self.podset_cache.init(client, self.nanny_service_cache, self.node_cache)
        self.pod_cache.init(client, self.podset_cache)
        self.podset_cache.pod_cache = self.pod_cache
        LOGGER = logger
