from .common import (
    build_yp_client,
    create_node_filter,
    print_pandas_table,
    register_common_args,
    register_common_nodes_args,
    register_measurement_args,
    register_nodes_status_args,
)

from yp.scripts.library.cluster_snapshot import select_node_resources

from yp.client import BatchingOptions

from yp.logger import logger

import pandas as pd

import argparse


# Handles both |YsonEntity| and |None|.
def _is_yson_null(yson_value):
    return yson_value == None  # noqa


def parse_args(argv):
    parser = argparse.ArgumentParser(add_help=True)

    parser.add_argument("--show-free", "--show_free", required=False, help="Show free resources instead used", action="store_true")
    register_common_args(parser)
    register_common_nodes_args(parser)
    register_measurement_args(parser)
    register_nodes_status_args(parser)

    return parser.parse_args(argv)


def main(argv):
    args = parse_args(argv)
    client = build_yp_client(args)

    use_cpu_unit = args.cpu

    timestamp = client.generate_timestamp()

    nodes = client.select_objects(
        "node",
        filter=create_node_filter(args, client, default_node_status="up"),
        selectors=[
            "/meta/id",
            "/labels",
            "/spec/network_module_id",
            "/status/alerts/0",
        ],
        batching_options=BatchingOptions(),
        timestamp=timestamp,
    )

    internet_addresses = client.select_objects(
        "internet_address",
        selectors=["/meta/id", "/spec/network_module_id", "/status/pod_id"],
        timestamp=timestamp,
    )

    available_internet_addresses = dict()
    for internet_address, network_module_id, pod_id in internet_addresses:
        if not pod_id:
            if network_module_id not in available_internet_addresses:
                available_internet_addresses[network_module_id] = list()

            available_internet_addresses[network_module_id].append(internet_address)

    resources = select_node_resources(
        client,
        node_ids=[node[0] for node in nodes],
        selectors=[
            "/meta/node_id",
            "/spec",
            "/status/free",
            "/meta/kind",
            "/status/used",
        ],
        timestamp=timestamp,
        logger=logger,
    )
    resources_by_node_id = dict()
    for node in nodes:
        assert node[0] not in resources_by_node_id
        resources_by_node_id[node[0]] = []
    for resource in resources:
        resources_by_node_id[resource[0]].append(resource)

    gpu_models = tuple(sorted(set(
        spec["gpu"]["model"]
        for _, spec, _, kind, _ in resources
        if kind == "gpu"
    )))

    resource_fields = [
        "cpu", "memory",
        "hdd", "ssd", "hdd_bw", "ssd_bw",
        "hdd_lvm", "ssd_lvm", "hdd_lvm_bw", "ssd_lvm_bw",
        "net_bw",
    ]
    for model in gpu_models:
        resource_fields.append(model)

    rows = []
    for node in nodes:
        host, labels, network_module_id, first_alert = node
        has_alerts = not _is_yson_null(first_alert)
        resources = resources_by_node_id[host]

        segment = labels["segment"]
        has_network_10g = labels.get("extras", {}).get("network", {}).get("bandwidth_10G", False)

        def default_resources_dict():
            return {field: 0 for field in resource_fields}

        total_internet_addresses = None
        if network_module_id and network_module_id in available_internet_addresses:
            total_internet_addresses = len(available_internet_addresses[network_module_id])

        total_vector = default_resources_dict()
        free_vector = default_resources_dict()
        used_vector = default_resources_dict()
        cpu_to_vcpu_factor = 1

        for _, spec, free, kind, used in resources:
            if kind == "cpu":
                cpu_to_vcpu_factor = spec[kind]["cpu_to_vcpu_factor"]
                total_vector[kind] += spec[kind]["total_capacity"]
                free_vector[kind] += free[kind]["capacity"]
                used_vector[kind] += used[kind]["capacity"]
            elif kind == "disk":
                storage_class = spec[kind]["storage_class"]
                if storage_class not in ("ssd", "hdd"):
                    continue

                storage_provisioner = spec[kind].get("storage_provisioner", "")
                if not storage_provisioner:
                    storage_provisioner = "shared"
                if storage_provisioner not in ("shared", "lvm"):
                    continue

                field_prefix = storage_class
                if storage_provisioner != "shared":
                    field_prefix += "_" + storage_provisioner

                total_vector[field_prefix] += spec[kind]["total_capacity"]
                total_vector[field_prefix + "_bw"] += spec[kind]["total_bandwidth"]

                free_vector[field_prefix] += free[kind]["capacity"]
                free_vector[field_prefix + "_bw"] += free[kind]["bandwidth"]

                used_vector[field_prefix] += used[kind]["capacity"]
                used_vector[field_prefix + "_bw"] += used[kind]["bandwidth"]
            elif kind == "network":
                total_vector["net_bw"] += spec[kind]["total_bandwidth"]
                free_vector["net_bw"] += free[kind]["bandwidth"]
                used_vector["net_bw"] += used[kind]["bandwidth"]
            elif kind == "gpu":
                model = spec["gpu"]["model"]
                total_vector[model] += 1
                free_vector[model] += free[kind]["capacity"]
                used_vector[model] += used[kind]["capacity"]
            elif kind == "memory":
                total_vector[kind] += spec[kind]["total_capacity"]
                free_vector[kind] += free[kind]["capacity"]
                used_vector[kind] += used[kind]["capacity"]
            else:
                continue

        if use_cpu_unit:
            total_vector["cpu"] /= cpu_to_vcpu_factor
            free_vector["cpu"] /= cpu_to_vcpu_factor
            used_vector["cpu"] /= cpu_to_vcpu_factor

        transient_vector = free_vector if args.show_free else used_vector

        row = [host, segment]
        for field in resource_fields:
            row.extend([transient_vector[field], total_vector[field]])
        row.extend([total_internet_addresses, has_network_10g, has_alerts])

        rows.append(row)

    cpu_column = "cpu" if use_cpu_unit else "vcpu"
    column_prefix = "free" if args.show_free else "used"

    columns = ["host", "segment"]
    for field in resource_fields:
        column = field
        if field == "cpu":
            column = cpu_column
        columns.append(column_prefix + "_" + column)
        columns.append("total_" + column)
    columns.extend(["ipv4", "net_10G", "alerts"])

    table = pd.DataFrame(rows, columns=columns)

    print_pandas_table(
        table,
        args,
        table.columns[2:-3],  # Strip non-resource fields from prefix and suffix.
    )
