#include "controller.h"

#include <infra/yp_yandex_dns_export/libs/sensors/sensors.h>

#include <infra/libs/logger/protos/events.ev.pb.h>
#include <infra/libs/sensors/macros.h>

#include <yp/yp_proto/yp/client/api/proto/data_model.pb.h>

#include <infra/contrib/pdns/power_dns/common_startup.hh>
#include <infra/contrib/pdns/power_dns/dnsparser.hh>
#include <infra/contrib/pdns/power_dns/dnsrecords.hh>
#include <infra/contrib/pdns/power_dns/iputils.hh>
#include <infra/contrib/pdns/power_dns/misc.hh>

#include <util/generic/algorithm.h>
#include <util/generic/serialized_enum.h>
#include <util/generic/xrange.h>
#include <util/string/join.h>
#include <util/thread/pool.h>

#include <util/datetime/cputimer.h>

namespace NInfra::NYandexDnsExport {

using namespace NReceivers;
using namespace NRetrievers;

namespace {

constexpr TStringBuf CREATED_BY_YP_DNS_EXPORT_LABEL_VALUE = "yp_dns_export";

const std::array<QType, 4> ALLOWED_RECORD_TYPES = {{
    QType(QType::A),
    QType(QType::PTR),
    QType(QType::AAAA),
    QType(QType::SRV),
}};

TString GetRecordClass(const DNSResourceRecord& record) {
    switch (record.qclass) {
        case QClass::IN:
            return "IN";
        case QClass::CHAOS:
            return "CHAOS";
        case QClass::NONE:
            return "NONE";
        case QClass::ANY:
            return "ANY";
        default:
            Y_UNREACHABLE();
    }
}

TString GetRecordData(const DNSResourceRecord& resourceRecord) {
    DNSRecord record(resourceRecord);
    switch (record.d_type) {
        case QType::PTR: {
            auto content = getRR<PTRRecordContent>(record);
            return TString(content->getZoneRepresentation());
        }
        case QType::A:
        case QType::AAAA: {
            return TString(getAddr(record).toString());
        }
        case QType::SRV: {
            auto content = getRR<SRVRecordContent>(record);
            return TString(content->getZoneRepresentation());
        }
        default:
            Y_UNREACHABLE();
    }
}

TMaybe<NYP::NClient::NApi::NProto::TDnsRecordSetSpec::TResourceRecord> MakeResourceRecord(const DNSResourceRecord& record) {
    if (!IsIn(ALLOWED_RECORD_TYPES, record.qtype)) {
        return Nothing();
    }

    NYP::NClient::NApi::NProto::TDnsRecordSetSpec::TResourceRecord result;
    if (record.ttl) {
        result.set_ttl(record.ttl);
    }
    result.set_class_(GetRecordClass(record));
    NYP::NClient::NApi::NProto::TDnsRecordSetSpec::TResourceRecord::EType type;
    NYP::NClient::NApi::NProto::TDnsRecordSetSpec::TResourceRecord::EType_Parse(TString(record.qtype.getName()), &type);
    result.set_type(type);
    result.set_data(GetRecordData(record));
    return result;
}


int CompareDnsResourceRecords(
    const NYP::NClient::NApi::NProto::TDnsRecordSetSpec::TResourceRecord& lhs,
    const NYP::NClient::NApi::NProto::TDnsRecordSetSpec::TResourceRecord& rhs
) {
    if (lhs.type() != rhs.type()) {
        return lhs.type() < rhs.type() ? -1 : 1;
    } else if (const int cmpClasses = TString::compare(lhs.class_(), rhs.class_()); cmpClasses != 0) {
        return cmpClasses;
    }
    return TString::compare(lhs.data(), rhs.data());
}

void SortUniqueRecords(NYP::NClient::TDnsRecordSet& recordSet) {
    Sort(*recordSet.MutableSpec()->mutable_records(), [](const auto& lhs, const auto& rhs) {
        return CompareDnsResourceRecords(lhs, rhs) < 0;
    });
    auto uniqueIt = Unique(recordSet.MutableSpec()->mutable_records()->begin(), recordSet.MutableSpec()->mutable_records()->end(), [](const auto& lhs, const auto& rhs) {
        return CompareDnsResourceRecords(lhs, rhs) == 0;
    });
    recordSet.MutableSpec()->mutable_records()->erase(uniqueIt, recordSet.MutableSpec()->mutable_records()->end());
}

void SortUniqueSources(NYP::NClient::TDnsRecordSet& recordSet) {
    auto& sources = (*recordSet.MutableLabels())["sources"];
    if (!sources.IsArray()) {
        return;
    }

    auto& sourcesArray = sources.GetArraySafe();
    SortUniqueBy(sourcesArray, [](const NJson::TJsonValue& value) {
        return value.GetString();
    });
}

TVector<TDnsRecordSet> MakeRecordSets(const TZoneConfig& zoneConfig, TVector<TResourceRecord> records, const THashMap<TString, TInstant>& sourceTimestamps) {
    const TString& zone = zoneConfig.GetName();

    TVector<TDnsRecordSet> result;
    result.reserve(records.size());

    TVector<std::tuple<TString, TString, DNSResourceRecord>> normalizedRecords;
    normalizedRecords.reserve(records.size());
    Transform(records.begin(), records.end(), std::back_inserter(normalizedRecords), [](TResourceRecord record) {
        record.Record.qname.makeUsLowerCase();
        return std::make_tuple(std::move(record.SourceName), TString(record.Record.qname.toStringNoDot()), std::move(record.Record));
    });

    SortBy(normalizedRecords, [](const auto& record) { return std::get<1>(record); });

    NYP::NClient::TDnsRecordSet recordSet;
    TInstant maxSourceTimestamp = TInstant::Zero();
    TInstant minSourceTimestamp = TInstant::Max();

    auto addRecordSetIfNonempty = [&result, &zoneConfig, &minSourceTimestamp, &maxSourceTimestamp](NYP::NClient::TDnsRecordSet& recordSet) {
        if (!recordSet.Meta().id().empty()) {
            SortUniqueRecords(recordSet);
            if (zoneConfig.GetSetSourcesLabel()) {
                SortUniqueSources(recordSet);
            }
            recordSet.MutableLabels()->SetValueByPath("timestamp", minSourceTimestamp.Seconds());
            result.emplace_back(std::move(recordSet), minSourceTimestamp, maxSourceTimestamp);
        }
    };

    for (auto& [sourceName, id, record] : normalizedRecords) {
        TMaybe<NYP::NClient::NApi::NProto::TDnsRecordSetSpec::TResourceRecord> newRecord = MakeResourceRecord(record);
        if (!newRecord.Defined()) {
            continue;
        }

        if (record.qname != DNSName(recordSet.Meta().id())) {
            addRecordSetIfNonempty(recordSet);

            recordSet = NYP::NClient::TDnsRecordSet{};
            recordSet.MutableMeta()->set_id(id);
            recordSet.MutableLabels()->SetValueByPath("zone", zone);
            recordSet.MutableLabels()->SetValueByPath("created_by", CREATED_BY_YP_DNS_EXPORT_LABEL_VALUE);
            maxSourceTimestamp = TInstant::Zero();
            minSourceTimestamp = TInstant::Max();
        }

        *recordSet.MutableSpec()->add_records() = std::move(newRecord.GetRef());
        if (zoneConfig.GetSetSourcesLabel()) {
            (*recordSet.MutableLabels())["sources"].AppendValue(sourceName);
        }
        maxSourceTimestamp = Max(maxSourceTimestamp, sourceTimestamps.at(sourceName));
        minSourceTimestamp = Min(minSourceTimestamp, sourceTimestamps.at(sourceName));
    }

    addRecordSetIfNonempty(recordSet);

    return result;
}

template <typename TMessage>
TMessage MakeRecordSetLoggingMessage(const NYP::NClient::TDnsRecordSet& recordSet) {
    TMessage recordSetLoggingMessage;
    recordSetLoggingMessage.SetName(recordSet.Meta().id());
    for (const auto& dnsRecord : recordSet.Spec().records()) {
        auto& logRecord = *recordSetLoggingMessage.add_records();
        logRecord.SetType(NYP::NClient::NApi::NProto::TDnsRecordSetSpec::TResourceRecord::EType_Name(dnsRecord.type()));
        logRecord.SetData(dnsRecord.data());
    }
    return recordSetLoggingMessage;
}

template <typename T>
void Fill(T* result, const NYP::NClient::TSelectorResult& selectorResult, int idx) {
    if (idx >= selectorResult.Values().value_payloads().size()) {
        return;
    }

    if (selectorResult.Values().value_payloads(idx).null()) {
        return;
    }

    const TString& value = selectorResult.Values().value_payloads(idx).yson();
    if (NYP::NClient::NYsonUtil::IsNull(value)) {
        return;
    }

    auto node = NYT::NodeFromYsonString(value);
    *result = node.ConvertTo<T>();
}

} // namespace

TZoneRecordsManager::TZoneRecordsManager(
    TZoneConfig config,
    const TMap<TString, TDnsRecordSet>& existingRecordSets,
    const THashSet<TString>& uncontrolledDomains,
    TVersionedRecords actualRecords,
    const EUpdateModeFlags updateMode,
    TSensorGroup sensorGroup
)
    : ZoneConfig_(std::move(config))
    , ExistingRecordSets_(existingRecordSets)
    , UncontrolledDomains_(uncontrolledDomains)
    , ActualRecords_(std::move(actualRecords))
    , UpdateMode_(updateMode)
    , SensorGroup_(std::move(sensorGroup))
{
    SensorGroup_.AddLabel("zone", ZoneConfig_.GetName());
}

TString TZoneRecordsManager::GetObjectId() const {
    return ZoneConfig_.GetName();
}

void TZoneRecordsManager::GenerateYpUpdates(
    const ISingleClusterObjectManager::TDependentObjects& /* dependentObjects */,
    TVector<ISingleClusterObjectManager::TRequest>& requests,
    TLogFramePtr frame
) const {
    TVector<TDnsRecordSet> actualRecordSets = MakeRecordSets(ZoneConfig_, ActualRecords_.ResourceRecords, ActualRecords_.SourceTimestamps);

    if (frame->AcceptLevel(ELogPriority::TLOG_RESOURCES)) {
        for (const TDnsRecordSet& actualRecordSet : actualRecordSets) {
            frame->LogEvent(ELogPriority::TLOG_RESOURCES, MakeRecordSetLoggingMessage<NLogEvent::TRetrievedRecordSet>(actualRecordSet.GetObject()));
        }
        for (const auto& [id, existingRecordSet] : ExistingRecordSets_) {
            frame->LogEvent(ELogPriority::TLOG_RESOURCES, MakeRecordSetLoggingMessage<NLogEvent::TExistingRecordSet>(existingRecordSet.GetObject()));
        }
    }

    TString updateMode = ToString(EUpdateMode(UpdateMode_.ToBaseType()));
    frame->LogEvent(NLogEvent::TZoneUpdateMode(ZoneConfig_.GetName(), updateMode));
    NInfra::TRateSensor(SensorGroup_, NSensors::GENERATE_UPDATES, {{"mode", updateMode}}).Inc();

    if (ZoneConfig_.GetDisableUpdate()) {
        Y_ENSURE(requests.empty());
        frame->LogEvent(NLogEvent::TZoneUpdateDisabled(ZoneConfig_.GetName()));
        return;
    }

    auto existingSetIt = ExistingRecordSets_.begin();
    auto actualSetIt = actualRecordSets.begin();
    while (existingSetIt != ExistingRecordSets_.end() || actualSetIt != actualRecordSets.end()) {
        if (existingSetIt == ExistingRecordSets_.end()) {
            if (NYP::NClient::TDnsRecordSet actualRecordSet = actualSetIt->GetObject(); UpdateMode_.HasFlags(EUpdateMode::CREATE) && !UncontrolledDomains_.contains(actualRecordSet.Meta().id())) {
                requests.emplace_back(NYP::NClient::TCreateObjectRequest(std::move(actualRecordSet)));
            }
            ++actualSetIt;
        } else if (actualSetIt == actualRecordSets.end()) {
            if (const NYP::NClient::TDnsRecordSet existingRecordSet = existingSetIt->second.GetObject();
                UpdateMode_.HasFlags(EUpdateMode::REMOVE)
                && existingRecordSet.Labels()["timestamp"].GetUIntegerRobust() < actualSetIt->MinSourceTimestamp().Seconds()
            ) {
                requests.emplace_back(NYP::NClient::TRemoveObjectRequest(NYP::NClient::TDnsRecordSet::ObjectType, existingRecordSet.Meta().id()));
            }
            ++existingSetIt;
        } else {
            NYP::NClient::TDnsRecordSet actualRecordSet = actualSetIt->GetObject();
            const NYP::NClient::TDnsRecordSet existingRecordSet = existingSetIt->second.GetObject();
            if (const int compareResult = TString::compare(existingRecordSet.Meta().id(), actualRecordSet.Meta().id()); compareResult < 0) {
                if (UpdateMode_.HasFlags(EUpdateMode::REMOVE) && existingRecordSet.Labels()["timestamp"].GetUIntegerRobust() < actualSetIt->MinSourceTimestamp().Seconds()) {
                    requests.emplace_back(NYP::NClient::TRemoveObjectRequest(NYP::NClient::TDnsRecordSet::ObjectType, existingRecordSet.Meta().id()));
                }
                ++existingSetIt;
            } else if (compareResult > 0) {
                if (UpdateMode_.HasFlags(EUpdateMode::CREATE) && !UncontrolledDomains_.contains(actualRecordSet.Meta().id())) {
                    requests.emplace_back(NYP::NClient::TCreateObjectRequest(std::move(actualRecordSet)));
                }
                ++actualSetIt;
            } else {
                if (UpdateMode_.HasFlags(EUpdateMode::UPDATE)) {
                    TVector<NYP::NClient::TSetRequest> setRequests;
                    if (existingRecordSet.Labels()["timestamp"].GetUIntegerRobust() < actualSetIt->MaxSourceTimestamp().Seconds() &&
                        !google::protobuf::util::MessageDifferencer::Equals(existingRecordSet.Spec(), actualRecordSet.Spec()))
                    {
                        setRequests.push_back(NYP::NClient::TSetRequest("/spec", actualRecordSet.Spec()));
                        setRequests.push_back(NYP::NClient::TSetRequest("/labels/timestamp", actualRecordSet.Labels()["timestamp"].GetUIntegerRobust()));
                    }
                    if (ZoneConfig_.GetSetSourcesLabel() && existingRecordSet.Labels()["sources"] != actualRecordSet.Labels()["sources"]) {
                        setRequests.push_back(NYP::NClient::TSetRequest("/labels/sources", NYT::NodeFromJsonValue(actualRecordSet.Labels()["sources"])));
                    }

                    if (!setRequests.empty()) {
                        requests.emplace_back(NYP::NClient::TUpdateRequest(
                            NYP::NClient::TDnsRecordSet::ObjectType,
                            actualRecordSet.Meta().id(),
                            /* set */ std::move(setRequests),
                            /* remove */ {}
                        ));
                    }
                }

                ++existingSetIt;
                ++actualSetIt;
            }
        }

        if (requests.size() >= ZoneConfig_.GetMaxUpdateObjectsNumber()) {
            break;
        }
    }
}

TZoneRecordsExportManagerFactory::TZoneRecordsExportManagerFactory(
    TDnsZonesExportConfig config
    , const NReceivers::TReceivingZones& receivingZones
    , const NRetrievers::TRetrievingZones& retrievingZones
    , NController::TClientConfig ypClientConfig
    , NController::TShardPtr shard
)
    : ISingleClusterObjectManagersFactory(TStringBuilder() << "yandex_dns_export_manager_factory_" << TStringBuf(ypClientConfig.GetAddress()).Before('.'), shard)
    , Config_(std::move(config))
    , ClusterName_(TStringBuf(ypClientConfig.GetAddress()).Before('.'))
    , YpClientConfig_(std::move(ypClientConfig))
    , RetrievingZones_(retrievingZones)
    , SensorGroup_(NSensors::ZONE_EXPORT_MANAGER_GROUP_NAME)
{
    // initialize PowerDNS lib
    seedRandom("/dev/urandom");
    reportBasicTypes();
    declareArguments();

    SensorGroup_.AddLabel("yp_address", YpClientConfig_.GetAddress());

    ZoneNames_.reserve(Config_.GetZones().size());
    for (const TZoneConfig& zoneConfig : Config_.GetZones()) {
        ZoneNames_.emplace_back(zoneConfig.GetName());
    }

    for (const auto& [path, receivingZone] : receivingZones) {
        ReceivingZones_[TString{receivingZone->SourceName()}] = receivingZone;
    }

    for (const TZoneConfig& zoneConfig : Config_.GetZones()) {
        for (const TString& source : zoneConfig.GetSources()) {
            // TODO: Revert after TRAFFIC-11961
            if (ClusterName_ == "man" && source.StartsWith("slayer")) {
                Cerr << "Skip source " << source << " for zone " << zoneConfig.GetName() << " in cluster yp-" << ClusterName_ << Endl;
                continue;
            }
            Source2Zones_[source].emplace_back(zoneConfig.GetName());
        }
    }

    InitSensors();
}

TMaybe<NController::TClientConfig> TZoneRecordsExportManagerFactory::GetYpClientConfig() const {
    return YpClientConfig_;
}

TVector<NController::ISingleClusterObjectManager::TSelectArgument> TZoneRecordsExportManagerFactory::GetSelectArguments(
    const TVector<TVector<NController::TSelectorResultPtr>>& /* aggregateResults */,
    NInfra::TLogFramePtr
) const {
    NYP::NClient::TSelectObjectsOptions options;
    options.SetLimit(Config_.GetSelectChunkSize());

    return {NController::ISingleClusterObjectManager::TSelectArgument{
        NYP::NClient::TDnsRecordSet::ObjectType,
        /* selectors = */ {
            "/meta/id",
            "/spec",
            "/labels/timestamp",
            "/labels/zone",
            "/labels/created_by",
            "/labels/sources",
        },
        /* filter */ "",
        options, /* options */
    }};
}

void TZoneRecordsExportManagerFactory::FillSelectedRecords(const TVector<NController::TSelectorResultPtr>& selectorResults) const {
    for (const TDnsName& zoneName : ZoneNames_) {
        Zone2RecordsSets_[zoneName].clear();
        Zone2UncontrolledDomains_[zoneName].clear();
    }

    for (const NController::TSelectorResultPtr& selectedRecordSet : selectorResults) {
        TString id;
        Fill(&id, *selectedRecordSet, /* /meta/id index */ 0);

        TString zone;
        Fill(&zone, *selectedRecordSet, /* /labels/zone index */ 3);

        TString createdBy;
        Fill(&createdBy, *selectedRecordSet, /* /labels/created_by index */ 4);

        if (zone.empty()) {
            const DNSName recordSetName(id);
            for (const TDnsName& zoneName : ZoneNames_) {
                if (recordSetName.isPartOf(zoneName.DnsName())) {
                    zone = zoneName;
                    break;
                }
            }
        }

        if (zone.empty()) {
            continue;
        }

        if (createdBy != CREATED_BY_YP_DNS_EXPORT_LABEL_VALUE) {
            Zone2UncontrolledDomains_[std::move(zone)].insert(std::move(id));
        } else if (auto* zoneRecordSetsIt = Zone2RecordsSets_.FindPtr(zone)) {
            NYP::NClient::TDnsRecordSet recordSet;
            ui64 timestamp = 0;
            TString zoneInObject;
            NJson::TJsonValue sources;
            selectedRecordSet->Fill(
                recordSet.MutableMeta()->mutable_id(),
                recordSet.MutableSpec(),
                &timestamp,
                &zoneInObject,
                &createdBy,
                &sources
            );

            recordSet.MutableLabels()->SetValueByPath("timestamp", timestamp);
            recordSet.MutableLabels()->SetValueByPath("sources", sources);

            zoneRecordSetsIt->emplace(
                std::piecewise_construct,
                std::forward_as_tuple(std::move(id)),
                std::forward_as_tuple(recordSet, TInstant::Seconds(timestamp), TInstant::Seconds(timestamp)));
        }
    }
}

TVector<TExpected<NController::TSingleClusterObjectManagerPtr, TZoneRecordsExportManagerFactory::TValidationError>> TZoneRecordsExportManagerFactory::GetSingleClusterObjectManagers(
    const TVector<NController::TSelectObjectsResultPtr>& selectorResults,
    TLogFramePtr frame
) const {
    THolder<IThreadPool> pool = CreateThreadPool(Config_.GetSourcesPollThreadPoolSize(), 0, IThreadPool::TParams().SetBlocking(false).SetCatching(false));

    pool->SafeAddFunc([this, &selectorResults] {
        FillSelectedRecords(selectorResults[0]->Results);
    });

    THashMap<TString, THashSet<TString>> zone2UnprocessedSources(Config_.GetZones().size());
    THashMap<TString, TMutex> actualRecordsMutexes;
    THashMap<TString, TVersionedRecords> zone2ActualResourceRecords(Config_.GetZones().size());
    for (const TZoneConfig& zone : Config_.GetZones()) {
        THashSet<TString> sources(zone.GetSources().size());
        for (const TString& source : zone.GetSources()) {
            // TODO: Revert after TRAFFIC-11961
            if (ClusterName_ == "man" && source.StartsWith("slayer")) {
                continue;
            }
            sources.insert(source);
        }
        zone2UnprocessedSources.try_emplace(zone.GetName(), std::move(sources));
        Y_UNUSED(actualRecordsMutexes[zone.GetName()]);
        zone2ActualResourceRecords[zone.GetName()].ResourceRecords.reserve(zone.GetResourceRecordsReserve());
    }

    auto addActualRecords = [this, frame, &actualRecordsMutexes, &zone2UnprocessedSources, &zone2ActualResourceRecords](const TString& sourceName, TZoneSnapshot actualRecords) {
        TSensorGroup sensorGroup = SensorGroup_;
        sensorGroup.AddLabel("source", sourceName);

        const TVector<TDnsName>* zones = Source2Zones_.FindPtr(sourceName);
        if (!zones) {
            return;
        }

        for (const TDnsName& zone : *zones) {
            TGuard<TMutex> guard(actualRecordsMutexes[zone]);
            zone2UnprocessedSources[zone].erase(sourceName);
            auto& actualResourceRecords = zone2ActualResourceRecords[zone];
            actualResourceRecords.SourceTimestamps.emplace(sourceName, actualRecords.Timestamp);
            for (const DNSResourceRecord& resourceRecord : *actualRecords.Records) {
                if (resourceRecord.qname.isPartOf(zone.DnsName())) {
                    actualResourceRecords.ResourceRecords.push_back(TResourceRecord{sourceName, resourceRecord});
                }
            }
        }
    };

    for (const auto& [sourceName, receivingZone] : ReceivingZones_) {
        pool->SafeAddFunc([this, addActualRecords, &sourceName = sourceName, receivingZone = receivingZone, frame] {
            TZoneSnapshot records;
            try {
                records = ReceiveRecords(sourceName, receivingZone.Get(), frame);
            } catch (...) {
                return;
            }
            addActualRecords(sourceName, std::move(records));
        });
    }

    for (const auto& [sourceName, retrievingZone] : RetrievingZones_) {
        pool->SafeAddFunc([this, addActualRecords, &sourceName = sourceName, retrievingZone = retrievingZone, frame] {
            TZoneSnapshot records;
            try {
                records = RetrieveRecords(sourceName, retrievingZone.Get(), frame);
            } catch (...) {
                return;
            }
            addActualRecords(sourceName, std::move(records));
        });
    }

    pool->Stop();

    TVector<TExpected<NController::TSingleClusterObjectManagerPtr, TZoneRecordsExportManagerFactory::TValidationError>> result;
    result.reserve(Config_.GetZones().size());
    for (const TZoneConfig& zoneConfig : Config_.GetZones()) {
        const auto& recordSets = Zone2RecordsSets_.at(zoneConfig.GetName());
        const auto& uncontrolledDomains = Zone2UncontrolledDomains_.at(zoneConfig.GetName());
        auto& actualResourceRecords = zone2ActualResourceRecords[zoneConfig.GetName()];

        frame->LogEvent(ELogPriority::TLOG_INFO, NLogEvent::TSelectedZone(zoneConfig.GetName(), recordSets.size(), uncontrolledDomains.size(), actualResourceRecords.ResourceRecords.size()));

        if (const THashSet<TString>& unprocessedSources = zone2UnprocessedSources.at(zoneConfig.GetName()); unprocessedSources.empty()) {
            result.emplace_back(new TZoneRecordsManager(zoneConfig, recordSets, uncontrolledDomains, std::move(actualResourceRecords), TZoneRecordsManager::EUpdateMode::ALL, SensorGroup_));
        } else {
            result.emplace_back(new TZoneRecordsManager(zoneConfig, recordSets, uncontrolledDomains, std::move(actualResourceRecords), TZoneRecordsManager::EUpdateMode::CREATE_AND_UPDATE, SensorGroup_));
            NLogEvent::TZoneUnprocessedSources unprocessedSourcesMessage;
            unprocessedSourcesMessage.SetZone(zoneConfig.GetName());
            unprocessedSourcesMessage.MutableUnprocessedSources()->Reserve(unprocessedSources.size());
            for (const TString& unprocessedSource : unprocessedSources) {
                *unprocessedSourcesMessage.AddUnprocessedSources() = unprocessedSource;
            }
            frame->LogEvent(ELogPriority::TLOG_WARNING, std::move(unprocessedSourcesMessage));
        }
    }

    return result;
}

TZoneSnapshot TZoneRecordsExportManagerFactory::DoReceiveRecords(const TString& sourceName, TReceivingZone* receivingZone, TLogFramePtr /* frame */, const TSensorGroup& /* sensorGroup */) const {
    if (!receivingZone) {
        ythrow yexception() << "No receiving zone for source \"" << sourceName << "\"";
    }

    TZoneSnapshot zone = receivingZone->GetSnapshot();
    if (!zone.Records || zone.Records->empty()) {
        ythrow yexception() << "Source \"" << sourceName << "\": no records found";
    }

    return zone;
}

void TZoneRecordsExportManagerFactory::InitSensors() {
    TSensorGroup receiveSensorGroup(SensorGroup_, NSensors::RECEIVE_GROUP_NAME);
    for (const auto& [sourceName, receivingZone] : ReceivingZones_) {
        TSensorGroup sensorGroup = receiveSensorGroup;
        sensorGroup.AddLabel("source", sourceName);
        TRateSensor(sensorGroup, NSensors::RECEIVE_NUMBER);
        TRateSensor(sensorGroup, NSensors::RECEIVE_SUCCESS);
        TRateSensor(sensorGroup, NSensors::RECEIVE_FAILED);
    }

    TSensorGroup retrieveSensorGroup(SensorGroup_, NSensors::RETRIEVE_GROUP_NAME);
    for (const auto& [sourceName, retrievingZone] : RetrievingZones_) {
        TSensorGroup sensorGroup = retrieveSensorGroup;
        sensorGroup.AddLabel("source", sourceName);
        TRateSensor(sensorGroup, NSensors::RETRIEVE_NUMBER);
        TRateSensor(sensorGroup, NSensors::RETRIEVE_SUCCESS);
        TRateSensor(sensorGroup, NSensors::RETRIEVE_FAILED);
    }

    for (const TZoneConfig& zone : Config_.GetZones()) {
        TSensorGroup sensorGroup = SensorGroup_;
        sensorGroup.AddLabel("zone", zone.GetName());
        for (const auto& mode : GetEnumAllValues<TZoneRecordsManager::EUpdateMode>()) {
            TRateSensor(sensorGroup, NSensors::GENERATE_UPDATES, {{"mode", ToString(mode)}});
        }
    }
}

TZoneSnapshot TZoneRecordsExportManagerFactory::ReceiveRecords(const TString& sourceName, TReceivingZone* receivingZone, TLogFramePtr frame) const {
    TSensorGroup receiveSensorGroup(SensorGroup_, NSensors::RECEIVE_GROUP_NAME);
    receiveSensorGroup.AddLabel("source", sourceName);

    TDurationSensor(receiveSensorGroup, NSensors::RECEIVE_DURATION).Start();
    NON_STATIC_INFRA_RATE_SENSOR(receiveSensorGroup, NSensors::RECEIVE_NUMBER);

    frame->LogEvent(NLogEvent::TStartManagerReceiveZone(sourceName));

    TZoneSnapshot records;
    try {
        records = DoReceiveRecords(sourceName, receivingZone, frame, receiveSensorGroup);
    } catch (...) {
        frame->LogEvent(ELogPriority::TLOG_ERR, NLogEvent::TManagerReceiveZoneFailure(sourceName, CurrentExceptionMessage()));
        NON_STATIC_INFRA_RATE_SENSOR(receiveSensorGroup, NSensors::RECEIVE_FAILED);
        TDurationSensor(receiveSensorGroup, NSensors::RECEIVE_DURATION).Update();
        throw;
    }

    NON_STATIC_INFRA_RATE_SENSOR(receiveSensorGroup, NSensors::RECEIVE_SUCCESS);
    TDurationSensor(receiveSensorGroup, NSensors::RECEIVE_DURATION).Update();

    frame->LogEvent(NLogEvent::TManagerReceiveZoneResult(records.Records->size()));

    return records;
}

TZoneSnapshot TZoneRecordsExportManagerFactory::DoRetrieveRecords(const TString& sourceName, TRetrievingZone* retrievingZone, TLogFramePtr /* frame */, const TSensorGroup& /* sensorGroup */) const {
    if (!retrievingZone) {
        ythrow yexception() << "No retrieving zone for source \"" << sourceName << "\"";
    }

    TZoneSnapshot zone = retrievingZone->GetSnapshot();
    if (!zone.Records || zone.Records->empty()) {
        ythrow yexception() << "Source \"" << sourceName << "\": no records found";
    }

    return zone;
}

TZoneSnapshot TZoneRecordsExportManagerFactory::RetrieveRecords(const TString& sourceName, NRetrievers::TRetrievingZone* retrievingZone, TLogFramePtr frame) const {
    TSensorGroup retrieveSensorGroup(SensorGroup_, NSensors::RETRIEVE_GROUP_NAME);
    retrieveSensorGroup.AddLabel("source", sourceName);

    TDurationSensor(retrieveSensorGroup, NSensors::RETRIEVE_DURATION).Start();
    NON_STATIC_INFRA_RATE_SENSOR(retrieveSensorGroup, NSensors::RETRIEVE_NUMBER);

    frame->LogEvent(NLogEvent::TManagerStartRetrieveZone(sourceName));

    TZoneSnapshot records;
    try {
        records = DoRetrieveRecords(sourceName, retrievingZone, frame, retrieveSensorGroup);
    } catch (...) {
        frame->LogEvent(ELogPriority::TLOG_ERR, NLogEvent::TManagerRetrieveZoneFailure(sourceName, CurrentExceptionMessage()));
        NON_STATIC_INFRA_RATE_SENSOR(retrieveSensorGroup, NSensors::RETRIEVE_FAILED);
        TDurationSensor(retrieveSensorGroup, NSensors::RETRIEVE_DURATION).Update();
        throw;
    }

    NON_STATIC_INFRA_RATE_SENSOR(retrieveSensorGroup, NSensors::RETRIEVE_SUCCESS);
    TDurationSensor(retrieveSensorGroup, NSensors::RETRIEVE_DURATION).Update();

    frame->LogEvent(NLogEvent::TManagerRetrieveZoneResult(records.Records->size()));

    return records;
}

} // namespace NInfra::NYandexDnsExport
