#pragma once

#include <infra/yp_yandex_dns_export/libs/config/config.pb.h>
#include <infra/yp_yandex_dns_export/libs/zone_receivers/receiver.h>
#include <infra/yp_yandex_dns_export/libs/zone_receivers/receiving_zone.h>
#include <infra/yp_yandex_dns_export/libs/zone_retrievers/retriever.h>
#include <infra/yp_yandex_dns_export/libs/zone_retrievers/retrieving_zone.h>

#include <infra/libs/controller/config/config.pb.h>
#include <infra/libs/controller/object_manager/object_manager.h>
#include <infra/libs/sensors/sensor.h>
#include <infra/libs/sensors/sensor_group.h>
#include <yp/cpp/yp/data_model.h>

#include <util/generic/flags.h>
#include <util/generic/hash.h>

namespace NInfra::NYandexDnsExport {

struct TResourceRecord {
    TString SourceName;
    DNSResourceRecord Record;
};

struct TVersionedRecords {
    THashMap<TString, TInstant> SourceTimestamps;
    TVector<TResourceRecord> ResourceRecords;
};

class TDnsName: public TString {
public:
    using TString::TString;

    const DNSName& DnsName() const {
        if (!DnsName_.Defined()) {
            DnsName_ = DNSName(*this);
        }
        return *DnsName_;
    }

private:
    mutable TMaybe<DNSName> DnsName_;
};

class TDnsRecordSet {
public:
    explicit TDnsRecordSet() = default;

    explicit TDnsRecordSet(
        const NYP::NClient::TDnsRecordSet& recordSet,
        TInstant minSourceTimestamp,
        TInstant maxSourceTimestamp
    )
        : Serialized_(recordSet.Serialize())
        , MinSourceTimestamp_(minSourceTimestamp)
        , MaxSourceTimestamp_(maxSourceTimestamp)
    {
    }

    NYP::NClient::TDnsRecordSet GetObject() const {
        NYP::NClient::TDnsRecordSet object;
        object.Deserialize(Serialized_);
        return object;
    }

    TInstant MinSourceTimestamp() const {
        return MinSourceTimestamp_;
    }

    TInstant MaxSourceTimestamp() const {
        return MaxSourceTimestamp_;
    }

private:
    TString Serialized_;
    TInstant MinSourceTimestamp_;
    TInstant MaxSourceTimestamp_;
};

class TZoneRecordsManager : public NController::ISingleClusterObjectManager {
public:
    enum class EUpdateMode {
        NOTHING = 0,
        CREATE = 1,
        UPDATE = 2,
        REMOVE = 4,
        CREATE_AND_UPDATE = 3,
        CREATE_AND_REMOVE = 5,
        UPDATE_AND_REMOVE = 6,
        ALL = 7,
    };
    Y_DECLARE_FLAGS(EUpdateModeFlags, EUpdateMode)

    TZoneRecordsManager(
        TZoneConfig config,
        const TMap<TString, TDnsRecordSet>& existingRecordSets,
        const THashSet<TString>& uncontrolledDomains,
        TVersionedRecords actualRecords,
        const EUpdateModeFlags updateMode,
        TSensorGroup sensorGroup
    );

    TString GetObjectId() const override final;

    void GenerateYpUpdates(
        const ISingleClusterObjectManager::TDependentObjects& dependentObjects,
        TVector<ISingleClusterObjectManager::TRequest>& requests,
        TLogFramePtr frame
    ) const override final;

private:
    const TZoneConfig ZoneConfig_;
    const TMap<TString, TDnsRecordSet>& ExistingRecordSets_;
    const THashSet<TString>& UncontrolledDomains_;
    const TVersionedRecords ActualRecords_;
    const EUpdateModeFlags UpdateMode_;
    TSensorGroup SensorGroup_;
};

Y_DECLARE_OPERATORS_FOR_FLAGS(TZoneRecordsManager::EUpdateModeFlags)

class TZoneRecordsExportManagerFactory : public NController::ISingleClusterObjectManagersFactory {
public:
    TZoneRecordsExportManagerFactory(
        TDnsZonesExportConfig exportConfig,
        const NReceivers::TReceivingZones& receivingZones,
        const NRetrievers::TRetrievingZones& retrievingZones,
        NController::TClientConfig ypClientConfig,
        NController::TShardPtr shard
    );

    TMaybe<NController::TClientConfig> GetYpClientConfig() const override;

    TVector<NController::ISingleClusterObjectManager::TSelectArgument> GetSelectArguments(
        const TVector<TVector<NController::TSelectorResultPtr>>& /* aggregateResults */ = {},
        NInfra::TLogFramePtr = {}
    ) const override final;

    void FillSelectedRecords(const TVector<NController::TSelectorResultPtr>& selectorResults) const;

    TVector<TExpected<NController::TSingleClusterObjectManagerPtr, TValidationError>> GetSingleClusterObjectManagers(
        const TVector<NController::TSelectObjectsResultPtr>& selectorResults,
        TLogFramePtr frame
    ) const override final;

private:
    void InitSensors();

    TZoneSnapshot ReceiveRecords(const TString& sourceName, NReceivers::TReceivingZone* receivingZone, TLogFramePtr frame) const;
    TZoneSnapshot RetrieveRecords(const TString& sourceName, NRetrievers::TRetrievingZone* retrievingZone, TLogFramePtr frame) const;

    TZoneSnapshot DoReceiveRecords(const TString& sourceName, NReceivers::TReceivingZone* receivingZone, TLogFramePtr frame, const TSensorGroup& sensorGroup) const;
    TZoneSnapshot DoRetrieveRecords(const TString& sourceName, NRetrievers::TRetrievingZone* retrievingZone, TLogFramePtr frame, const TSensorGroup& sensorGroup) const;

private:
    const TDnsZonesExportConfig Config_;
    const TString ClusterName_;
    const NController::TClientConfig YpClientConfig_;
    TVector<TDnsName> ZoneNames_;
    NReceivers::TReceivingZones ReceivingZones_;
    NRetrievers::TRetrievingZones RetrievingZones_;
    THashMap<TString, TVector<TDnsName>> Source2Zones_;

    TSensorGroup SensorGroup_;

    mutable THashMap<TString, TMap<TString, TDnsRecordSet>> Zone2RecordsSets_;
    mutable THashMap<TString, THashSet<TString>> Zone2UncontrolledDomains_;
};

} // namespace NInfra::NYandexDnsExport
