#include "dynamic_zones_export_controller.h"

#include <infra/yp_yandex_dns_export/libs/util/regexp.h>

#include <infra/libs/logger/protos/events.ev.pb.h>

#include <yp/cpp/yp/data_model.h>

#include <library/cpp/yson/node/node_io.h>

namespace NInfra::NYandexDnsExport {

////////////////////////////////////////////////////////////////////////////////

TDynamicZonesExportManager::TDynamicZonesExportManager(
    TVector<NYP::NClient::TDnsZone> existingZones,
    THashMap<TString, TZonesListSnapshot> actualZonesLists,
    const IMatcher& zonesFilterMatcher
)
    : ExistingZones_(std::move(existingZones))
    , ActualZonesLists_(std::move(actualZonesLists))
    , ZonesFilterMatcher_(zonesFilterMatcher)
{
}

TString TDynamicZonesExportManager::GetObjectId() const {
    return "all_dynamic_zones";
}

void TDynamicZonesExportManager::GenerateYpUpdates(
    const NController::IObjectManager::TDependentObjects& dependentObjects,
    TVector<NController::IObjectManager::TRequest>& requests,
    TLogFramePtr logFrame
) const {
    TVector<NYP::NClient::TDnsZone> actuallyEnabledZones;
    size_t actualZonesNumber = 0;
    for (const auto& [sourceName, zonesListSnapshot] : ActualZonesLists_) {
        Y_ENSURE(zonesListSnapshot.Zones);
        actualZonesNumber += zonesListSnapshot.Zones->size();
    }
    actuallyEnabledZones.reserve(actualZonesNumber);

    TInstant minSourceTimestamp = TInstant::Max();
    for (const auto& [sourceName, zonesListSnapshot] : ActualZonesLists_) {
        minSourceTimestamp = Min(minSourceTimestamp, zonesListSnapshot.Timestamp);
        for (NYP::NClient::TDnsZone zone : *zonesListSnapshot.Zones) {
            if (!ZonesFilterMatcher_.Match(zone.Meta().id())) {
                logFrame->LogEvent(NLogEvent::TFilterZoneOut(zone.Meta().id(), "matcher failed"));
                continue;
            }
            zone.MutableLabels()->SetValueByPath(
                "yp_dns_export.timestamp",
                zonesListSnapshot.Timestamp.Seconds());
            actuallyEnabledZones.emplace_back(std::move(zone));
        }
    }

    auto getZoneSortKey = [](const NYP::NClient::TDnsZone& zone) {
        return std::tuple(
            zone.Meta().id(),
            -zone.Labels()["yp_dns_export"]["timestamp"].GetIntegerRobust()
        );
    };
    auto getZoneUniqueKey = [](const NYP::NClient::TDnsZone& zone) { return zone.Meta().id(); };

    auto& existingZones = ExistingZones_;

    SortBy(existingZones, getZoneSortKey);
    SortBy(actuallyEnabledZones, getZoneSortKey);
    UniqueBy(actuallyEnabledZones.begin(), actuallyEnabledZones.end(), getZoneUniqueKey);

    auto existingZonesIt = existingZones.begin();
    auto actuallyEnabledZonesIt = actuallyEnabledZones.begin();

    while (existingZonesIt != existingZones.end()) {
        const NYP::NClient::TDnsZone& existingZone = *existingZonesIt;

        logFrame->LogEvent(NLogEvent::THandleZones(
            existingZone.Meta().id(),
            actuallyEnabledZonesIt != actuallyEnabledZones.end()
                ? actuallyEnabledZonesIt->Meta().id()
                : TString{}));

        TVector<NYP::NClient::TSetRequest> setRequests;
        if (actuallyEnabledZonesIt == actuallyEnabledZones.end()) {
            setRequests = GenerateUpdates(existingZone, Nothing(), minSourceTimestamp, logFrame);

            ++existingZonesIt;
        } else {
            const NYP::NClient::TDnsZone& actuallyEnabledZone = *actuallyEnabledZonesIt;
            if (const int compareIdsResult =
                    TString::compare(existingZone.Meta().id(), actuallyEnabledZone.Meta().id());
                compareIdsResult < 0)
            {
                setRequests = GenerateUpdates(existingZone, Nothing(), minSourceTimestamp, logFrame);

                ++existingZonesIt;
            } else if (compareIdsResult > 0) {
                logFrame->LogEvent(TLOG_WARNING, NLogEvent::TGenerateUpdatesToZonePass(
                    actuallyEnabledZone.Meta().id(),
                    "Enabled zone not found in YP: cannot update it"));

                ++actuallyEnabledZonesIt;
            } else {
                setRequests = GenerateUpdates(existingZone, actuallyEnabledZone, logFrame);

                ++existingZonesIt;
                ++actuallyEnabledZonesIt;
            }
        }
        if (!setRequests.empty()) {
            requests.emplace_back(NYP::NClient::TUpdateRequest(
                NYP::NClient::TDnsZone::ObjectType,
                existingZone.Meta().id(),
                /* set */ std::move(setRequests),
                /* remove */ {}
            ));
        }
    }

    while (actuallyEnabledZonesIt != actuallyEnabledZones.end()) {
        logFrame->LogEvent(TLOG_WARNING, NLogEvent::TGenerateUpdatesToZonePass(
            actuallyEnabledZonesIt->Meta().id(),
            "Enabled zone not found in YP: cannot update it"));

        ++actuallyEnabledZonesIt;
    }
}

TVector<NYP::NClient::TSetRequest> TDynamicZonesExportManager::GenerateUpdates(
    const NYP::NClient::TDnsZone& existing,
    const NYP::NClient::TDnsZone& actual,
    NInfra::TLogFramePtr logFrame
) const {
    Y_ENSURE(existing.Meta().id() == actual.Meta().id());
    logFrame->LogEvent(NLogEvent::TGenerateUpdatesToEnableZone(existing.Meta().id()));

    if (existing.Labels()["yp_dns_export"]["timestamp"].GetUIntegerRobust()
            > actual.Labels()["yp_dns_export"]["timestamp"].GetUIntegerRobust()) {
        logFrame->LogEvent(NLogEvent::TGenerateUpdatesToZonePass(
            existing.Meta().id(),
            Sprintf("New timestamp [%llu] is less than existing [%llu]",
                actual.Labels()["yp_dns_export"]["timestamp"].GetUIntegerRobust(),
                existing.Labels()["yp_dns_export"]["timestamp"].GetUIntegerRobust())));
        return {};
    }

    auto existingZoneSpec = existing.Spec();
    auto actualZoneSpec = actual.Spec();
    for (auto* spec : {&existingZoneSpec, &actualZoneSpec}) {
        spec->mutable_config()->mutable_default_soa_record()->mutable_soa()->clear_serial();
        // TODO: remove after bridge is ready to send expire field in response
        spec->mutable_config()->mutable_default_soa_record()->mutable_soa()->clear_expire();
    }

    if (!google::protobuf::util::MessageDifferencer::Equals(existingZoneSpec, actualZoneSpec)) {
        logFrame->LogEvent(NLogEvent::TGenerateUpdatesToZonePass(
            existing.Meta().id(),
            "Specs of existing zone and actual zone are different"));
        return {};
    }

    if (existing.Labels()["yp_dns"]["enable"].GetBooleanRobust()) {
        logFrame->LogEvent(NLogEvent::TGenerateUpdatesToZonePass(
            existing.Meta().id(),
            "Existing zone already enabled"));
        return {};
    }

    logFrame->LogEvent(NLogEvent::TSetUpdateForZone(
        existing.Meta().id(),
        "/labels/yp_dns/enable",
        ToString(true)));
    logFrame->LogEvent(NLogEvent::TSetUpdateForZone(
        existing.Meta().id(),
        "/labels/yp_dns_export/timestamp",
        ToString(actual.Labels()["yp_dns_export"]["timestamp"].GetUIntegerRobust())));

    return {
        NYP::NClient::TSetRequest(
            "/labels/yp_dns/enable",
            true),
        NYP::NClient::TSetRequest(
            "/labels/yp_dns_export/timestamp",
            actual.Labels()["yp_dns_export"]["timestamp"].GetUIntegerRobust()),
    };
}

TVector<NYP::NClient::TSetRequest> TDynamicZonesExportManager::GenerateUpdates(
    const NYP::NClient::TDnsZone& existing,
    const TNothing& actual,
    const TInstant minSourceTimestamp,
    NInfra::TLogFramePtr logFrame
) const {
    logFrame->LogEvent(NLogEvent::TGenerateUpdatesToDisableZone(existing.Meta().id()));

    TVector<NYP::NClient::TSetRequest> result;

    if (existing.Labels()["yp_dns_export"]["timestamp"].GetUIntegerRobust() > minSourceTimestamp.Seconds()) {
        logFrame->LogEvent(NLogEvent::TGenerateUpdatesToZonePass(
            existing.Meta().id(),
            Sprintf("Min known source timestamp [%lu] is less than existing timestamp [%llu]",
                minSourceTimestamp.Seconds(),
                existing.Labels()["yp_dns_export"]["timestamp"].GetUIntegerRobust())));
        return result;
    }

    if (existing.Labels()["yp_dns"]["enable"].GetBooleanRobust()) {
        logFrame->LogEvent(NLogEvent::TSetUpdateForZone(
            existing.Meta().id(),
            "/labels/yp_dns/enable",
            ToString(false)));
        result.emplace_back("/labels/yp_dns/enable", false);

        if (existing.Labels()["yp_dns_export"]["timestamp"].GetUIntegerRobust() != minSourceTimestamp.Seconds()) {
            logFrame->LogEvent(NLogEvent::TSetUpdateForZone(
                existing.Meta().id(),
                "/labels/yp_dns_export/timestamp",
                ToString(minSourceTimestamp.Seconds())));
            result.emplace_back("/labels/yp_dns_export/timestamp", minSourceTimestamp.Seconds());
        } else {
            logFrame->LogEvent(NLogEvent::TSkipSetUpdateForZone(
                existing.Meta().id(),
                "/labels/yp_dns_export/timestamp",
                Sprintf("Min known source timestamp is equal to existing timestamp: [%lu]",
                    minSourceTimestamp.Seconds())));
        }
    } else {
        logFrame->LogEvent(NLogEvent::TSkipSetUpdateForZone(
            existing.Meta().id(),
            "/labels/yp_dns/enable",
            "Existing zone already disabled"));
    }

    return result;
}

////////////////////////////////////////////////////////////////////////////////


TDynamicZonesExportManagersFactory::TDynamicZonesExportManagersFactory(
    TDnsZonesListExportConfig config,
    NRetrievers::TRetrievingZonesLists retrievingZonesLists
    , NController::TClientConfig ypClientConfig
    , NController::TShardPtr shard
)
    : ISingleClusterObjectManagersFactory(
        TStringBuilder()
            << "dynamic_zones_export_manager_factory_"
            << TStringBuf(ypClientConfig.GetAddress()).Before('.')
        , shard)
    , Config_(std::move(config))
    , YpClientConfig_(std::move(ypClientConfig))
    , RetrievingZonesLists_(std::move(retrievingZonesLists))
    , ZonesFilterMatcher_(
        Config_.GetZonesFiltrationConfig().GetEnabled()
            ? CreateAnyPatternMatcher(TVector<TString>{
                Config_.GetZonesFiltrationConfig().GetRegexps().begin(),
                Config_.GetZonesFiltrationConfig().GetRegexps().end()})
            : CreateConstMatcher(true))
{
}

TMaybe<NController::TClientConfig> TDynamicZonesExportManagersFactory::GetYpClientConfig() const {
    return YpClientConfig_;
}

TVector<NController::ISingleClusterObjectManager::TSelectArgument>
TDynamicZonesExportManagersFactory::GetSelectArguments(
    const TVector<TVector<NController::TSelectorResultPtr>>& /* aggregateResults */,
    NInfra::TLogFramePtr /* logFrame */
) const {
    return {
        NController::ISingleClusterObjectManager::TSelectArgument{
            NYP::NClient::TDnsZone::ObjectType,
            /* selectors = */ {
                "/meta/id",
                "/spec",
                "/labels/yp_dns_export",
                "/labels/yp_dns"
            },
            /* filter */ "",
            NYP::NClient::TSelectObjectsOptions()
                .SetLimit(1000)
        }
    };
}

TVector<TDynamicZonesExportManagersFactory::TObjectManagerOrError>
TDynamicZonesExportManagersFactory::GetSingleClusterObjectManagers(
    const TVector<NController::TSelectObjectsResultPtr>& selectObjectsResults,
    TLogFramePtr logFrame
) const {
    Y_ENSURE(selectObjectsResults.size() == 1);

    TVector<NYP::NClient::TDnsZone> existingZones = MakeZonesList(selectObjectsResults[0]);

    THashMap<TString, TZonesListSnapshot> zonesLists;
    for (const auto& [sourceName, retrievingZonesList] : RetrievingZonesLists_) {
        zonesLists.emplace(
            sourceName,
            RetrieveZonesList(
                sourceName,
                retrievingZonesList.Get(),
                logFrame
            ));
    }

    TVector<TDynamicZonesExportManagersFactory::TObjectManagerOrError> result;
    result.emplace_back(
        new TDynamicZonesExportManager(
            std::move(existingZones),
            std::move(zonesLists),
            *ZonesFilterMatcher_));

    return result;
}

TZonesListSnapshot TDynamicZonesExportManagersFactory::RetrieveZonesList(
    const TString& sourceName,
    const NRetrievers::TRetrievingZonesList* retrievingZonesList,
    NInfra::TLogFramePtr logFrame
) const {
    TZonesListSnapshot zones;
    try {
        zones = DoRetrieveZonesList(sourceName, retrievingZonesList, logFrame);
    } catch (...) {
        // TODO: log error
        throw;
    }

    return zones;
}

TZonesListSnapshot TDynamicZonesExportManagersFactory::DoRetrieveZonesList(
    const TString& sourceName,
    const NRetrievers::TRetrievingZonesList* retrievingZonesList,
    NInfra::TLogFramePtr logFrame
) const  {
    if (!retrievingZonesList) {
        ythrow yexception() << "No retrieving zones list for source \"" << sourceName << "\"";
    }

    TZonesListSnapshot zones = retrievingZonesList->GetSnapshot();
    if (!zones.Zones || zones.Zones->empty()) {
        ythrow yexception() << "Source \"" << sourceName << "\": empty zones list";
    }

    return zones;
}

TVector<NYP::NClient::TDnsZone> TDynamicZonesExportManagersFactory::MakeZonesList(
    NController::TSelectObjectsResultPtr selectObjectsResult
) const {
    Y_ENSURE(selectObjectsResult);

    TVector<NYP::NClient::TDnsZone> result;
    result.reserve(selectObjectsResult->Results.size());

    for (const NController::TSelectorResultPtr& selectedZone : selectObjectsResult->Results) {
        NYP::NClient::TDnsZone& zone = result.emplace_back();
        selectedZone->Fill(
            zone.MutableMeta()->mutable_id(),
            zone.MutableSpec(),
            &(*zone.MutableLabels())["yp_dns_export"],
            &(*zone.MutableLabels())["yp_dns"]
        );
    }

    return result;
}

////////////////////////////////////////////////////////////////////////////////

} // namespace NInfra::NYandexDnsExport
