#include "requests.h"

#include <infra/yp_yandex_dns_export/libs/sensors/sensors.h>

#include <infra/libs/sensors/macros.h>

#include <library/cpp/retry/retry.h>

#include <util/string/join.h>

namespace NInfra::NYandexDnsExport::NHttp {

namespace {

NNeh::TResponseRef Request(const NNeh::TMessage& message, const TDuration& executionTimeLimit, ui64 retries, const TSensorGroup* sensorGroup) {
    NNeh::TResponseRef response;

    auto onStart = [&sensorGroup] {
        if (!sensorGroup) {
            return;
        }
        NON_STATIC_INFRA_RATE_SENSOR(*sensorGroup, NSensors::HTTP_REQUEST_NUMBER);
    };

    auto onStop = [&sensorGroup](const NNeh::TResponseRef& response) {
        if (!sensorGroup) {
            return;
        }
        if (!response) {
            NON_STATIC_INFRA_RATE_SENSOR(*sensorGroup, NSensors::HTTP_REQUEST_TIMED_OUT);
            return;
        }
        if (response->IsError()) {
            NON_STATIC_INFRA_RATE_SENSOR(*sensorGroup, NSensors::HTTP_REQUEST_ERROR);
            NON_STATIC_INFRA_RATE_SENSOR(*sensorGroup, Join('.', NSensors::HTTP_REQUEST_ERROR, response->GetErrorCode()));
        }
        NON_STATIC_INFRA_INT_GAUGE_SENSOR(*sensorGroup, NSensors::HTTP_REQUEST_DURATION, response->Duration.MilliSeconds());
        NON_STATIC_INFRA_INT_GAUGE_SENSOR(*sensorGroup, NSensors::HTTP_REQUEST_RESPONSE_SIZE, response->Data.size());
    };

    auto onFail = [&sensorGroup] {
        if (!sensorGroup) {
            return;
        }
        NON_STATIC_INFRA_RATE_SENSOR(*sensorGroup, NSensors::HTTP_REQUEST_FAILED);
    };

    auto request = [&] {
        onStart();

        auto handler = NNeh::Request(message);
        NNeh::TResponseRef resp = handler->Wait(executionTimeLimit);

        onStop(resp);

        if (!resp) {
            ythrow yexception() << "Request timed out";
        } else if (resp->IsError()) {
            ythrow yexception() << "Error (code " << resp->GetErrorCode() << ") in request: [" << resp->GetErrorText() << "]";
        } else {
            response = resp;
        }
    };

    const TRetryOptions retryOptions = TRetryOptions().WithCount(retries).WithSleep(TDuration::Seconds(5));
    try {
        DoWithRetry(request, retryOptions, /* throwLast */ true);
    } catch (...) {
        onFail();
        throw;
    }

    if (sensorGroup) {
        NON_STATIC_INFRA_RATE_SENSOR(*sensorGroup, NSensors::HTTP_REQUEST_SUCCESS);
    }

    return response;
}

} // namespace

NNeh::TResponseRef Request(const NNeh::TMessage& message, const TDuration& executionTimeLimit, ui64 retries) {
    return Request(message, executionTimeLimit, retries, nullptr);
}

NNeh::TResponseRef Request(const NNeh::TMessage& message, const TDuration& executionTimeLimit, ui64 retries, const TSensorGroup& sensorGroup) {
    return Request(message, executionTimeLimit, retries, &sensorGroup);
}

} // namespace
