#include "service.h"

#include <infra/yp_yandex_dns_export/libs/router_api/router_api.h>
#include <infra/yp_yandex_dns_export/libs/sensors/sensors.h>
#include <infra/yp_yandex_dns_export/libs/zone_receivers/slayer_reverse/slayer_reverse_receiver.h>
#include <infra/yp_yandex_dns_export/libs/zone_retrievers/awacs/awacs_zone_retriever.h>
#include <infra/yp_yandex_dns_export/libs/zone_retrievers/axfr/axfr_retriever.h>
#include <infra/yp_yandex_dns_export/libs/zone_retrievers/qloud/qloud_zone_retriever.h>
#include <infra/yp_yandex_dns_export/libs/zones_list_retrievers/bridge/bridge_zones_list_retriever.h>

#include <infra/libs/logger/protos/events.ev.pb.h>
#include <infra/libs/service_iface/errors.h>
#include <infra/libs/sensors/sensor_group.h>
#include <infra/libs/sensors/macros.h>

#include <infra/contrib/pdns/power_dns/dns.hh>
#include <infra/contrib/pdns/power_dns/pdnsexception.hh>

#include <library/cpp/http/misc/httpcodes.h>
#include <library/cpp/json/json_writer.h>
#include <library/cpp/threading/future/async.h>

#include <contrib/libs/grpc/include/grpcpp/impl/codegen/status_code_enum.h>

#include <util/system/hostname.h>

namespace NInfra::NYandexDnsExport {

namespace {

THolder<NRetrievers::IZoneRetriever> GetZoneRetriever(const TSource& sourceConfig) {
    if (const TRetrieveZoneConfig& retrieveConfig = sourceConfig.GetRetrieve(); retrieveConfig.HasAxfr()) {
        return MakeHolder<NRetrievers::TAXFRRetriever>(sourceConfig);
    } else if (retrieveConfig.HasQloud()) {
        return MakeHolder<NRetrievers::TQloudZoneRetriever>(sourceConfig);
    } else if (retrieveConfig.HasAwacs()) {
        return MakeHolder<NRetrievers::TAwacsZoneRetriever>(sourceConfig);
    }
    return nullptr;
}

THolder<NRetrievers::IZonesListRetriever> GetZonesListRetriever(const TZonesListSource& sourceConfig) {
    if (sourceConfig.GetRetrieveConfig().HasBridge()) {
        return MakeHolder<NRetrievers::TBridgeZonesListRetriever>(sourceConfig);
    }
    return nullptr;
}

} // anonymous namespace

TService::TService(
    const THttpServiceConfig& cfg,
    TLogger& logger,
    NController::TControllerLoop& controller,
    const NReceivers::TReceivingZones& receivingZones,
    const NRetrievers::TRetrievingZones& retrievingZones,
    const NRetrievers::TRetrievingZonesLists& retrievingZonesLists
)
    : IService(cfg, logger, controller, CreateRouter(*this, receivingZones))
    , ReceivingZones_(receivingZones)
    , RetrievingZones_(retrievingZones)
    , RetrievingZonesLists_(retrievingZonesLists)
    , SensorGroup_(NSensors::SERVICE_GROUP_NAME)
{
    for (const auto& [sourceName, retrievingZone] : RetrievingZones_) {
        const TSource& sourceConfig = retrievingZone->SourceConfig();
        if (sourceConfig.HasRetrieve()) {
            THolder<NRetrievers::IZoneRetriever> retriever = GetZoneRetriever(sourceConfig);
            if (!retriever) {
                ythrow yexception() << "Failed to create zone retriever for source " << sourceConfig.GetName();
            }
            RetrieverThreads_.emplace_back(MakeHolder<NInfra::TBackgroundThread>(
                [this, &sourceName = sourceName, retrieverPtr = retriever.Get()] {
                    NThreading::Async([this, &sourceName, retrieverPtr] {
                        if (Controller_.LockAcquired()) {
                            RetrieveZone(sourceName, retrieverPtr);
                        }
                    }, RetrieversPool_).Wait();
                },
                TDuration::Seconds(5)));
            ZoneRetrievers_.emplace_back(std::move(retriever));
        }
    }

    for (const auto& [sourceName, retrievingZonesList] : RetrievingZonesLists_) {
        const TZonesListSource& sourceConfig = retrievingZonesList->SourceConfig();
        THolder<NRetrievers::IZonesListRetriever> retriever = GetZonesListRetriever(sourceConfig);
        if (!retriever) {
            ythrow yexception() << "Failed to create zones list retriever for source " << sourceConfig.GetName();
        }
        RetrieverThreads_.emplace_back(MakeHolder<NInfra::TBackgroundThread>(
            [this, &sourceName = sourceName, retrieverPtr = retriever.Get()] {
                NThreading::Async([this, &sourceName, retrieverPtr] {
                    if (Controller_.LockAcquired()) {
                        RetrieveZonesList(sourceName, retrieverPtr);
                    }
                }, RetrieversPool_).Wait();
            },
            TDuration::Seconds(5)));
        ZonesListRetrievers_.emplace_back(std::move(retriever));
    }

    InitSensors();

    RetrieversPool_.Start(2);
    for (auto& retrieverThread : RetrieverThreads_) {
        retrieverThread->Start();
    }
}

void TService::Ping(TRequestPtr<NController::TReqPing>, TReplyPtr<NController::TRspPing> reply) {
    NController::TRspPing result;
    result.SetData("pong");
    reply->Set(result);
    reply->SetAttribute("X-Server-Name", HostName());
}

void TService::CheckIsLeader(TRequestPtr<NApi::TReqCheckIsLeader>, TReplyPtr<NApi::TRspCheckIsLeader> reply) {
    if (!Controller_.LockAcquired()) {
        ythrow TServiceError{HTTP_I_AM_A_TEAPOT, grpc::ABORTED} << "I am not a leader";
    }

    NApi::TRspCheckIsLeader result;
    result.SetData("yes");

    reply->SetAttribute("X-Server-Name", HostName());
    reply->Set(result);
}

void TService::ReceiveZone(TRequestPtr<NJson::TJsonValue> request, TReplyPtr<NApi::TRspReceiveZone> reply) {
    TSensorGroup receiveSensorGroup(SensorGroup_, NSensors::RECEIVE_GROUP_NAME);
    NInfra::TLogFramePtr logFrame = Logger_.SpawnFrame();

    auto receivingZone = ReceivingZones_.at(request->Path());
    Y_ENSURE(receivingZone->ReceiveConfig().HasSlayerReverse(), "Only slayer reverse receiver is supported (path: " << request->Path() << ")");

    logFrame->LogEvent(NLogEvent::TStartReceiveZone(request->Path(), TString{receivingZone->SourceName()}));

    receiveSensorGroup.AddLabel("source", receivingZone->SourceName());
    TDurationSensor(receiveSensorGroup, NSensors::RECEIVE_DURATION).Start();
    NON_STATIC_INFRA_RATE_SENSOR(receiveSensorGroup, NSensors::RECEIVE_NUMBER);

    if (logFrame->AcceptLevel(ELogPriority::TLOG_DEBUG)) {
        TString attributes;
        for (const auto& [key, value] : request->Attributes()) {
            attributes += TString::Join(key, "=", value, ";");
        }
        logFrame->LogEvent(ELogPriority::TLOG_DEBUG, NLogEvent::TReceiveZoneRequestData(NJson::WriteJson(request->Get(), /* format */ false), attributes));
    }

    NReceivers::TSlayerReverseReceiver receiver(receivingZone->ReceiveConfig().GetSlayerReverse(), logFrame, receiveSensorGroup);
    TVector<DNSResourceRecord> records;
    try {
        records = receiver.Receive(request->Get());
    } catch (...) {
        logFrame->LogEvent(ELogPriority::TLOG_ERR, NLogEvent::TReceivedRequestParseError(CurrentExceptionMessage()));
        NON_STATIC_INFRA_RATE_SENSOR(receiveSensorGroup, NSensors::RECEIVE_FAILED);
        throw;
    }
    logFrame->LogEvent(NLogEvent::TReceiveZoneResult(records.size()));

    receivingZone->SetRecords(std::move(records));

    NApi::TRspReceiveZone response;
    response.SetData("ok");

    reply->Set(response);
    reply->SetAttribute("X-Server-Name", HostName());

    NON_STATIC_INFRA_RATE_SENSOR(receiveSensorGroup, NSensors::RECEIVE_SUCCESS);
    TDurationSensor(receiveSensorGroup, NSensors::RECEIVE_DURATION).Update();
}

void TService::InitSensors() {
    TSensorGroup receiveSensorGroup(SensorGroup_, NSensors::RECEIVE_GROUP_NAME);
    for (const auto& [sourceName, receivingZone] : ReceivingZones_) {
        TSensorGroup sensorGroup = receiveSensorGroup;
        sensorGroup.AddLabel("source", receivingZone->SourceName());
        TRateSensor(sensorGroup, NSensors::RECEIVE_NUMBER);
        TRateSensor(sensorGroup, NSensors::RECEIVE_SUCCESS);
        TRateSensor(sensorGroup, NSensors::RECEIVE_FAILED);
    }

    TSensorGroup retrieveSensorGroup(SensorGroup_, NSensors::RETRIEVE_GROUP_NAME);
    for (const auto& [sourceName, retrievingZone] : RetrievingZones_) {
        TSensorGroup sensorGroup = retrieveSensorGroup;
        sensorGroup.AddLabel("source", retrievingZone->SourceName());
        TRateSensor(sensorGroup, NSensors::RETRIEVE_NUMBER);
        TRateSensor(sensorGroup, NSensors::RETRIEVE_SUCCESS);
        TRateSensor(sensorGroup, NSensors::RETRIEVE_FAILED);
    }

    TSensorGroup retrieveZonesSensorGroup(SensorGroup_, NSensors::RETRIEVE_ZONES_GROUP_NAME);
    for (const auto& [sourceName, retrievingZonesList] : RetrievingZonesLists_) {
        TSensorGroup sensorGroup = retrieveZonesSensorGroup;
        sensorGroup.AddLabel("source", retrievingZonesList->SourceName());
        TRateSensor(sensorGroup, NSensors::RETRIEVE_NUMBER);
        TRateSensor(sensorGroup, NSensors::RETRIEVE_SUCCESS);
        TRateSensor(sensorGroup, NSensors::RETRIEVE_FAILED);
    }
}

void TService::RetrieveZone(const TString& sourceName, const NRetrievers::IZoneRetriever* retriever) const {
    TSensorGroup retrieveSensorGroup(SensorGroup_, NSensors::RETRIEVE_GROUP_NAME);
    retrieveSensorGroup.AddLabel("source", sourceName);
    NInfra::TLogFramePtr logFrame = Logger_.SpawnFrame();

    TDurationSensor(retrieveSensorGroup, NSensors::RETRIEVE_DURATION).Start();
    NON_STATIC_INFRA_RATE_SENSOR(retrieveSensorGroup, NSensors::RETRIEVE_NUMBER);

    logFrame->LogEvent(NLogEvent::TStartRetrieveZone(sourceName));

    TVector<DNSResourceRecord> records;
    try {
        records = retriever->Retrieve(logFrame, retrieveSensorGroup);
    } catch (const PDNSException& e) {
        logFrame->LogEvent(ELogPriority::TLOG_ERR, NLogEvent::TRetrieveZoneFailure(sourceName, TStringBuilder() << "Source \"" << sourceName << "\" failed to retrieve records [" << e.reason << "]"));
        NON_STATIC_INFRA_RATE_SENSOR(retrieveSensorGroup, NSensors::RETRIEVE_FAILED);
        TDurationSensor(retrieveSensorGroup, NSensors::RETRIEVE_DURATION).Update();
        throw;
    } catch (...) {
        logFrame->LogEvent(ELogPriority::TLOG_ERR, NLogEvent::TRetrieveZoneFailure(sourceName, TStringBuilder() << "Source \"" << sourceName << "\" failed to retrieve records [" << CurrentExceptionMessage() << "]"));
        NON_STATIC_INFRA_RATE_SENSOR(retrieveSensorGroup, NSensors::RETRIEVE_FAILED);
        TDurationSensor(retrieveSensorGroup, NSensors::RETRIEVE_DURATION).Update();
        throw;
    }

    logFrame->LogEvent(NLogEvent::TRetrieveZoneResult(records.size(), sourceName));

    RetrievingZones_.at(sourceName)->SetRecords(std::move(records));

    NON_STATIC_INFRA_RATE_SENSOR(retrieveSensorGroup, NSensors::RETRIEVE_SUCCESS);
    TDurationSensor(retrieveSensorGroup, NSensors::RETRIEVE_DURATION).Update();
}

void TService::RetrieveZonesList(
    const TString& sourceName,
    const NRetrievers::IZonesListRetriever* retriever
) const {
    TSensorGroup retrieveZonesSensorGroup(SensorGroup_, NSensors::RETRIEVE_ZONES_GROUP_NAME);
    retrieveZonesSensorGroup.AddLabel("source", sourceName);
    NInfra::TLogFramePtr logFrame = Logger_.SpawnFrame();

    TDurationSensor(retrieveZonesSensorGroup, NSensors::RETRIEVE_DURATION).Start();
    NON_STATIC_INFRA_RATE_SENSOR(retrieveZonesSensorGroup, NSensors::RETRIEVE_NUMBER);

    logFrame->LogEvent(NLogEvent::TStartRetrieveZonesList(sourceName));

    TVector<NYP::NClient::TDnsZone> zones;
    try {
        zones = retriever->Retrieve(logFrame, retrieveZonesSensorGroup);
    } catch (const PDNSException& e) {
        logFrame->LogEvent(ELogPriority::TLOG_ERR,
            NLogEvent::TRetrieveZonesListFailure(sourceName, TStringBuilder() << "Source \"" << sourceName << "\" failed to retrieve zones list [" << e.reason << "]"));
        NON_STATIC_INFRA_RATE_SENSOR(retrieveZonesSensorGroup, NSensors::RETRIEVE_FAILED);
        TDurationSensor(retrieveZonesSensorGroup, NSensors::RETRIEVE_DURATION).Update();
        throw;
    } catch (...) {
        logFrame->LogEvent(ELogPriority::TLOG_ERR,
            NLogEvent::TRetrieveZonesListFailure(sourceName, TStringBuilder() << "Source \"" << sourceName << "\" failed to retrieve zones list [" << CurrentExceptionMessage() << "]"));
        NON_STATIC_INFRA_RATE_SENSOR(retrieveZonesSensorGroup, NSensors::RETRIEVE_FAILED);
        TDurationSensor(retrieveZonesSensorGroup, NSensors::RETRIEVE_DURATION).Update();
        throw;
    }

    logFrame->LogEvent(NLogEvent::TRetrieveZonesListResult(sourceName, zones.size()));

    RetrievingZonesLists_.at(sourceName)->SetZonesList(std::move(zones));

    NON_STATIC_INFRA_RATE_SENSOR(retrieveZonesSensorGroup, NSensors::RETRIEVE_SUCCESS);
    TDurationSensor(retrieveZonesSensorGroup, NSensors::RETRIEVE_DURATION).Update();
}

} // namespace NInfra::NYandexDnsExport
