#pragma once

#include <infra/yp_yandex_dns_export/libs/config/config.pb.h>
#include <infra/yp_yandex_dns_export/libs/controller/controller.h>
#include <infra/yp_yandex_dns_export/libs/zone_receivers/receiving_zone.h>
#include <infra/yp_yandex_dns_export/libs/zone_retrievers/awacs/awacs_zone_retriever.h>
#include <infra/yp_yandex_dns_export/libs/zone_retrievers/axfr/axfr_retriever.h>
#include <infra/yp_yandex_dns_export/libs/zone_retrievers/qloud/qloud_zone_retriever.h>
#include <infra/yp_yandex_dns_export/libs/zone_retrievers/retrieving_zone.h>

#include <infra/libs/controller/object_manager/object_manager.h>
#include <infra/libs/controller/standalone_controller/standalone_controller.h>
#include <infra/libs/sensors/sensor.h>

#include <library/cpp/proto_config/config.h>

namespace NInfra::NYandexDnsExport {

TVector<NController::TSingleClusterObjectManagersFactoryPtr> CreateFactories(const TConfig& config, const NRetrievers::TRetrievingZones& retrievingZones) {
    NInfra::NController::TSharding shardsFactory(config.GetController().GetLeadingInvader());

    TVector<NController::TSingleClusterObjectManagersFactoryPtr> objectFactories;
    for (const TString& ypAddress : config.GetYpAddresses()) {
        NController::TClientConfig ypClientConfig = config.GetController().GetYpClient();
        ypClientConfig.SetAddress(ypAddress);

        objectFactories.push_back(new TZoneRecordsExportManagerFactory(
            config.GetExportConfig()
            , NReceivers::InitReceivingZones(config.GetExportConfig())
            , retrievingZones
            , std::move(ypClientConfig)
            , shardsFactory.GetShard(0)));
    }
    return objectFactories;
}

class TStandaloneController: public NInfra::NController::TStandaloneController {
public:
    TStandaloneController(const TConfig config, const NRetrievers::TRetrievingZones& retrievingZones)
        : NInfra::NController::TStandaloneController(
            config.GetController(),
            CreateFactories(config, retrievingZones)
        )
        , Config_(config)
        , RetrievingZones_(retrievingZones)
    {
    }

    void Sync() {
        NInfra::TLogFramePtr logFrame = Logger_.SpawnFrame();
        RetrieveZones(logFrame);
        NInfra::NController::TStandaloneController::Sync();
    }

    bool SafeSync() {
        try {
            NInfra::TLogFramePtr logFrame = Logger_.SpawnFrame();
            RetrieveZones(logFrame);
            return NInfra::NController::TStandaloneController::SafeSync();
        } catch (...) {
            return false;
        }
        Y_UNREACHABLE();
    }

private:
    THolder<NRetrievers::IZoneRetriever> GetZoneRetriever(const TSource& sourceConfig) {
        if (const TRetrieveZoneConfig& retrieveConfig = sourceConfig.GetRetrieve(); retrieveConfig.HasAxfr()) {
            return MakeHolder<NRetrievers::TAXFRRetriever>(sourceConfig);
        } else if (retrieveConfig.HasQloud()) {
            return MakeHolder<NRetrievers::TQloudZoneRetriever>(sourceConfig);
        } else if (retrieveConfig.HasAwacs()) {
            return MakeHolder<NRetrievers::TAwacsZoneRetriever>(sourceConfig);
        }
        return nullptr;
    }

    void RetrieveZones(NInfra::TLogFramePtr logFrame) {
        for (const auto& [sourceName, retrievingZone] : RetrievingZones_) {
            const TSource& sourceConfig = retrievingZone->SourceConfig();
            if (sourceConfig.HasRetrieve()) {
                THolder<NRetrievers::IZoneRetriever> retriever = GetZoneRetriever(sourceConfig);
                if (!retriever) {
                    ythrow yexception() << "Failed to create zone retriever for source " << sourceConfig.GetName();
                }
                retrievingZone->SetRecords(retriever->Retrieve(logFrame, NInfra::TSensorGroup{sourceName}));
            }
        }
    }

private:
    const TConfig Config_;
    const NRetrievers::TRetrievingZones RetrievingZones_;
};

class TStandaloneControllerWrapper {
public:
    TStandaloneControllerWrapper(const TStringBuf config)
        : Config_(NProtoConfig::ParseConfigFromJson<TConfig>(config))
        , RetrievingZones_(NRetrievers::InitRetrievingZones(Config_.GetExportConfig()))
        , Controller_(Config_, RetrievingZones_)
    {
    }

    void Sync() {
        return Controller_.Sync();
    }

    bool SafeSync() {
        return Controller_.SafeSync();
    }

private:
    const TConfig Config_;
    const NRetrievers::TRetrievingZones RetrievingZones_;
    TStandaloneController Controller_;
};

} // namespace NInfra::NYandexDnsExport
