#include "dns.h"

#include <util/generic/adaptor.h>
#include <util/draft/ip.h>
#include <util/generic/yexception.h>
#include <util/string/ascii.h>
#include <util/string/builder.h>
#include <util/string/hex.h>
#include <util/string/join.h>
#include <util/string/split.h>

namespace NInfra::NYandexDnsExport {

DNSName BuildIp4PtrDnsAddress(const TStringBuf ip) {
    TVector<TStringBuf> parts = StringSplitter(ip).Split('.').SkipEmpty();
    Y_ASSERT(parts.size() == 4);
    return DNSName(TStringBuilder() << JoinRange(".", parts.rbegin(), parts.rend()) << ".in-addr.arpa");
}

DNSName BuildIp6PtrDnsAddress(const TStringBuf ip) {
    TIp6 address = Ip6FromString(ip.data());
    TString octets = HexEncode(address.Data, sizeof(address.Data));

    TString result;
    result.reserve(octets.size() * 2 + TStringBuf("ip6.arpa").size());
    for (char c : Reversed(octets)) {
        result += AsciiToLower(c);
        result += ".";
    }
    result += "ip6.arpa";

    return DNSName(result);
}

DNSName BuildPtrDnsAddress(const TString& ip, QType forwardQtype) {
    switch (forwardQtype.getCode()) {
        case QType::A:
            return BuildIp4PtrDnsAddress(ip);
        case QType::AAAA:
            return BuildIp6PtrDnsAddress(ip);
        default:
            ythrow yexception() << "Invalid query type \"" << forwardQtype.getName() << "\"";
    }
}

DNSResourceRecord CreateRecord(const DNSName& qname, const TString& content, QType qtype, ui32 ttl) {
    DNSResourceRecord record;
    record.qname = DNSName(qname);
    record.content = content;
    record.qtype = qtype;
    record.ttl = ttl;
    return record;
}

DNSResourceRecord CreateReverseRecord(const DNSName& qname, const TString& ip, QType qtype, ui32 ttl) {
    DNSResourceRecord record;
    record.qname = BuildPtrDnsAddress(ip, qtype);
    record.content = qname.toString();
    record.qtype = QType::PTR;
    record.ttl = ttl;
    return record;
}

TString MakeSrvContent(ui64 priority, ui64 weight, ui64 port, TStringBuf target) {
    return Join(' ', priority, weight, port, target);
}

TString MakeSrvContent(ui64 priority, ui64 weight, ui64 port, const DNSName& target) {
    return MakeSrvContent(priority, weight, port, target.toStringRootDot());
}

} // namespace NInfra::NYandexDnsExport
