#include "slayer_reverse_receiver.h"

#include <infra/yp_yandex_dns_export/libs/sensors/sensors.h>

#include <infra/libs/sensors/macros.h>

#include <infra/contrib/pdns/power_dns/qtype.hh>

namespace NInfra::NYandexDnsExport::NReceivers {

namespace {

static const TMap<TString, uint64_t> QTYPE_BY_NAME = {{
    {"IN", QClass::IN},
    {"CHAOS", QClass::CHAOS},
    {"NONE", QClass::NONE},
    {"ANY", QClass::ANY},
}};

uint64_t GetQClass(const TString& qclass) {
    if (auto* qtype = QTYPE_BY_NAME.FindPtr(qclass)) {
        return *qtype;
    }
    ythrow yexception() << "Unknown class " << qclass;
}

DNSResourceRecord CreateRecord(const NJson::TJsonValue& record) {
    DNSResourceRecord result;
    result.qname = DNSName(record["name"].GetStringSafe());
    result.content = record["rdata"].GetStringSafe();
    result.ttl = record["ttl"].GetIntegerSafe();
    result.qtype = QType::chartocode(record["type"].GetStringSafe().data());
    result.qclass = GetQClass(record["class"].GetStringSafe());
    return result;
}

} // anonymous namespace

class TSlayerReverseReceiver::TImpl {
public:
    TImpl(TSlayerReverseConfig config, TLogFramePtr logFrame, TSensorGroup sensorGroup)
        : LogFramePtr_(logFrame)
        , SensorGroup_(std::move(sensorGroup))
        , Config_(std::move(config))
    {
    }

    TVector<DNSResourceRecord> Receive(const NJson::TJsonValue& records) const {
        if (!records.IsArray()) {
            const TString errorMessage = TStringBuilder() << "Incorrect zone format: expected Array, found " << records.GetType();
            LogFramePtr_->LogEvent(ELogPriority::TLOG_ERR, NLogEvent::TReceivedRequestParseError(errorMessage));
            NON_STATIC_INFRA_RATE_SENSOR(SensorGroup_, NSensors::RECEIVE_PARSE_ERROR);
            ythrow yexception() << errorMessage;
        }

        TVector<DNSResourceRecord> result;
        result.reserve(records.GetArray().size());

        const DNSName zoneName(Config_.GetZone());
        for (const NJson::TJsonValue& record : records.GetArray()) {
            DNSResourceRecord resourceRecord;
            try {
                resourceRecord = CreateRecord(record);
            } catch (...) {
                LogFramePtr_->LogEvent(ELogPriority::TLOG_ERR, NLogEvent::TReceivedRequestParseError(CurrentExceptionMessage()));
                NON_STATIC_INFRA_RATE_SENSOR(SensorGroup_, NSensors::RECEIVE_PARSE_ERROR);
                continue;
            }
            if (!resourceRecord.qname.isPartOf(zoneName)) {
                LogFramePtr_->LogEvent(ELogPriority::TLOG_ERR, NLogEvent::TRecordZoneMismatch(resourceRecord.qname.toStringNoDot().data(), zoneName.toStringNoDot().data()));
                NON_STATIC_INFRA_RATE_SENSOR(SensorGroup_, NSensors::RECORD_ZONE_MISMATCH);
                continue;
            }
            result.push_back(std::move(resourceRecord));
        }

        return result;
    }

private:
    TLogFramePtr LogFramePtr_;
    const TSensorGroup SensorGroup_;
    const TSlayerReverseConfig Config_;
};

TSlayerReverseReceiver::TSlayerReverseReceiver(TSlayerReverseConfig config, TLogFramePtr logFrame, TSensorGroup sensorGroup)
    : Impl_(MakeHolder<TImpl>(std::move(config), logFrame, std::move(sensorGroup)))
{
}

TVector<DNSResourceRecord> TSlayerReverseReceiver::Receive(const NJson::TJsonValue& records) const {
    return Impl_->Receive(records);
}

TSlayerReverseReceiver::~TSlayerReverseReceiver() = default;


} // namespace NInfra::NYandexDnsExport::NReceivers
