#include "awacs_zone_retriever.h"

#include <infra/yp_yandex_dns_export/libs/http/requests.h>
#include <infra/yp_yandex_dns_export/libs/sensors/sensors.h>
#include <infra/yp_yandex_dns_export/libs/util/dns.h>

#include <infra/awacs/proto/api.pb.h>

#include <infra/libs/sensors/sensor.h>

#include <library/cpp/neh/http_common.h>
#include <library/cpp/neh/http_headers.h>

#include <contrib/libs/grpc/include/grpc++/create_channel.h>

#include <util/system/env.h>

namespace NInfra::NYandexDnsExport::NRetrievers {

namespace {

awacs::api::GetNameServerConfigResponse GetNameServerConfig(const NNeh::TMessage& request, const TDuration timeout, ui64 retries, const NInfra::TSensorGroup& sensorGroup) {
    NNeh::TResponseRef resp = NHttp::Request(request, timeout, retries, NInfra::TSensorGroup(sensorGroup, NSensors::HTTP_REQUEST));
    awacs::api::GetNameServerConfigResponse response;
    response.ParseFromStringOrThrow(resp->Data);
    return response;
}

} // namespace

class TAwacsZoneRetriever::TImpl {
public:
    TImpl(const TSource& config)
        : SourceName_(config.GetName())
        , Config_(config.GetRetrieve().GetAwacs())
        , RequestTimeout_(TDuration::Parse(Config_.GetTimeout()))
        , RequestRetries_(Config_.GetRequestRetries())
        , Labels_({{"source", SourceName_}})
    {
        awacs::api::GetNameServerConfigRequest request;
        request.set_id(Config_.GetId());
        request.set_namespace_id(Config_.GetNamespaceId());

        RequestMessage_.Addr = Config_.GetApiUrl();
        const TString headers = TStringBuilder() << "Authorization: OAuth " << GetEnv("AWACS_TOKEN") << "\n"
                                                 << "Accept: application/x-protobuf\n";

        Y_ENSURE(NNeh::NHttp::MakeFullRequest(RequestMessage_, headers, request.SerializeAsString(), "application/x-protobuf", NNeh::NHttp::ERequestType::Post));
    }

    TVector<DNSResourceRecord> Retrieve(TLogFramePtr logFrame, TSensorGroup sensorGroup) const {
        sensorGroup.AddLabels(Labels_);

        logFrame->LogEvent(ELogPriority::TLOG_INFO, NLogEvent::TStartRetrieveAwacsZone(SourceName_));
        awacs::api::GetNameServerConfigResponse nameserverConfig = GetNameServerConfig(RequestMessage_, RequestTimeout_, RequestRetries_, sensorGroup);

        TVector<DNSResourceRecord> records;
        records.reserve(nameserverConfig.records().size());

        const DNSName zone(nameserverConfig.zone().data());
        for (const awacs::api::GetNameServerConfigResponse_Record& record : nameserverConfig.records()) {
            if (record.type() == awacs::api::GetNameServerConfigResponse_Record_Type_ADDRESS) {
                const DNSName fqdn = DNSName(record.address().zone()) + zone;
                for (const awacs::api::GetNameServerConfigResponse_Record_Ipv6Addr& address : record.address().ipv6_addrs()) {
                    records.push_back(CreateRecord(fqdn, address.value(), QType(QType::AAAA)));
                }
            }
        }

        logFrame->LogEvent(ELogPriority::TLOG_INFO, NLogEvent::TRetrieveAwacsZoneResult(SourceName_, records.size()));

        return records;
    }

private:
    const TString SourceName_;
    const TAwacsImportConfig Config_;
    NNeh::TMessage RequestMessage_;
    const TDuration RequestTimeout_;
    const ui32 RequestRetries_;
    const TVector<std::pair<TStringBuf, TStringBuf>> Labels_;
};

TAwacsZoneRetriever::TAwacsZoneRetriever(const TSource& sourceConfig)
    : Impl_(MakeHolder<TImpl>(sourceConfig))
{
}

TVector<DNSResourceRecord> TAwacsZoneRetriever::Retrieve(TLogFramePtr logFrame, TSensorGroup sensorGroup) const {
    return Impl_->Retrieve(logFrame, std::move(sensorGroup));
}

TAwacsZoneRetriever::~TAwacsZoneRetriever() {
}

} // namespace NInfra::NYandexDnsExport::NRetrievers
