#include "qloud_zone_retriever.h"

#include <infra/yp_yandex_dns_export/libs/http/requests.h>
#include <infra/yp_yandex_dns_export/libs/sensors/sensors.h>
#include <infra/yp_yandex_dns_export/libs/util/dns.h>

#include <infra/libs/logger/protos/events.ev.pb.h>
#include <infra/libs/sensors/macros.h>

#include <library/cpp/json/json_reader.h>
#include <library/cpp/json/json_value.h>
#include <library/cpp/neh/neh.h>
#include <library/cpp/retry/retry.h>

#include <util/string/builder.h>
#include <util/string/cast.h>

namespace NInfra::NYandexDnsExport::NRetrievers {

namespace {

static const TMap<TString, QType> QTYPE_BY_FIELDNAME = {{
    {"ip4", QType(QType::A)},
    {"ip6", QType(QType::AAAA)},
}};

constexpr ui32 SRV_RECORD_TTL = 15;

void CreateRecordsFromIps(const NJson::TJsonValue& recordSets, TVector<DNSResourceRecord>& records) {
    records.reserve(records.size() + 2 * recordSets.GetArray().size());

    for (const NJson::TJsonValue& recordSet : recordSets.GetArray()) {
        const DNSName host(recordSet["host"].GetString());
        if (host.empty()) {
            continue;
        }

        for (const auto& [fieldName, qtype] : QTYPE_BY_FIELDNAME) {
            if (const TString& ip = recordSet[fieldName].GetString(); !ip.empty()) {
                records.push_back(CreateRecord(host, ip, qtype));
                records.push_back(CreateReverseRecord(host, ip, qtype));
            }
        }
    }
}

void CreateRecordsFromEnvironments(const NJson::TJsonValue& recordSets, TVector<DNSResourceRecord>& records) {
    records.reserve(records.size() + 4 * recordSets.GetArray().size());

    for (const NJson::TJsonValue& recordSet : recordSets.GetArray()) {
        TVector<DNSName> domains;

        domains.emplace_back(recordSet["name"].GetString());
        for (const NJson::TJsonValue& component : recordSet["components"].GetArray()) {
            domains.emplace_back(component["name"].GetString());
            for (const NJson::TJsonValue& instance : component["instances"].GetArray()) {
                const DNSName& instanceName = domains.emplace_back(instance["name"].GetString());
                const DNSName& instanceFqdn = domains.emplace_back(instance["fqdn"].GetString());

                for (const DNSName& domain : domains) {
                    if (domain == instanceFqdn) {
                        continue;
                    }

                    records.push_back(CreateRecord(DNSName("_host_") + domain, MakeSrvContent(0, 0, 0, instanceFqdn), QType(QType::SRV), SRV_RECORD_TTL));
                    records.push_back(CreateRecord(DNSName("_id_") + domain, MakeSrvContent(0, 0, 0, instanceName), QType(QType::SRV), SRV_RECORD_TTL));
                }

                for (const auto& [fieldName, qtype] : QTYPE_BY_FIELDNAME) {
                    if (const TString& ip = instance[fieldName].GetString(); !ip.empty()) {
                        for (const DNSName& domain : domains) {
                            records.push_back(CreateRecord(domain, ip, qtype));
                            if (domain != instanceFqdn) {
                                records.push_back(CreateRecord(DNSName("_ip_") + domain, MakeSrvContent(0, 0, 0, ip), QType(QType::SRV), SRV_RECORD_TTL));
                            }
                        }
                        records.push_back(CreateReverseRecord(instanceFqdn, ip, qtype));
                    }
                }

                domains.pop_back();
                domains.pop_back();
            }
            domains.pop_back();
        }
    }
}

} // namespace

class TQloudZoneRetriever::TImpl {
public:
    TImpl(const TSource& config)
        : SourceName_(config.GetName())
        , Config_(config.GetRetrieve().GetQloud())
        , RetrieveMessage_(TStringBuilder() << Config_.GetApiUrl() << Config_.GetHandler(), "")
        , Labels_({{"source", SourceName_}})
    {
    }

    TVector<DNSResourceRecord> Retrieve(TLogFramePtr logFrame, TSensorGroup sensorGroup) const {
        sensorGroup.AddLabels(Labels_);

        logFrame->LogEvent(ELogPriority::TLOG_INFO, NLogEvent::TStartRetrieveQloudZone(SourceName_));
        logFrame->LogEvent(ELogPriority::TLOG_INFO, NLogEvent::TStartQloudApiRequest(
            RetrieveMessage_.Addr, RetrieveMessage_.Data, Config_.GetRequestTimeout(), Config_.GetRequestRetries()
        ));

        NNeh::TResponseRef response = NHttp::Request(RetrieveMessage_, TDuration::Parse(Config_.GetRequestTimeout()), Config_.GetRequestRetries(), TSensorGroup(sensorGroup, NSensors::HTTP_REQUEST));

        logFrame->LogEvent(ELogPriority::TLOG_INFO, NLogEvent::TStopQloudApiRequest(response->Data.size(), response->Duration.ToString()));

        NJson::TJsonValue recordSets;
        try {
            NJson::ReadJsonTree(response->Data, &recordSets, /* throwOnError */ true);
        } catch (...) {
            NON_STATIC_INFRA_RATE_SENSOR(sensorGroup, NSensors::RESPONSE_PARSE_ERROR);
            throw;
        }

        TVector<DNSResourceRecord> records;
        switch (Config_.GetResponseFormat()) {
            case TQloudImportConfig::IPS:
                CreateRecordsFromIps(recordSets, records);
                break;
            case TQloudImportConfig::ENVIRONMENTS:
                CreateRecordsFromEnvironments(recordSets, records);
                break;
            default:
                Y_UNREACHABLE();
        }

        logFrame->LogEvent(ELogPriority::TLOG_INFO, NLogEvent::TRetrieveQloudZoneResult(SourceName_, records.size()));

        return records;
    }

private:
    const TString SourceName_;
    const TQloudImportConfig Config_;
    const NNeh::TMessage RetrieveMessage_;
    const TVector<std::pair<TStringBuf, TStringBuf>> Labels_;
};

TQloudZoneRetriever::TQloudZoneRetriever(const TSource& sourceConfig)
    : Impl_(MakeHolder<TImpl>(sourceConfig))
{
}

TVector<DNSResourceRecord> TQloudZoneRetriever::Retrieve(TLogFramePtr logFrame, TSensorGroup sensorGroup) const {
    return Impl_->Retrieve(logFrame, std::move(sensorGroup));
}

TQloudZoneRetriever::~TQloudZoneRetriever() {
}

} // namespace NInfra::NYandexDnsExport::NRetrievers
