import argparse
import json
import juggler_sdk
import logging
import os
import sys

from infra.yp_service_discovery.monitoring.solomon.src.util.solomon_api import SolomonApi


JUGGLER_HOST = 'yp-yandex-dns-export'
JUGGLER_NAMESPACE = 'yp.dns_export'

SOLOMON_PROJECT_ID = 'yp_yandex_dns_export'

DEBUG = 'debug'
WARN = 'warn'
CRIT = 'crit'

LOGINS = {
    DEBUG: [
    ],
    WARN: [
        '@svc_yp_dns:yp-dns-duty',
        'dns-monitoring',
    ],
    CRIT: [
        '@svc_yp_dns:yp-dns-duty',
        'dns-monitoring',
    ],
}

PHONE_ESCALATION_LOGINS = {
    DEBUG: [
    ],
    WARN: [
    ],
    CRIT: [
        '@svc_yp_dns:yp-dns-duty',
        'elshiko',
        'avitella',
    ],
}

MASTERS = [
    'man',
    'vla',
]

YP_MASTER_ADDRESSES = {
    master: '{}.yp.yandex-team.ru:8090'.format(master)
    for master in MASTERS
}


def milliseconds(seconds=0, minutes=0):
    return seconds * 1000 + minutes * 60 * 1000


def create_selectors(**kwargs):
    def create_selector(key, value):
        if value.startswith('!'):
            return '{}!="{}"'.format(key, value[1:])
        return '{}="{}"'.format(key, value)

    selectors = ', '.join(map(lambda kv: create_selector(*kv), kwargs.items()))
    return '{' + selectors + '}'


def notification(status, logins, notify_methods):
    template_kwargs = {
        "status": status,
        "login": logins,
        "method": notify_methods,
    }

    if (isinstance(status, str) and status == "CRIT") or (isinstance(status, dict) and status["to"] == "CRIT"):
        template_kwargs["repeat"] = 1800  # 30 min

    return juggler_sdk.NotificationOptions(
        template_name="on_status_change",
        template_kwargs=template_kwargs,
    )


def get_notifications_config(level=CRIT):
    logins = LOGINS[level]
    escalation_logins = PHONE_ESCALATION_LOGINS[level]

    result = []
    if logins:
        result.extend([
            notification(
                status={'from': 'OK', 'to': 'CRIT'},
                logins=logins,
                notify_methods=[
                    'sms',
                    'telegram',
                ]
            ),
            notification(
                status={'from': 'WARN', 'to': 'CRIT'},
                logins=logins,
                notify_methods=[
                    'sms',
                    'telegram',
                ]
            ),
            notification(
                status={'from': 'CRIT', 'to': 'OK'},
                logins=logins,
                notify_methods=[
                    'telegram',
                ]
            ),
        ])

    if escalation_logins:
        result.extend([
            juggler_sdk.NotificationOptions(
                template_name='phone_escalation',
                template_kwargs={
                    'delay': 60,
                    'logins': escalation_logins,
                }
            ),
        ])

    return result


def get_juggler_service(service_id):
    return '.'.join(['yp', 'dns_export', service_id])


def get_yp_downtime_tags(yp_cluster):
    return [
        "yp-downtime-9b8ee2a2b559b3afff751623e5e79546",
        "yp-{}-downtime-9b8ee2a2b559b3afff751623e5e79546".format(yp_cluster),
    ]


def get_solomon_lag_checks():
    return ['solomon:{}'.format(get_juggler_service('solomon_lag'))]


def get_yp_unreach_checks(yp_cluster):
    return ['yp-{}.yp.yandex.net:yp.infra.auto_downtime_'.format(yp_cluster)]


def get_yp_standard_unreach_checks(yp_cluster):
    return get_yp_unreach_checks(yp_cluster) + get_solomon_lag_checks()


def get_juggler_host(solomon_cluster):
    return '{}{}'.format(JUGGLER_HOST, solomon_cluster.get('host_suffix', ''))


def get_solomon_service_id(service_id, solomon_cluster):
    return '{}{}'.format(service_id, solomon_cluster.get('solomon_service_suffix', ''))


def get_name(name, solomon_cluster):
    return '{}{}'.format(name, solomon_cluster.get('name_suffix', ''))


def get_solomon_alert_url(alert_id):
    return {
        'url': 'https://solomon.yandex-team.ru/admin/projects/{}/alerts/{}'.format(SOLOMON_PROJECT_ID, alert_id),
        'title': 'Alert on solomon',
    }


def get_nanny_service_url():
    return {
        'url': 'https://nanny.yandex-team.ru/ui/#/services/catalog/yp-yandex-dns-export',
        'title': 'Nanny service',
        'type': 'nanny',
    }


def get_docs_url():
    return {
        'url': 'https://wiki.yandex-team.ru/ypdns/monitoring/#opisaniealertov',
        'title': 'Описание алертов',
        'type': 'wiki',
    }


def get_source_url():
    return {
        'url': 'https://a.yandex-team.ru/arc/trunk/arcadia/infra/yp_yandex_dns_export/monitoring/alerts',
        'title': 'Alerts generation source code',
    }


def get_standard_urls(alert_id):
    return [
        get_solomon_alert_url(alert_id),
        get_nanny_service_url(),
        get_docs_url(),
        get_source_url(),
    ]


def get_juggler_check_with_yp_downtime(host, service, yp_cluster, notification_level=CRIT,
                                       flaps_config=juggler_sdk.FlapOptions(stable=180, critical=300, boost=0),
                                       meta={}, tags=[]):
    return juggler_sdk.Check(
        host=host,
        service=service,
        namespace=JUGGLER_NAMESPACE,
        tags=get_yp_downtime_tags(yp_cluster) + tags,
        aggregator='logic_or',
        aggregator_kwargs={
            'unreach_checks': get_yp_standard_unreach_checks(yp_cluster),
        },
        notifications=get_notifications_config(notification_level),
        flaps_config=flaps_config,
        meta=meta,
    )


def get_juggler_check_downtime(host, service, notification_level=CRIT,
                               flaps_config=juggler_sdk.FlapOptions(stable=180, critical=300, boost=0),
                               meta={}, tags=[]):
    return juggler_sdk.Check(
        host=host,
        service=service,
        namespace=JUGGLER_NAMESPACE,
        tags=tags,
        aggregator='logic_or',
        aggregator_kwargs={
            'unreach_checks': get_solomon_lag_checks(),
        },
        notifications=get_notifications_config(notification_level),
        flaps_config=flaps_config,
        meta=meta,
    )


def get_no_updates_alert(solomon_cluster, master):
    host = get_juggler_host(solomon_cluster)
    service_id = 'no-updates-for-yp-{}'.format(master)

    sensor_names = [
        'yp_yandex_dns_export.controller.yandex_dns_export_manager_factory_{master}_dns_record_set.{action}'.format(master=master, action=action)
        for action in ['create', 'remove', 'update']
    ]
    sensors = [
        'replace_nan({' + 'cluster="{cluster}", service="main", host="all", sensor="{sensor_name}"'.format(cluster=solomon_cluster['name'], sensor_name=sensor_name) + '}, 0)'
        for sensor_name in sensor_names
    ]
    result_sensor = ' + '.join(sensors)

    return {
        'id': get_solomon_service_id(service_id, solomon_cluster),
        'projectId': SOLOMON_PROJECT_ID,
        'name': get_name('No updates to YP-{} from DNS export service'.format(master.upper()), solomon_cluster),
        'notificationChannels': [
            'juggler',
        ],
        'type': {
            'expression': {
                'program': 'let updates = {result_sensor};'.format(result_sensor=result_sensor),
                'checkExpression': 'sum(updates) == 0',
            }
        },
        'periodMillis': milliseconds(minutes=60),
        'delaySeconds': 30,
        'annotations': {
            'service': service_id,
            'host': host,
        },
    }, get_juggler_check_with_yp_downtime(host, service_id, master, notification_level=DEBUG, flaps_config=None)


def get_skipped_zones_alert(solomon_cluster, master):
    host = get_juggler_host(solomon_cluster)
    service_id = 'zones-update-skipped-for-yp-{}'.format(master)

    sensor_name = 'yp_yandex_dns_export.controller.yandex_dns_export_manager_factory_{master}_omitted_object'.format(master=master)

    return {
        'id': get_solomon_service_id(service_id, solomon_cluster),
        'projectId': SOLOMON_PROJECT_ID,
        'name': get_name('Some zones skipped for update to YP-{}'.format(master.upper()), solomon_cluster),
        'notificationChannels': [
            'juggler',
        ],
        'type': {
            'threshold': {
                'selectors': '{' + 'cluster="{}", service="main", host="all", sensor="{}"'.format(solomon_cluster['name'], sensor_name) + '}',
                'timeAggregation': 'SUM',
                'predicate': 'GT',
                'threshold': 0,
            }
        },
        'periodMillis': milliseconds(minutes=2),
        'delaySeconds': 30,
        'annotations': {
            'service': service_id,
            'host': host,
        },
    }, get_juggler_check_with_yp_downtime(host, service_id, master, flaps_config=None)


def get_update_time_alert(solomon_cluster, master):
    host = get_juggler_host(solomon_cluster)
    service_id = 'update-time-for-yp-{}'.format(master)

    sensor_name = 'yp_yandex_dns_export.controller.yandex_dns_export_manager_factory_{master}_sync_loop_time'.format(master=master)

    return {
        'id': get_solomon_service_id(service_id, solomon_cluster),
        'projectId': SOLOMON_PROJECT_ID,
        'name': get_name('Too long update for to YP-{}'.format(master.upper()), solomon_cluster),
        'notificationChannels': [
            'juggler',
        ],
        'type': {
            'threshold': {
                'selectors': '{' + 'cluster="{}", service="main", host="all", sensor="{}"'.format(solomon_cluster['name'], sensor_name) + '}',
                'timeAggregation': 'MAX',
                'predicate': 'GT',
                'threshold': solomon_cluster['max_update_time'],
            }
        },
        'periodMillis': milliseconds(seconds=30),
        'delaySeconds': 30,
        'annotations': {
            'service': service_id,
            'host': host,
        },
    }, get_juggler_check_with_yp_downtime(host, service_id, master, notification_level=DEBUG, flaps_config=juggler_sdk.FlapOptions(stable=300, critical=900, boost=0))


def get_source_retrieve_failed_alert(solomon_cluster, source):
    host = get_juggler_host(solomon_cluster)
    service_id = 'source-{}-retrieve-failed'.format(source)

    if not solomon_cluster['sources']['params'].get(source, {}).get('check_fails', True):
        return None, None

    return {
        'id': get_solomon_service_id(service_id, solomon_cluster),
        'projectId': SOLOMON_PROJECT_ID,
        'name': get_name('Failed to retrieve records from source "{}"'.format(source), solomon_cluster),
        'notificationChannels': [
            'juggler',
        ],
        'type': {
            'threshold': {
                'selectors': create_selectors(
                    cluster=solomon_cluster['name'],
                    service='main',
                    host='all',
                    sensor='yp_yandex_dns_export.service.retrieve.failed',
                    source=source,
                ),
                'timeAggregation': 'SUM',
                'predicate': 'GT',
                'threshold': 0,
            }
        },
        'periodMillis': milliseconds(seconds=30),
        'delaySeconds': 30,
        'annotations': {
            'service': service_id,
            'host': host,
            'alert_description': 'Failed retrieve records from source "{}"'.format(source),
        },
    }, get_juggler_check_downtime(host, service_id)


def get_source_no_retrieve_success_alert(solomon_cluster, source):
    host = get_juggler_host(solomon_cluster)
    service_id = 'source-{}-no-retrieve-success'.format(source)

    tags = []
    meta = {}
    if solomon_cluster.get('warden', {}).get('enable', False):
        tags += [
            'warden_auto_source',
            'warden_alert_category_boolean',
            'warden_functionality_{component}_{service}_yp-dns-export-zones-to-yp'.format(
                component=solomon_cluster['warden']['component']['name'],
                service=solomon_cluster['warden']['service']['name'],
            ),
        ]
        if solomon_cluster['sources']['params'].get(source, {}).get('warden', {}).get('create_spi', True):
            tags += [
                'warden_alert_create_spi',
                'warden_alert_account_metric',
            ]
        if solomon_cluster['sources']['params'].get(source, {}).get('warden', {}).get('start_flow', True):
            tags.append('warden_alert_start_flow')

        meta |= {
            'title': 'No successful polls of source "{}" in shard "{}"'.format(source, solomon_cluster['warden']['service']['shard_name']),
        }

    if 'update_frequency' in solomon_cluster['sources']['params'].get(source, {}):
        period_seconds = solomon_cluster['sources']['params'][source]['update_frequency'] * 1.5
        flaps_config = None
    else:
        period_seconds = 90
        flaps_config = flaps_config=juggler_sdk.FlapOptions(stable=period_seconds * 3, critical=period_seconds * 5, boost=0)
    return {
        'id': get_solomon_service_id(service_id, solomon_cluster),
        'projectId': SOLOMON_PROJECT_ID,
        'name': get_name('No successful records retrieves from source {}'.format(source), solomon_cluster),
        'notificationChannels': [
            'juggler',
        ],
        'type': {
            'threshold': {
                'selectors': create_selectors(
                    cluster=solomon_cluster['name'],
                    service='main',
                    host='all',
                    sensor='yp_yandex_dns_export.service.retrieve.success',
                    source=source,
                ),
                'timeAggregation': 'SUM',
                'predicate': 'EQ',
                'threshold': 0,
            }
        },
        'periodMillis': milliseconds(seconds=period_seconds),
        'delaySeconds': 30,
        'annotations': {
            'service': service_id,
            'host': host,
            'alert_description': 'No successful retrieves from source {}'.format(source),
        },
    }, get_juggler_check_downtime(host, service_id, flaps_config=flaps_config, meta=meta, tags=tags)


def get_no_successful_sync_cycles(solomon_cluster, master):
    host = get_juggler_host(solomon_cluster)
    service_id = 'no-successful-sync-cycles-{}'.format(master)

    sensor_name = "yp_yandex_dns_export.controller.yandex_dns_export_manager_factory_{master}_successful_sync_cycles".format(master=master)

    tags = []
    meta = {}
    if solomon_cluster.get('warden', {}).get('enable', False):
        tags += [
            'warden_auto_source',
            'warden_alert_create_spi',
            'warden_alert_start_flow',
            'warden_alert_account_metric',
            'warden_alert_category_boolean',
            'warden_functionality_{component}_{service}_yp-dns-export-zones-to-yp'.format(
                component=solomon_cluster['warden']['component']['name'],
                service=solomon_cluster['warden']['service']['name'],
            ),
        ]
        meta |= {
            'title': 'No successful export sync cycles for YP-{} in shard "{}"'.format(master.upper(), solomon_cluster['warden']['service']['shard_name']),
        }

    return {
        'id': get_solomon_service_id(service_id, solomon_cluster),
        'projectId': SOLOMON_PROJECT_ID,
        'name': get_name('No successful sync sycles for YP-{}'.format(master.upper()), solomon_cluster),
        'notificationChannels': [
            'juggler',
        ],
        'type': {
            'threshold': {
                'selectors': '{' + 'cluster="{}", service="main", host="all", sensor="{}"'.format(solomon_cluster['name'], sensor_name) + '}',
                'timeAggregation': 'SUM',
                'predicate': 'EQ',
                'threshold': 0,
            }
        },
        'periodMillis': milliseconds(minutes=10),
        'delaySeconds': 0,
        'annotations': {
            'service': service_id,
            'host': host,
        },
    }, get_juggler_check_with_yp_downtime(host, service_id, master, flaps_config=juggler_sdk.FlapOptions(stable=300, critical=900, boost=0), meta=meta, tags=tags)


def get_lock_acquired_alert(solomon_cluster):
    host = get_juggler_host(solomon_cluster)
    service_id = 'lock-acquired'

    return {
        'id': get_solomon_service_id(service_id, solomon_cluster),
        'projectId': SOLOMON_PROJECT_ID,
        'name': get_name('Failed to acquire lock', solomon_cluster),
        'notificationChannels': [
            'juggler',
        ],
        'type': {
            'threshold': {
                'selectors': '{' + 'cluster="{}", service="main", host="all", sensor="yp_yandex_dns_export.controller.lock_acquired"'.format(solomon_cluster['name']) + '}',
                'timeAggregation': 'LAST_NON_NAN',
                'predicate': 'EQ',
                'threshold': 0,
            }
        },
        'periodMillis': milliseconds(minutes=3),
        'delaySeconds': 30,
        'annotations': {
            'service': service_id,
            'host': host,
            'alert_description': 'All instances cannot acquire lock',
        },
    }, get_juggler_check_downtime(host, service_id, flaps_config=juggler_sdk.FlapOptions(stable=360, critical=1080, boost=0))


def get_yp_dns_racktables_zones_list_monitoring_check():
    service = 'racktables-zones-list-diff'

    return juggler_sdk.Check(
        host=JUGGLER_HOST,
        service=service,
        namespace=JUGGLER_NAMESPACE,
        aggregator='logic_or',
        ttl=900,
        refresh_time=300,
        notifications=get_notifications_config(level=WARN),
        flaps_config=juggler_sdk.FlapOptions(stable=1200, critical=1800, boost=0),
        children=[
            juggler_sdk.Child(
                host='{}_yp_dns'.format(dc),
                service='racktables_zones_list_monitoring',
                group_type='HOST',
                instance='',
            ) for dc in ['sas', 'man', 'vla', 'msk']
        ]
    )


def get_solomon_lag_check():
    return juggler_sdk.Check(
        host='solomon',
        service=get_juggler_service('solomon_lag'),
        namespace='yp.dns_export',
        aggregator='logic_or',
        children=[
            juggler_sdk.Child(
                host='solomon_alerting_project_lag',
                service='yp_yandex_dns_export',
                group_type='HOST',
                instance=''
            ),
        ],
    )


SOLOMON_CLUSTERS = [
    {
        'name': 'prod',
        'alerts_for_master': [
            get_no_updates_alert,
            get_skipped_zones_alert,
            get_update_time_alert,
            get_no_successful_sync_cycles,
        ],
        'alerts': {
            'all': [
                get_lock_acquired_alert,
            ],
            'by_retrieve_source': [
                get_source_retrieve_failed_alert,
                get_source_no_retrieve_success_alert,
            ],
        },
        'max_update_time': 80000,  # 80 seconds
        'sources': {
            'retrieve': [
                'in.yandex-team.ru',
                'in.yandex.net',
                'stable.qloud-d.yandex.net',
                'prestable.qloud-d.yandex.net',
                'test.qloud-d.yandex.net',
                'stable.qloud-b.yandex.net',
                'prestable.qloud-b.yandex.net',
                'test.qloud-b.yandex.net',
                'qloud-c.yandex.net',
            ],
            'params': {
                'test.qloud-d.yandex.net': {
                    'update_frequency': 30 * 60,  # 30 minutes
                    'check_fails': False,
                    'warden': {
                        'create_spi': False,
                        'start_flow': False,
                    },
                },
                'test.qloud-b.yandex.net': {
                    'update_frequency': 30 * 60,  # 30 minutes
                    'check_fails': False,
                    'warden': {
                        'create_spi': False,
                        'start_flow': False,
                    },
                },
            },
        },
        'juggler_params': {
            'mark': 'yp-dns-export-alerts',
        },
        'warden': {
            'enable': True,
            'component': {
                'name': 'dns',
            },
            'service': {
                'name': 'yp-dns',
                'shard_name': 'main',
            }
        },
    },
    {
        'name': 'qloud_d_prod',
        'host_suffix': '-qloud-d',
        'solomon_service_suffix': '-qloud-d',
        'name_suffix': ' (qloud-d)',
        'alerts_for_master': [
            get_no_updates_alert,
            get_skipped_zones_alert,
            get_update_time_alert,
            get_no_successful_sync_cycles,
        ],
        'alerts': {
            'all': [
                get_lock_acquired_alert,
            ],
            'by_retrieve_source': [
                get_source_retrieve_failed_alert,
                get_source_no_retrieve_success_alert,
            ],
        },
        'max_update_time': 40000,  # 40 seconds
        'sources': {
            'retrieve': [
                'stable.qloud-d.yandex.net',
                'prestable.qloud-d.yandex.net',
                'test.qloud-d.yandex.net',
            ],
            'params': {
                'test.qloud-d.yandex.net': {
                    'update_frequency': 30 * 60,  # 30 minutes
                    'check_fails': False,
                    'warden': {
                        'create_spi': False,
                        'start_flow': False,
                    },
                },
            },
        },
        'juggler_params': {
            'mark': 'yp-dns-export-qloud-d-alerts',
        },
        'warden': {
            'enable': True,
            'component': {
                'name': 'dns',
            },
            'service': {
                'name': 'yp-dns',
                'shard_name': 'qloud-d',
            }
        },
    },
]


def update_juggler_check(juggler_client, check):
    if isinstance(check, juggler_sdk.CheckFilter):
        result = juggler_client.remove_checks([check]).results[0]
        logging.info('Removed juggler check {}:{}: {}'.format(check.host, check.service, 'Success' if result.success else 'Failed'))
    else:
        diff = juggler_client.upsert_check(check).diff.to_dict()
        logging.info('Updated juggler check {}:{}'.format(check.service, check.service))
        logging.info('Diff:\n{}'.format(json.dumps(diff, indent=2)))


def update_alert(solomon_client, juggler_client, solomon_config, juggler_check):
    if solomon_config is None:
        return

    solomon_client.put_item('alerts', solomon_config['id'], solomon_config)
    juggler_check.meta.setdefault('urls', []).extend(get_standard_urls(solomon_config['id']))

    if juggler_check is not None:
        update_juggler_check(juggler_client, juggler_check)


def update_alerts_for_cluster(solomon_client, juggler_client, solomon_cluster):
    for master in MASTERS:
        for get_alert_func in solomon_cluster.get('alerts_for_master', []):
            update_alert(solomon_client, juggler_client, *get_alert_func(solomon_cluster, master))

    for get_alert_func in solomon_cluster.get('alerts', {}).get('all', []):
        update_alert(solomon_client, juggler_client, *get_alert_func(solomon_cluster))

    for get_alert_func in solomon_cluster.get('alerts', {}).get('by_retrieve_source', []):
        for source in solomon_cluster.get('sources', {}).get('retrieve', []):
            update_alert(solomon_client, juggler_client, *get_alert_func(solomon_cluster, source))


def update_alerts(solomon_client, juggler_client):
    update_juggler_check(juggler_client, get_yp_dns_racktables_zones_list_monitoring_check())

    update_juggler_check(juggler_client, get_solomon_lag_check())

    for solomon_cluster in SOLOMON_CLUSTERS:
        with juggler_sdk.JugglerApi('http://juggler-api.search.yandex.net', oauth_token=os.environ['JUGGLER_OAUTH_TOKEN'], mark=solomon_cluster['juggler_params'].get('mark')) as juggler_client:
            update_alerts_for_cluster(solomon_client, juggler_client, solomon_cluster)


def parse_args(argv):
    parser = argparse.ArgumentParser()
    parser.add_argument('--solomon-token', required=True, help='solomon token. https://solomon.yandex-team.ru/api/internal/auth')
    parser.add_argument('--juggler-token', required=True, help='juggler token. https://juggler.yandex-team.ru/')
    parser.add_argument('--verbose', action="store_true", help='Enable verbose mode')

    args = parser.parse_args(argv)

    if args.juggler_token is not None:
        os.environ['JUGGLER_OAUTH_TOKEN'] = args.juggler_token

    return args


def main(argv):
    args = parse_args(argv)
    logging.basicConfig(level=(logging.DEBUG if args.verbose else logging.INFO))

    solomon_client = SolomonApi(SOLOMON_PROJECT_ID, args.solomon_token)
    juggler_client = juggler_sdk.JugglerApi('http://juggler-api.search.yandex.net', oauth_token=os.environ['JUGGLER_OAUTH_TOKEN'])
    update_alerts(solomon_client, juggler_client)


if __name__ == '__main__':
    main(sys.argv[1:])
