from . import constants

import logging
import os
import subprocess
import time

logger = logging.getLogger(__name__)


def sudo_mkdirp(path):
    subprocess.check_call(["sudo", "mkdir", "-p", path])


def sudo_chown(path, username):
    subprocess.check_call(["sudo", "chown", username, path])


def sudo_chmod(path, mod):
    subprocess.check_call(["sudo", "chmod", mod, path])


def generate_id_rsa_pub(id_rsa_path):
    return subprocess.check_output(["ssh-keygen", "-y", "-f", id_rsa_path])


class SshDaemon:
    def __init__(self, username, hostname_list, port, key_blob):
        self.username = username
        self.key_blob = key_blob

        # virtual "/slot/sandbox" homedir of this user (unique to every ssh user but has the same path)
        self.homedir = os.path.expanduser("~/")
        self.ssh_dir = os.path.join(self.homedir, ".ssh")
        self.passwd_home = os.path.join('/home', self.username)

        if self.passwd_home != self.homedir:
            os.symlink(self.ssh_dir, os.path.join(self.passwd_home, ".ssh"))
        # real homedir of this user (as per /etc/passwd)
        #
        # need this one for proper setup of sshd, as "authorized_keys" file is looked up
        # using real homedir of user not some virtual one (this is if _relative_ path
        # to "authorized_keys" is given in sshd config - which is the case here, as _absolute_
        # path requires sudo to setup root permissions on dirs of the absolute path)
        self.sshd_dir = os.path.join(self.passwd_home, "sshd")

        self.hostname_list = hostname_list
        self.port = port

    def prepare(self):
        if not os.path.exists(self.sshd_dir):
            os.mkdir(self.sshd_dir)
            os.chmod(self.sshd_dir, 0o700)

        if not os.path.exists(self.ssh_dir):
            os.mkdir(self.ssh_dir)
            os.chmod(self.ssh_dir, 0o700)

        pseudo_bash_path = os.path.join(self.homedir, "pseudo_bash.sh")
        with open(pseudo_bash_path, "w") as fout:
            fout.write(constants.PSEUDO_BASH)
        os.chmod(pseudo_bash_path, 0o777)

        id_rsa_path = os.path.join(self.ssh_dir, "id_rsa")
        with open(id_rsa_path, "wb") as fout:
            fout.write(self.key_blob)
        os.chmod(id_rsa_path, 0o700)

        id_rsa_pub = generate_id_rsa_pub(id_rsa_path)

        authorized_keys_path = os.path.join(self.sshd_dir, "authorized_keys")
        if not os.path.exists(authorized_keys_path):
            with open(authorized_keys_path, "wb") as fout:
                fout.write(f"command=\"{pseudo_bash_path}\" ".encode("ascii"))
                fout.write(id_rsa_pub)
            os.chmod(authorized_keys_path, 0o600)

        ssh_config_path = os.path.join(self.ssh_dir, "config")
        with open(ssh_config_path, "w") as fout:
            fout.write(constants.DEFAULT_SSH_CONFIG_TEMPLATE)
            for host in self.hostname_list:
                fout.write(constants.SSH_CONFIG_HOST_TEMPLATE.format(host=host['hostname'], hostname=host["sshd_hostname"],
                                                                     username=host["sshd_user"], port=host["sshd_port"]))
        os.chmod(ssh_config_path, 0o600)

        sshd_config_path = os.path.join(self.homedir, "sshd.config")
        with open(sshd_config_path, "w") as fout:
            fout.write(constants.SSHD_CONFIG_TEMPLATE.format(homedir=self.homedir))

        pseudo_ssh_path = os.path.join(self.ssh_dir, "pseudo_ssh")
        with open(pseudo_ssh_path, "w") as fout:
            fout.write(constants.PSEUDO_SSH_TEMPLATE.format(homedir=self.homedir))
        os.chmod(pseudo_ssh_path, 0o700)

        self.sshd_args = ["/usr/sbin/sshd", "-D", "-e", "-p", str(self.port), "-f", sshd_config_path]

        for method in ("rsa", "dsa"):
            key_path = os.path.join(self.ssh_dir, "host_{}_key".format(method))
            subprocess.check_call(["ssh-keygen", "-t", method, "-f", key_path, "-N", ""])
            self.sshd_args += ["-h", key_path]

        self.pseudo_ssh_path = pseudo_ssh_path

    def start(self):
        sshd_log_path = os.path.join(self.homedir, "sshd.log")
        self.proc = subprocess.Popen(self.sshd_args, stderr=open(sshd_log_path, "w"))

        time.sleep(0.1)
        if self.proc.returncode is not None:
            raise Exception("Ssh daemon exited with non-zero code (code: {}, stderr: {})"
                            .format(self.proc.returncode, open(sshd_log_path).read()))

        logger.info("Ssh daemon started")
