import logging
import signal

import gevent
import gevent.queue
import gevent.threading
import ylock
from infra.swatlib.gevent import geventutil as gutil
from infra.swatlib.gevent import greenthread

from sepelib.util import retry


_STANDOFF_TIMEOUT_EXCEEDED = object()
_DISCONNECTED = object()


class Runnable(object):
    def run(self):
        raise NotImplementedError


class YtLockGeventFriendly(object):
    """
    Executes ytlock operations in separate thread to prevent blocking eventloop.
    """
    def __init__(self, lock, pool_size=1):
        """
        :type lock: ylock.backends.yt.YTLock
        :type pool_size: int
        """
        self.lock = lock
        self.pool = gevent.threadpool.ThreadPool(pool_size)

    def acquire(self, timeout=None):
        return self.pool.apply(self.lock.acquire, kwds={'timeout': timeout})

    def release(self, remove_node=True):
        return self.pool.apply(self.lock.release, kwds={'remove_node': remove_node})

    def check_acquired(self):
        return self.pool.apply(self.lock.check_acquired)


class YtExclusiveService(greenthread.GreenThread):
    """
    A service than runs under a YT lock.
    Designed to be used as a wrapper for unaware service.
    """
    # it's a good practice, mentioned in google chubby papers
    # to give up control once in a while
    DEFAULT_STANDOFF_STRATEGY = lambda: 12 * 3600

    def __init__(self, cfg, name, runnable,
                 lock_timeout=None,  # transaction life timeout without get/ping
                 acquire_timeout=None,
                 standoff_strategy=DEFAULT_STANDOFF_STRATEGY,
                 metrics_registry=None):
        super(YtExclusiveService, self).__init__()

        self._service = None
        if callable(runnable):
            self._run = runnable
        else:
            self._run = runnable.run
        self.name = name
        self.acquire_timeout = acquire_timeout
        self._log = logging.getLogger('exclusive({})'.format(self.name))

        manager = ylock.create_manager(**cfg)
        lock = manager.lock(
            name=self.name,
            block=True,
            block_timeout=self.acquire_timeout,
            timeout=lock_timeout)
        # Send SIGUSR1 which terms process by default. Otherwise it will be send
        # SIGINT which can be catched by try/except.
        lock._yt.config['transaction_use_signal_if_ping_failed'] = True
        self._lock = lock
        # self._lock = YtLockGeventFriendly(lock)
        self._standoff_strategy = standoff_strategy

        self._stopped = False
        self._stopping_lock = gevent.threading.Lock()

        self._disconnect_event = gevent.event.Event()

        if metrics_registry is not None:
            self._metrics_registry = metrics_registry.path('exclusive-service', name)
        else:
            self._metrics_registry = metrics_registry

    def _sigusr_handler(self, signum, frame):
        if signum == signal.SIGUSR1:
            self._disconnect_event.set()

    def _wait_disconnect(self, chan, standoff_timeout):
        self._disconnect_event.wait(timeout=standoff_timeout)
        if self._disconnect_event.is_set():
            chan.put(_DISCONNECTED)
        else:
            chan.put(_STANDOFF_TIMEOUT_EXCEEDED)

    def _wait_service(self, chan):
        try:
            rv = self._service.get()
        except Exception as e:
            chan.put(e)
        else:
            chan.put(rv)

    def _wait_disconnect_or_service(self, standoff_timeout=None):
        chan = gevent.queue.Queue()
        gs = (
            gevent.spawn(self._wait_disconnect, chan=chan, standoff_timeout=standoff_timeout),
            gevent.spawn(self._wait_service, chan=chan),
        )
        try:
            return chan.get()
        finally:
            for g in gs:
                gutil.force_kill_greenlet(g, kill_timeout=1)

    def _release_lock_and_exit(self):
        sleeper = retry.RetrySleeper(max_tries=5)
        while True:
            self._log.info("releasing lock...")
            try:
                self._lock.release()
                self._log.info("successfully released lock")
                self._set_acquired_locks_gauge(0)
                break
            except Exception as e:
                if not sleeper.increment(exception=False):
                    self._log.info("failed to release lock: %s", e)
                    self._set_acquired_locks_gauge(1)
                    break
        # DEPLOY-1583: exit in any of cases:
        # * standoff timeout
        # * service exception
        # * service stop
        self._log.info('exiting...')
        raise SystemExit(123)

    def _set_acquired_locks_gauge(self, v):
        if self._metrics_registry:
            self._metrics_registry.get_summable_gauge('acquired_locks').set(v)

    def run(self):
        sleeper = retry.RetrySleeper(max_delay=5)
        while not self._stopped:
            self._log.info("acquiring lock with timeout of %s seconds...", self.acquire_timeout)
            try:
                acquired = self._lock.acquire(timeout=self.acquire_timeout)
            except Exception as e:
                self._log.info("failed to acquire lock: %s", e)
                self._lock.release()
                sleeper.increment()
                continue

            if not acquired:
                self._log.info("failed to acquire lock within timeout of %s seconds", self.acquire_timeout)
                self._set_acquired_locks_gauge(0)
            else:
                signal.signal(signal.SIGUSR1, self._sigusr_handler)
                self._set_acquired_locks_gauge(1)
                break

        # we locked, yey!
        # now start service and wait for session state change
        standoff_timeout = self._standoff_strategy()
        self._log.info("became singleton - "
                       "starting service with standoff timeout of %s seconds...", standoff_timeout)

        with self._stopping_lock:
            if self._stopped:
                return
            self._service = gevent.Greenlet(self._run)
            self._service.start()
        res = self._wait_disconnect_or_service(standoff_timeout=standoff_timeout)
        if res is _DISCONNECTED:
            self._log.warn("disconnected - stopping service")
        elif res is _STANDOFF_TIMEOUT_EXCEEDED:
            self._log.info("was leading for too long (more than %s seconds), "
                           "stand off - stopping service", standoff_timeout)
        elif isinstance(res, Exception):
            self._log.info("service raised an exception")
        else:
            self._log.info("service returned %r", res)

        gutil.force_kill_greenlet(self._service, log=self._log)
        self._log.info("service stopped")
        self._release_lock_and_exit()

    def stop(self):
        self._log.info("stopping exclusive service...")
        with self._stopping_lock:
            self._stopped = True
            if self._service:
                self._log.info("stopping service itself...")
                gutil.force_kill_greenlet(self._service, log=self._log)

        super(YtExclusiveService, self).stop()
        self._log.info("stopped exclusive service")
        self._release_lock_and_exit()
