import requests
import sys
import logging
import subprocess
from collections import defaultdict
from requests.packages.urllib3 import Retry
import xml.etree.ElementTree as ET
import re
import json
requests.packages.urllib3.disable_warnings()


def create_session(oath_token):
    session = requests.Session()

    if oath_token is not None:
        session.headers["Authorization"] = "OAuth {}".format(oath_token)

    retries = 10
    retry = Retry(
        backoff_factor=0.3,
        read=retries,
        connect=retries,
        total=retries,
    )
    session.mount(
        'http://',
        requests.adapters.HTTPAdapter(max_retries=retry),
    )
    session.mount(
        'https://',
        requests.adapters.HTTPAdapter(max_retries=retry),
    )
    return session


def setup_custom_logger(name):
    logging.basicConfig(level=logging.DEBUG)
    formatter = logging.Formatter(fmt='%(asctime)s %(levelname)-8s %(message)s',
                                  datefmt='%Y-%m-%d %H:%M:%S')
    screen_handler = logging.StreamHandler(stream=sys.stdout)
    screen_handler.setFormatter(formatter)
    logger = logging.getLogger(name)
    logger.handlers = []
    logger.setLevel(logging.DEBUG)
#    logger.addHandler(screen_handler)
    logging.getLogger("requests").setLevel(logging.WARNING)
    logging.getLogger("urllib3").setLevel(logging.WARNING)
    return logger


LOGGER = setup_custom_logger('binary')


class StartrackTicket:
    def __init__(self, ticket_id, ticket_info, session):
        self.ticket_id = ticket_id
        self.session = session
        self.ticket = ticket_info
        self.last_commit = None
        self.pr = None

        self._load_info()

    @staticmethod
    def from_ticket(ticket_id, ticket_info, session):
        return StartrackTicket(ticket_id, ticket_info, session)

    def get_description(self):
        if "description" in self.ticket:
            return self.ticket["description"]
        else:
            return ""

    def _get_epoch_last_commit_info(self):
        description = self.get_description()
        epoch_info = StartrackTicket.epoch_from_ticket_description(description)
        if epoch_info is not None and "last_commit" in epoch_info:
            return epoch_info["last_commit"]
        else:
            return None

    def _load_info(self):
        LOGGER.debug("Retrieving remote info for ticket {}".format(self.ticket["key"]))
        links = self.session.get("https://st-api.yandex-team.ru/v2/issues/{}/remotelinks".format(self.ticket["key"]),
                                 timeout=20)
        links.raise_for_status()
        links = links.json()

        pullrequests = [link["object"]["id"] for link in links
                        if link["object"]["application"]["id"] == "ru.yandex.arcanum"]

        merge_commits = []
        merge_commits_to_pr = {}
        for pr in pullrequests:
            LOGGER.debug("Retrieving pull request info for ticket {}".format(self.ticket["key"]))

            commit_info = self.session.get("https://a.yandex-team.ru/api/v1/review-requests/{}?fields=id,merge_commits".format(pr), verify=False)
            commit_info.raise_for_status()
            commit_info = commit_info.json()

            if "data" in commit_info and "merge_commits" in commit_info["data"]:
                pr_merge_commits = commit_info["data"]["merge_commits"]
                LOGGER.debug("PR {} merge_commits {}".format(pr, pr_merge_commits))
                merge_commits.extend(pr_merge_commits)
                for pr_merge_commit in pr_merge_commits:
                    merge_commits_to_pr[int(pr_merge_commit)] = int(pr)

        LOGGER.debug("Ticket {} merge_commits {}".format(self.ticket["key"], merge_commits))

        last_commit = self._get_epoch_last_commit_info()
        if last_commit is None:
            if len(merge_commits) > 0:
                last_commit = int(max(merge_commits))

        self.last_commit = last_commit
        self.pr = merge_commits_to_pr[self.last_commit]
        LOGGER.debug("Ticket {ticket} last commit {commit} pr={pr}".format(
            ticket=self.ticket["key"], commit=self.last_commit, pr=self.pr))

    @staticmethod
    def epoch_from_ticket_description(description):
        description_index = description.find(StartrackTicket.Z2_FOOTER)
        if description_index >= 0:
            z2_description = description[description_index:]
            z2_epoch_footer_index = z2_description.find(StartrackTicket.Z2_EPOCH_FOOTER)
            if z2_epoch_footer_index >= 0:
                z2_epoch_text = z2_description[z2_epoch_footer_index+len(StartrackTicket.Z2_EPOCH_FOOTER):]
                return json.loads(z2_epoch_text)

        return None

    def get_pr(self):
        return self.pr

    def get_last_commit(self):
        return self.last_commit

    def build_epoch_info(self):
        return {"last_commit": self.last_commit}

    def get_components(self):
        return [int(item["id"]) for item in self.ticket["components"]]

    Z2_FOOTER = "<# <!-- Z2 FOOTER --> #>"
    Z2_EPOCH_FOOTER = "<# <!-- Z2 EPOCH FOOTER --> #>"

    def update_status(self, tracking_info, deploy_locations, debug_messages):
        LOGGER.debug("Want to update ticket {} tracking_info={}".format(self.ticket_id, tracking_info))

        issue = self.session.get("https://st-api.yandex-team.ru/v2/issues/{}".format(self.ticket_id), timeout=20)
        issue.raise_for_status()
        issue = issue.json()
        description = ""
        if "description" in issue:
            description = issue["description"]

        old_description = description

        description_index = description.find(StartrackTicket.Z2_FOOTER)

        if description_index >= 0:
            description = description[:description_index]

        description += StartrackTicket.Z2_FOOTER + "\n"
        description += """
    <# <!-- This table is generated automatically. If you want to remove it, do not forget
    <# to prevent z2 from new modifications (by default it is enough to remove
    tag z2-release-tracking from the ticket)
    --> #>
    """

        footer = "===== Z2 deploy information\n"
        footer += "#|\n"
        footer += "||{}||\n".format("|".join(["**" + location["name"] + "**" for location in deploy_locations]))

        footer += "||{}||\n".format(
            "|".join(["!!(green)Deployed!!" if tracking_info[location["name"]] else "!!(red)Pending!!"
                      for location in deploy_locations]))

        footer += "|#\n"
        footer += "======+ Z2 deploy Technical info\n{technical_info}\n".format(
            technical_info="\n".join(debug_messages))
        footer += "{footer}\n{epoch_info}".format(footer=StartrackTicket.Z2_EPOCH_FOOTER,
                                                  epoch_info=json.dumps(self.build_epoch_info()))
        description += footer

        if old_description != description:
            LOGGER.debug("Updating ticket {} description".format(self.ticket_id))

            result = self.session.patch("https://st-api.yandex-team.ru/v2/issues/{}".format(self.ticket_id),
                                        json={"description": description},
                                        timeout=20)
            result.raise_for_status()
        else:
            LOGGER.debug("Old description is the same as new one. Not updating ticket {}".format(self.ticket_id))


def load_tickets(st_session, st_queues, st_statuses, startrack_component_id):
    LOGGER.debug("Loading tickets for component {}".format(startrack_component_id))

    filters = {
        "filter":
            {
                "queue": st_queues,
                "status": st_statuses,
                "components": startrack_component_id,
                "tags": ["z2-release-tracking"]
            }
    }

    results_count = 100
    page = 1

    tickets = {}

    while results_count == 100:
        LOGGER.debug("Retrieving tickets, page {}".format(page))
        results = st_session.post("https://st-api.yandex-team.ru/v2/issues/"
                                  "_search?perPage={}&page={}&orderBy=keyd&orderAsc=false]".format(results_count, page),
                                  json=filters,
                                  timeout=20)
        results.raise_for_status()
        results = results.json()
        for ticket in results:
            startrack_ticket = StartrackTicket.from_ticket(ticket["key"], ticket, st_session)
            tickets[ticket["key"]] = startrack_ticket

        results_count = len(results)
        page += 1

    LOGGER.debug("Loaded {} tickets, tickets={}".format(len(tickets), tickets))
    return tickets


class SvnBranch:
    def __init__(self, branch, base_branch, revision):
        self.branch = branch
        self.base_branch = base_branch
        self.revision = revision
        self.merged_revisions = None

        LOGGER.debug("branch={branch} base_branch={base_branch}".format(branch=branch, base_branch=base_branch))
        self._load_info()

    def _load_info(self):
        if self.branch != "trunk":
            self.merged_revisions = self._load_svn_merge_info(self.branch, self.base_branch, self.revision)

    def has_merged_revision(self, revision):
        has_merged_revision = False
        if self.branch == "trunk":
            has_merged_revision = revision < self.revision
        else:
            has_merged_revision = revision in self.merged_revisions

        LOGGER.debug("Branch {} looking for revision {}, found {}".format(self.branch, revision, has_merged_revision))
        return has_merged_revision

    @staticmethod
    def parse_int(s, base=10, val=None):
        if s.isdigit():
            return int(s, base)
        else:
            return val

    @staticmethod
    def _load_svn_merge_info(svn_branch, base_branch, revision):
        merged_revisions = set()
        query = "svn log svn+ssh://arcadia.yandex.ru/arc/branches/{}/{} --xml  -l 100000 -r {}:1".format(base_branch, svn_branch, revision)
        LOGGER.debug("Executing svn command [{}]".format(query))
        p = subprocess.Popen(query, stdout=subprocess.PIPE, shell=True)
        (output, err) = p.communicate()
        root = ET.fromstring(output)
        for log in root.findall(".//logentry"):
            if "revision" in log.attrib:
                merged_revisions.add(int(log.attrib["revision"]))

            msg_node = log.find("msg")
            if msg_node is not None:
                msg_text = msg_node.text
                if msg_text.lower().find("merge from trunk:") == 0:
                    revisions = msg_text.split("\n")[0].split(":")[1].strip()
                    for revision in re.split(",| ", revisions):
                        revision.strip()
                        if len(revision) > 0 and revision[0] == "r":
                            revision = revision[1:]

                        revision = re.search(r'\d+.*', revision)
                        if revision is not None:
                            revision = revision.group(0)
                            if SvnBranch.parse_int(revision):
                                merged_revisions.add(int(revision))
                else:
                    if "revision" in log.attrib:
                        revision = log.attrib["revision"]
                        if SvnBranch.parse_int(revision):
                            merged_revisions.add(int(revision))

        return merged_revisions


def z2_process_tickets(startrack_arcanum_token, st_queues, st_statuses, z2_components):
    svn_branches = {}
    tracked_tickets = {}
    released_components = defaultdict(list)

    for component in z2_components.keys():
        z2_package = z2_components[component]["z2_package"]
        deploy_locations = z2_components[component]["locations"]

        for location_config in deploy_locations:
            location = location_config["name"]
            LOGGER.debug("Retrieving z2 info for {}".format(location))
            z2_config_id = location_config["z2_configId"]

            z2_token = location_config["token"]

            payload = {"configId": z2_config_id, "apiKey": z2_token}
            z2_status = requests.get('https://z2.yandex-team.ru/api/v1/items', params=payload, verify=False)
            z2_status.raise_for_status()
            z2_status = z2_status.json()

            LOGGER.debug("Z2 {} status {}".format(z2_config_id, z2_status))

            if "success" not in z2_status or not z2_status["success"]:
                LOGGER.debug("Z2 {} status not ok, skipping".format(z2_config_id))
                raise Exception("Z2 communication error")

            LOGGER.debug("Z2 {} looking for component {}".format(z2_config_id, z2_package))

            components = [z2_item["version"] for z2_item in z2_status["response"]["items"] if
                          z2_item["name"] == z2_package]
            if len(components) == 0:
                LOGGER.debug("Z2 {} component {} not found, skipping".format(z2_config_id, z2_package))
                continue

            z2_svn_version = int(components[0].split(".")[0])
            base_branch = z2_components[component]["branch_name"]
            branch_filter = z2_components[component]["branch_filter"]

            z2_branch = branch_filter(components[0].split(".")[1])
            svn_branch_name = "{z2_branch}-{z2_svn_version}".format(z2_branch=z2_branch, z2_svn_version=z2_svn_version)
            if svn_branch_name not in svn_branches:
                svn_branches[svn_branch_name] = SvnBranch(z2_branch, base_branch, z2_svn_version)

            released_components[component].append((location, svn_branches[svn_branch_name]))

            LOGGER.debug("Z2 {} component {} svn_version {}".format(z2_config_id, z2_package, z2_svn_version))

    st_session = create_session(startrack_arcanum_token)
    for component in z2_components.keys():
        startrack_component_id = z2_components[component]["startrack_component"]
        tickets = load_tickets(st_session, st_queues, st_statuses, startrack_component_id)
        tracked_tickets.update(tickets.items())

    LOGGER.debug("Tracked tickets {}".format(tracked_tickets))

    for ticket_id in tracked_tickets.keys():
        ticket = tracked_tickets[ticket_id]
        last_commit = ticket.get_last_commit()
        components = ticket.get_components()

        released_in_locations = dict()
        for released_component in released_components:
            component_info = released_components[released_component]
            startrack_component_id = z2_components[released_component]["startrack_component"]
            if startrack_component_id not in components:
                continue
            messages = list()
            LOGGER.debug("Release component {}".format(released_component))
            for location, svn_branch in component_info:
                released = svn_branch.has_merged_revision(last_commit)
                released_in_locations[location] = released

                LOGGER.debug("ticket {} consided to be released {} in {}".format(ticket_id, released, location))
                messages.append("Location {location} branch {svn_branch} commit {last_commit}/PR {pr} from trunk "
                                "{status} as merged into branch".format(
                                    svn_branch=svn_branch.branch,
                                    pr=ticket.get_pr(),
                                    status="found" if released else "not found",
                                    last_commit=last_commit,
                                    location=location))

            ticket.update_status(released_in_locations, z2_components[released_component]["locations"], messages)

    LOGGER.debug("Done")
