import logging
from typing import Optional, List, Tuple

from django.conf import settings
from ldap.controls import SimplePagedResultsControl
import ldap

from ad_system.ad import models
from ad_system.ad.exceptions import WrongADRoleError, ADError

_PAGE_SIZE = 100
log = logging.getLogger(__name__)


def connect(client, uri, user, password):
    log.info('connecting to "%s" with user "%s"', uri, user)
    ldo = client.initialize(uri)

    ldo.set_option(client.OPT_DEBUG_LEVEL, 255)
    ldo.set_option(client.OPT_NETWORK_TIMEOUT, 1)
    ldo.set_option(client.OPT_PROTOCOL_VERSION, client.VERSION3)
    ldo.set_option(client.OPT_REFERRALS, 0)
    ldo.set_option(client.OPT_TIMEOUT, 1)
    ldo.set_option(client.OPT_X_TLS, client.OPT_X_TLS_DEMAND)
    ldo.set_option(client.OPT_X_TLS_CACERTFILE, settings.CA_BUNDLE)
    ldo.set_option(client.OPT_X_TLS_DEMAND, True)
    ldo.set_option(client.OPT_X_TLS_REQUIRE_CERT, client.OPT_X_TLS_NEVER)
    # следующая опция должна выставляться после всех остальных TLS-опций, иначе они игнорируются
    ldo.set_option(client.OPT_X_TLS_NEWCTX, 0)

    ldo.start_tls_s()
    ldo.simple_bind_s(user, password)
    return ldo


class NewLDAP(object):
    """
    Коннектор к LDAP для многократной проверки активности пользователей в AD
    """
    ldap_client = ldap

    def __init__(self):
        self.ldap = None
        self.lc = SimplePagedResultsControl(True, size=_PAGE_SIZE, cookie='')

    def __enter__(self):
        self.connect()
        return self

    def __exit__(self, type, value, traceback):
        self.disconnect()

    def connect(self):
        if self.ldap is not None:
            return

        self.ldap = connect(self.ldap_client, settings.AD_LDAP_HOST, settings.AD_LDAP_USERNAME, settings.AD_LDAP_PASSWD)

    def disconnect(self):
        """Отключается от LDAP"""

        if self.ldap is not None:
            self.ldap.unbind_s()
            self.ldap = None
            log.info('disconnected from ldap')

    def _get_page(self, base, search_flt, search_fields, serverctrls, first=False):
        messid = None
        if not first:
            pctrls = [c for c in serverctrls if c.controlType == SimplePagedResultsControl.controlType]
            if pctrls:
                if pctrls[0].cookie:
                    self.lc.cookie = pctrls[0].cookie
                    messid = self.ldap.search_ext(
                        base, ldap.SCOPE_SUBTREE, search_flt,
                        search_fields, serverctrls=[self.lc]
                    )
                    rtype, rdata, rmsgid, serverctrls = self.ldap.result3(messid)
                    return rdata, serverctrls
            else:
                log.warning('server ignores RFC 2696 control.')

        else:
            messid = self.ldap.search_ext(
                base, ldap.SCOPE_SUBTREE, search_flt,
                search_fields, serverctrls=[self.lc]
            )
            rtype, rdata, rmsgid, serverctrls = self.ldap.result3(messid)
            return rdata, serverctrls
        return None, None

    def search(self, base: str, search_flt: str, search_fields: List[str]) -> List[Tuple[str, dict]]:
        """
        Запросить LDAP и вернуть итератор по всем найденным записям.
        """
        rdata, serverctrls = self._get_page(base, search_flt, search_fields, None, True)

        while rdata is not None:
            for entry in rdata:
                yield entry
            rdata, serverctrls = self._get_page(base, search_flt, search_fields, serverctrls)

    def check_is_group_in_idm(self, group_dn: str) -> bool:
        """
        Проверить, что группа находится под управлением IDM
        """
        # extensionAttribute1 хранит информацию о том, надо ли контролировать группу в IDM
        # Тикет: https://st.yandex-team.ru/INFRAWIN-336
        data = self.fetch_ad_group_data(group_dn=group_dn, fields=['extensionAttribute1'])
        if data is None:
            raise WrongADRoleError(f'Группы {group_dn} нет в AD')

        if self.fetch_extension_attribute_1(data):
            return True

        return False

    def fetch_responsible_users(self, group_dn: str) -> List[str]:
        """
        Получение ответственных в группе
        """
        # extensionAttribute2 хранит информацию об ответственных в следующем формате: (User1,User2,...)
        # Тикет: https://st.yandex-team.ru/INFRAWIN-336
        data = self.fetch_ad_group_data(group_dn=group_dn, fields=['extensionAttribute1', 'extensionAttribute2'])
        if data is None:
            raise WrongADRoleError(f'Группы {group_dn} нет в AD')

        if not self.fetch_extension_attribute_1(data):
            raise WrongADRoleError(f'Группы {group_dn} нет под контролем IDM')

        result_str = data[0][1].get('extensionAttribute2', [b'()'])[0].decode()
        result = result_str[1:-1].split(',')

        return result

    def create_ad_group(self, group_dn: str, system_slug: str):
        """
        Создает новую AD-группу, которая связана с системой со слагом system_slug
        """
        if not group_dn.endswith(settings.AD_LDAP_IDM_OU):
            raise WrongADRoleError(f'Невозможно создать группу {group_dn}')

        name = group_dn.split(',')[0][3:].encode()
        mod_attrs = [
            ('objectClass', [b'group', b'top']),
            ('groupType', settings.AD_LDAP_NEW_GROUP_TYPE.encode()),
            ('cn', name),
            ('name', name),
            ('sAMAccountName', name),
            ('extensionAttribute1', system_slug.encode()),
        ]

        self.ldap.add_s(group_dn, mod_attrs)

    def update_extension_attribute_1(self, group_dn: str, value: str):
        """
        Обновить значение extensionAttribute1
        """
        mod_attrs = [(ldap.MOD_REPLACE, 'extensionAttribute1', value.encode())]
        self.ldap.modify_s(group_dn, mod_attrs)

    def clean_extension_attribute_1(self, group_dn: str):
        """
        Очистить значение extensionAttribute1
        """
        mod_attrs = [(ldap.MOD_DELETE, 'extensionAttribute1', None)]
        self.ldap.modify_s(group_dn, mod_attrs)

    @staticmethod
    def fetch_extension_attribute_1(group_info: dict):
        """
        Достает значение extensionAttribute1
        """
        return group_info.get('extensionAttribute1', [b''])[0].decode()

    def update_responsible_users(self, group_dn: str, new_users: List[str]):
        """
        Обновление ответственных в группе
        """
        mod_attrs = [(ldap.MOD_REPLACE, 'extensionAttribute2', f'({",".join(new_users)})'.encode())]
        self.ldap.modify_s(group_dn, mod_attrs)

    def add_user_to_ad_group(self, user: str, group_dn: str):
        """
        Возвращает результат записи пользователя login в AD группу ad_group.
        """
        if not self.check_is_group_in_idm(group_dn):
            raise WrongADRoleError(f'Группа {group_dn} не находится под контролем IDM')

        account = self.search_user(user)

        if account is None:
            log.warning('Tried to add non-existing user %s to group %s', user, group_dn)
            raise ADError(f'Пользватель {user} не существует в AD')

        try:
            account.add_to_group(group_dn)
        except Exception:
            log.exception('Cannot add user %s to group %s', user, group_dn)
            raise

    def add_responsible_user_to_ad_group(self, user: str, group_dn: str):
        """
        Добавление ответственного в AD-группу
        """
        if not self.check_is_group_in_idm(group_dn):
            raise WrongADRoleError(f'Группа {group_dn} не находится под контролем IDM')

        responsible_users = self.fetch_responsible_users(group_dn)

        if user not in responsible_users:
            self.update_responsible_users(group_dn, responsible_users + [user])
        else:
            raise ADError(f'Пользователь {user} уже является ответственным в {group_dn}')

    def fetch_ad_group_data(self, group_dn: str, fields: List[str]) -> Optional[dict]:
        """
        Проверка на наличие AD-группы в системе
        """
        try:
            data = list(self.search(group_dn, '(objectclass=group)', fields))
        except Exception as e:
            # Если AD ничего не находит, то он выдает ошибку No Such Object
            return None
        for group_name, group_info in data:
            if group_name == group_dn:
                return group_info
        return None

    def remove_user_from_ad_group(self, user: str, group_dn: str, is_fired: bool = False):
        """
        Возвращает результат удаления пользователя login из AD группы ad_group.
        """
        if not self.check_is_group_in_idm(group_dn):
            raise WrongADRoleError(f'Группа {group_dn} не находится под контролем IDM')

        account = self.search_user(user)

        if account is None:
            log.warning('Tried to remove non-existing user %s from group %s', user, group_dn)
            raise ADError(f'Пользователь {user} не существует в AD')

        try:
            account.remove_from_group(group_dn)
        except Exception:
            log.exception('Cannot remove user %s from group %s', user, group_dn)
            if not is_fired:
                # Если пользователь уволен, то делаем вид, что удалили его из группы
                # все равно IDM сам это делает для уволенных
                raise

    def remove_responsible_user_from_ad_group(self, user: str, group_dn: str):
        """
        Удаляет ответственного из AD-группы
        """
        if not self.check_is_group_in_idm(group_dn):
            raise WrongADRoleError(f'Группа {group_dn} не находится под контролем IDM')

        responsible_users = self.fetch_responsible_users(group_dn)

        if user in responsible_users:
            responsible_users.remove(user)
            self.update_responsible_users(group_dn, responsible_users)
        else:
            raise ADError(f'Пользователь {user} не является ответственным в {group_dn}')

    def search_user(self, username: str) -> models.Account:
        return models.AccountManager(connector=self.ldap).search(username)
