# coding: utf-8


import logging
import attr
import ldap
from attr.validators import instance_of
from django.conf import settings
from django.utils.encoding import force_bytes, force_text

from ad_system.ad.exceptions import ADError, UserAlreadyInGroup

log = logging.getLogger(__name__)


@attr.s
class AccountManager(object):
    connector = attr.ib()

    def search(self, username: str):
        """
        Ищет пользователя по логину среди Users, ForeignUsers и OldUser.
        """
        log.info(f'Searching user {username}')
        account = None
        account_data = self.connector.search_s(
            settings.AD_LDAP_DC,
            ldap.SCOPE_SUBTREE,
            f'(&(SAMAccountName={username}){settings.LDAP_NOT_BLOCKED_USER_FILTER})'
        )
        account_data = list(filter(lambda data: data[0], account_data))
        if account_data:
            log.info(f'User {username} was found')
            account = Account.from_ldap(
                username=username,
                account_data=account_data,
                connector=self.connector,
            )
            return account
        log.info(f'User {username} was not found in AD')
        return account


@attr.s
class Account(object):

    connector = attr.ib()
    username = attr.ib(validator=instance_of(str))
    display_name = attr.ib(validator=instance_of(str))
    properties = attr.ib(validator=instance_of(dict))

    @classmethod
    def from_ldap(cls, account_data, username, connector=None):
        if not isinstance(account_data, list):
            raise TypeError('Wrong account data')
        if len(account_data) != 1:
            raise TypeError('Wrong account data len')
        account_data = account_data[0]
        if len(account_data) != 2:
            raise TypeError('Wrong account data sublen')
        display_name, properties = account_data
        return cls(
            display_name=display_name,
            username=username,
            properties=properties,
            connector=connector,
        )

    def is_active(self) -> bool:
        account_control = int(self.properties['userAccountControl'][0])
        result = account_control & settings.ADS_UF_ACCOUNTDISABLE == 0
        return result

    def add_to_group(self, group_dn: str):
        """
        Добавить учетку в группу в AD
        """

        user_groups = set(self.properties.get('memberOf', []))
        log.info(f'Adding user {self.username} to group {group_dn}')

        if force_bytes(group_dn) in user_groups:
            log.warning(f'User {self.username} already in group {group_dn}')
            raise UserAlreadyInGroup(f'Пользователь {self.username} уже находится в группе {group_dn}')

        diff = [(ldap.MOD_ADD, 'member', force_bytes(self.display_name))]
        self.connector.modify_s(force_text(group_dn), diff)

    def remove_from_group(self, group_dn: str):
        """
        Удалить учетку из группы в AD
        """

        log.info(f'Removing user {self.username} from group {group_dn}')
        groups = set(self.properties.get('memberOf', []))
        if force_bytes(group_dn) not in groups:
            log.warning(f'User {self.username} is not a member in group {group_dn}')
            raise ADError(f'Пользователь {self.username} не находится в группе {group_dn}')

        diff = [(ldap.MOD_DELETE, 'member', force_bytes(self.display_name))]
        self.connector.modify_s(force_text(group_dn), diff)
