from collections import defaultdict

from django_idm_api.hooks import Hooks as IDMHooks
from django_idm_api.hooks import IntranetHooksMixin
from django.conf import settings
from django.utils.encoding import force_text

from ad_system.ad.connector import NewLDAP
from ad_system.ad.exceptions import WrongADRoleError, UserAlreadyInGroup


class Hooks(IDMHooks, IntranetHooksMixin):

    def info(self):
        return {
            "code": 0,
            "roles": {
                "slug": "type",
                "name": "тип роли",
                "values": self.info_impl()}
        }

    def info_impl(self, **kwargs):
        return {
            "global": {
                "unique_id": "global",
                "name": "global",
                "roles": {
                    "slug": "role",
                    "name": "роль",
                    "values": self._global_roles(),
                }
            },
            "roles_in_groups": {
                "unique_id": "roles_in_groups",
                "name": "roles_in_groups",
                "roles": {
                    "slug": "ad_group",
                    "name": "AD-группа",
                    "values": self._roles_in_groups(),
                }
            }
        }

    @staticmethod
    def _global_roles():
        return {
            "system_group_relation": {
                'name': {
                    'en': 'system-group relation',
                    'ru': 'связь системы с группой',
                },
                'unique_id': 'system_group_relation',
                'fields': [
                    {
                        "is_required": True,
                        'name': {
                            'ru': 'слаг системы',
                            'en': 'system slug',
                        },
                        "slug": "system",
                    },
                    {
                        "is_required": True,
                        'name': {
                            'ru': 'DN группы',
                            'en': 'group DN',
                        },
                        "slug": "group_dn",
                    },

                ]
            }
        }

    @staticmethod
    def _roles_in_groups():
        """
        Ручка info
        """
        group_values = {}
        with NewLDAP() as ldap:
            all_groups = ldap.search(settings.AD_LDAP_DC, '(objectclass=group)', ['sAMAccountName'])
            for group_name, _ in all_groups:
                if group_name is None:
                    continue  # https://st.yandex-team.ru/IDM-9519
                current_group_dict = {
                    'name': group_name,
                    'unique_id': group_name,
                    'roles': {
                        'slug': 'group_roles',
                        'name': {
                            'ru': 'Роли группы',
                            'en': 'Group roles',
                        },
                        'values': {
                            'member': {
                                'unique_id': f'{group_name}::member',
                                'name': {
                                    'ru': 'Участник',
                                    'en': 'Member',
                                }
                            },
                            'responsible': {
                                'unique_id': f'{group_name}::responsible',
                                'name': {
                                    'ru': 'Ответственный',
                                    'en': 'Responsible',
                                }
                            }
                        }
                    }
                }
                group_values[group_name] = current_group_dict
        return group_values

    def add_role_impl(self, login, role, fields, **kwargs):
        """
        Добавить пользователя в AD-группу
        """
        # Роль "Связь роли с системой"
        # Нужна для создания связи системы в IDM с AD группой
        if role.get('type') == 'global':
            with NewLDAP() as ldap:
                current_group_dn = fields['group_dn']
                current_system_slug = fields['system']
                group_data = ldap.fetch_ad_group_data(group_dn=current_group_dn, fields=['extensionAttribute1'])
                if group_data is None:
                    ldap.create_ad_group(group_dn=current_group_dn, system_slug=current_system_slug)
                else:
                    extension_attribute_1_value = NewLDAP.fetch_extension_attribute_1(group_info=group_data)
                    if extension_attribute_1_value not in ['true', '']:
                        raise WrongADRoleError(f'You cannot create such relation {current_group_dn}'
                                               f'-{current_system_slug}')
                    else:
                        ldap.update_extension_attribute_1(group_dn=current_group_dn, value=current_system_slug)
            return {'data': fields}

        # Обратная совместимость на время миграции
        group_dn = role.get('ad_group') or role.get('role')

        if role['group_roles'] == 'member':
            with NewLDAP() as ldap:
                try:
                    ldap.add_user_to_ad_group(user=login, group_dn=group_dn)
                except UserAlreadyInGroup:
                    pass

        elif role['group_roles'] == 'responsible':
            with NewLDAP() as ldap:
                ldap.add_responsible_user_to_ad_group(user=login, group_dn=group_dn)

        else:
            raise WrongADRoleError(f"'group_roles': Expected `member`, `responsible` or `system_group_relation` "
                                   f"got {role['group_roles']}")

    def remove_role_impl(self, login, role, data, is_fired, **kwargs):
        """
        Убрать пользователя из AD-группы
        """
        if role.get('type') == 'global':
            with NewLDAP() as ldap:
                current_group_dn = data['group_dn']
                current_system_slug = data['system']
                group_data = ldap.fetch_ad_group_data(group_dn=current_group_dn, fields=['extensionAttribute1'])
                if group_data is None:
                    group_data = {}
                extension_attribute_1_value = NewLDAP.fetch_extension_attribute_1(group_data)
                if extension_attribute_1_value == current_system_slug:
                    ldap.clean_extension_attribute_1(group_dn=current_group_dn)
                else:
                    raise WrongADRoleError(f'There is no role linked with {current_group_dn} and {current_system_slug}')
            return

        # Обратная совместимость на время миграции
        group_dn = role.get('ad_group') or role.get('role')

        if role['group_roles'] == 'member':
            try:
                with NewLDAP() as ldap:
                    ldap.remove_user_from_ad_group(user=login, group_dn=group_dn, is_fired=is_fired)
            except WrongADRoleError:
                if not is_fired:
                    raise
        elif role['group_roles'] == 'responsible':
            with NewLDAP() as ldap:
                ldap.remove_responsible_user_from_ad_group(user=login, group_dn=group_dn)
        else:
            raise WrongADRoleError(f"'group_roles': Expected `member`, `responsible` or `system_group_relation` "
                                   f"got {role['group_roles']}")

    def get_all_roles_impl(self, **kwargs):
        """
        Отдаем всех пользователей в системе и их роли
        """
        result = defaultdict(list)

        with NewLDAP() as ldap:
            all_groups = list(ldap.search(settings.AD_LDAP_DC,
                                          '(objectclass=group)',
                                          ['extensionAttribute1', 'extensionAttribute2']))

        with NewLDAP() as ldap:
            all_users = list(ldap.search(settings.AD_LDAP_DC,
                                         settings.LDAP_NOT_BLOCKED_USER_FILTER,
                                         ['sAMAccountName', 'memberOf']))

        # 'group_roles': 'responsible'
        for group_name, group_info in all_groups:
            if group_name is None or NewLDAP.fetch_extension_attribute_1(group_info) != 'true':
                continue  # https://st.yandex-team.ru/IDM-9519
            # extensionAttribute2 хранит информацию об ответственных в следующем формате: (User1,User2,...)
            # Тикет: https://st.yandex-team.ru/INFRAWIN-336
            responsible_users_str = force_text(group_info.get('extensionAttribute2', ['()'])[0])
            responsible_users = filter(None, responsible_users_str[1:-1].split(','))
            for user in responsible_users:
                result[user].append({
                    'type': 'roles_in_groups',
                    'ad_group': group_name,
                    'group_roles': 'responsible',
                })

        # 'group_roles': 'member'
        for user_name, user_info in all_users:
            if user_name is None:
                continue  # https://st.yandex-team.ru/IDM-9519
            # python-ldap возвращает атрибуты AD групп в формате
            # 'attribute': [b'value']
            user = user_info.get('sAMAccountName', [b''])[0].decode()
            for group_name in user_info.get('memberOf', []):
                result[user].append({
                    'type': 'roles_in_groups',
                    'ad_group': group_name.decode(),
                    'group_roles': 'member',
                })

        # 'group_roles': 'system_group_relation'
        for group_name, group_info in all_groups:
            if group_name is None:
                continue  # https://st.yandex-team.ru/IDM-9519
            system_slug = NewLDAP.fetch_extension_attribute_1(group_info)
            if system_slug not in ['', 'true']:
                result[settings.AD_IDM_USERNAME].append([
                    {
                        'type': 'global',
                        'role': 'system_group_relation'
                    },
                    {
                        'system': system_slug,
                        'group_dn': group_name,
                    }
                ])
        return result.items()
