import baobab
import json
from datetime import datetime


class Result(object):
    fields = (
        'server_timestamp',
        'server_datetime',
        'reqid',
        'uid',
        'ip',
        'referer',
        'url',

        'block_name',
        'block_full_name',

        'shows_count',
        'total_clicks_count',
        'children_clicks_count',
        'direct_clicks_count',

        'events',
        'raw_event_data',
    )

    def __init__(self, **kwargs):
        for field in self.fields:
            setattr(self, field, kwargs.get(field))


def process_baobab_log(reqid, rows_iterator):
    transport = baobab.TransportLevelParser()
    parser = baobab.EventsParser()
    merger = baobab.TreesAccumulateMerger()
    visitor = baobab.JoinerVisitor()
    uid = None

    for row in rows_iterator:
        baobab_context = transport.create_from_raw_redir_log_fields_dict(row.data)
        if not baobab_context:
            continue
        # uid определён не у всех событий серии: у самых первых показов он может быть пустой,
        # потому что фронт в этот момент ещё ничего не знает о пользователе. Поэтому берём его
        # из какой-то из записей (это легально, потому что все записи сгруппированы по reqid -
        # то есть пользователей фактически у всех один и тот же)
        if baobab_context.uid:
            uid = baobab_context.uid
        merger.add(baobab_context)

    merge_info = merger.join()

    for ctx in merge_info.ctxs:
        parser.parse(ctx, visitor)

    visitor.joiner.join()

    show = visitor.joiner.get_show()

    if show and show.tree and show.tree.root:
        for block in baobab.tree.bfs_iterator(show.tree.root):
            result = Result(
                reqid=reqid,
                server_timestamp=show.context.server_time,
                server_datetime=datetime.fromtimestamp(show.context.server_time).isoformat(),
                uid=uid or '',
                ip=show.context.ip,
                referer=show.context.referer,
                url=show.context.url,
                block_name=block.name,
                block_full_name=get_full_block_name(block),
                shows_count=1,
                total_clicks_count=0,
                children_clicks_count=0,
                direct_clicks_count=0,
                raw_event_data=json.dumps(show.context._event_data),
                events='',
            )
            events = []

            for event in visitor.joiner.get_events_by_block_subtree(block):
                # Тут происходит итерирование по всем событиям, произошедшим с блоком или с его потомками.
                # В текущем отчёте нас интересуют только клики.
                if isinstance(event, baobab.common.event.Click):
                    result.total_clicks_count += 1
                    if event.block_id == block.id:
                        result.direct_clicks_count += 1
                    else:
                        result.children_clicks_count += 1
                events.append(event.context._event_data)
                if not result.uid and event.context.uid:
                    result.uid = event.context.uid
            result.events = json.dumps(events)
            yield result


def get_full_block_name(block, meaning_attrs=('key', )):
    """
    Склеивает полное имя блока от корня.
    :param block: блок, объект baobab.common.Block
    :param meaning_attrs: атрибуты блоков, которые тоже нужно добавить к имени
    :return: строка с полным именем блока, например "Application / PageDetails[key=department]"
    """
    names = []
    if hasattr(block, '_block'):
        block = block._block
    current_block = block
    while current_block:
        if hasattr(current_block, '_full_name_cache'):
            names.extend(current_block._full_name_cache)
            break

        name = current_block.name
        for attr in meaning_attrs:
            if attr in current_block.attrs:
                name += '[%s=%s]' % (attr, current_block.attrs[attr])
        names.append(name)
        current_block = current_block.parent

    block._full_name_cache = names
    return ' / '.join(reversed(names))
