# coding: utf-8



import logging
import datetime

from django import http
from django import forms as django_forms

from at.aux_ import BlogsAuxiliary
from at.api.frontend import forms
from at.common import utils


common_log = logging.getLogger(__name__)
trace_logger = logging.getLogger(__name__ + '.trace')
data_log = logging.getLogger(__name__ + '.data')


aux = BlogsAuxiliary.BlogsAuxiliaryMix()


def router_view(request, method_name):
    request._tracelist.append('router_view start')
    endpoint_meta = get_endpoint_meta(method_name)

    if endpoint_meta is None:
        msg = 'method %s not found'
        return http.HttpResponseNotFound(msg % method_name)

    servant_method_name = endpoint_meta['servant_method_name']
    form_cls = endpoint_meta['form_cls']
    needs_ai = endpoint_meta['needs_ai']
    http_method = endpoint_meta['http_method']
    response_status = endpoint_meta['response_status']

    if http_method != request.method:
        msg = 'expected %s http method' % http_method
        return http.HttpResponseNotAllowed(
            permitted_methods=[http_method],
            content=msg,
        )

    try:
        method = getattr(aux, servant_method_name)
    except AttributeError:
        msg = 'method %s not found'
        return http.HttpResponseNotFound(msg % servant_method_name)

    data = dict(list(request.GET.items()))
    if request.method == 'POST':
        data.update(list(request.POST.items()))

    data_log.debug(
        'Raw method %s params %s',
        servant_method_name,
        str(list(data.items())),
    )

    if form_cls is not None:
        form = form_cls(data)
        if not form.is_valid():
            error_str = method_name + ': ' + str(list(form.errors.items()))
            common_log.warning('Validation Error. ' + error_str)
            return http.HttpResponse('<error>' + error_str + '</error>')
        else:
            # для всяких динамических полей, не описанных в форме
            data.update(form.cleaned_data)

    if request.method == 'GET':
        method_kwargs = dict(data)
    else:
        method_kwargs = {'request': dict(data)}

    if needs_ai:
        method_kwargs.update({'ai': request.ai})

    # tmp debug method
    method_kwargs.pop('_method', None)
    data_log.debug(
        'METHODCALL %s (%s) with params %s',
        method_name,
        servant_method_name,
        str(list(method_kwargs.items())),
    )

    before = datetime.datetime.now()
    method_result = method(**method_kwargs)
    passed = datetime.datetime.now() - before

    data_log.debug(
        'METHODRESPONSE %s (%s): %s',
        method_name,
        servant_method_name,
        utils.force_unicode(method_result),
    )

    msg = 'TRACEMETHOD %s (%s) executed in %d.%06d'
    trace_logger.debug(
        msg,
        method_name,
        servant_method_name,
        passed.seconds,
        passed.microseconds,
    )

    request._tracelist.append('router_view end')

    return http.HttpResponse(method_result, status=response_status)


def endpoints_view(request):
    from collections import OrderedDict
    data = OrderedDict()
    for http_method_name in endpoints:
        meta = get_endpoint_meta(http_method_name)
        data[http_method_name] = {
            'servant_method_name': meta['servant_method_name'],
            'http_method': meta['http_method'],
        }

        form_cls = meta['form_cls']
        if form_cls:
            data[http_method_name]['params'] = params = []
            for name, field in list(form_cls.declared_fields.items()):
                param = {
                    'name': name,
                    'type': type(field).__name__,
                    'required': field.required,
                }
                if isinstance(field, django_forms.ChoiceField):
                    param['choices'] = dict(field.choices)
                params.append(param)

    return http.JsonResponse(data)


def underscore_to_camelcase(underscored_string):
    return ''.join(x.capitalize() or '_' for x in underscored_string.split('_'))


def get_endpoint_meta(http_method_name):
    endpoint_meta = endpoints.get(http_method_name)
    if endpoint_meta is None:
        return

    servant_method_name = endpoint_meta['servant_method_name']
    return {
        'needs_ai': endpoint_meta.get('needs_ai', True),
        'servant_method_name': servant_method_name,
        'form_cls': endpoint_meta.get(
            'form_cls',
            getattr(forms, servant_method_name + 'Form', None)
        ),
        'http_method': endpoint_meta.get('http_method', 'GET'),
        'broken': endpoint_meta.get('broken', False),
        'response_status': endpoint_meta.get('response_status', 200),
    }


endpoints = {
    # aux_
    'ping_as_xml': {
        'servant_method_name': 'PingAsXML',
        'needs_ai': False,
    },
    'get_feed_id': {
        'servant_method_name': 'GetFeedID',
        'api': True,
        'needs_ai': False,
    },
    'authenticate': {
        'servant_method_name': 'Authenticate',
        'needs_ai': False,
    },
    'get_ml_suggest': {
        'servant_method_name': 'GetMlSuggest',
    },
    'get_current_birthdays': {
        'servant_method_name': 'GetCurrentBirthdays',
    },
    'get_post_preview': {
        'servant_method_name': 'GetPostPreview',
        'http_method': 'POST',
    },
    'friends_do_add_by_uid': {
        'servant_method_name': 'FriendsDoAddByUID',
        'http_method': 'POST',
    },
    'friends_do_add_bulk_by_uid': {
        'servant_method_name': 'FriendsDoAddBulcByUID',
        'http_method': 'POST',
    },
    'friends_do_delete_by_uid': {
        'servant_method_name': 'FriendsDoDeleteByUID',
        'http_method': 'POST',
    },
    'friends_do_delete_bulk_by_uid': {
        'servant_method_name': 'FriendsDoDeleteBulcByUID',
        'http_method': 'POST',
    },
    'all_community_block': {
        'servant_method_name': 'AllCommunityBlock',
    },

    # profile
    'get_page_layout': {
        'servant_method_name': 'GetPageLayout',
    },
    'set_widget_settings_by_id': {
        'servant_method_name': 'SetWidgetSettingsByID',
        'http_method': 'POST',
    },
    'get_recommendations': {
        'servant_method_name': 'GetRecommendations',
        'needs_ai': True,
        'broken': True,
    },
    'get_all_widget_settings': {
        'servant_method_name': 'GetAllWidgetSettings',
        'needs_ai': False,
    },
    'get_profile_categories_by_id': {
        'servant_method_name': 'GetProfileCategoriesByID',
        'api': True,
    },
    'set_profile_group_for_id': {
        'servant_method_name': 'SetProfileGroupForID',
        'http_method': 'POST',
    },
    'set_basic_feed_info_for_id': {
        'servant_method_name': 'SetBasicFeedInfoForID',
        'http_method': 'POST',
    },
    'can_change_status': {
        'servant_method_name': 'CanChangeStatus',
        'needs_ai': False,
    },

    # friends
    'unjoin_from_community': {
        'servant_method_name': 'UnjoinFromCommunity',
        'http_method': 'POST',
    },
    'join_to_community': {
        'servant_method_name': 'JoinToCommunity',
        'http_method': 'POST',
    },
    'find_relations_str': {
        'servant_method_name': 'FindRelationsStr',
        'needs_ai': False,
    },
    'find_relations': {
        'servant_method_name': 'FindRelations',
        'needs_ai': False,
    },
    'update_intra_friends': {
        'servant_method_name': 'UpdateIntraFriends',
        'http_method': 'POST',
    },
    'mass_friending_action': {
        'servant_method_name': 'MassFriendingAction',
        'http_method': 'POST',
    },
    'get_staff_tree': {
        'servant_method_name': 'GetStaffTree',
        'needs_ai': False,
        'broken': True,
    },
    'all_friends_block': {
        'servant_method_name': 'AllFriendsBlock',
    },

    # items
    'get_like_details': {
        'servant_method_name': 'GetLikeDetails',
    },
    # 'like_post': {
    #     'servant_method_name': 'LikePost',
    #     'http_method': 'POST',
    # },
    'like_post': {
        'servant_method_name': 'LikePost2',
        'http_method': 'POST',
    },
    'unlike_post': {
        'servant_method_name': 'UnlikePost',
        'http_method': 'POST',
    },
    'replies_do_add_for_id': {
        'servant_method_name': 'RepliesDoAddForID',
        'http_method': 'POST',
        'response_status': 201,
    },
    'posts_do_add': {
        'servant_method_name': 'PostsDoAdd',
        'http_method': 'POST',
        'response_status': 201,
    },
    'vote': {
        'servant_method_name': 'Vote',
        'http_method': 'POST',
    },
    'delete_comment_by_feed_id': {
        'servant_method_name': 'DeleteCommentByFeedID',
        'http_method': 'POST',
    },
    'delete_post_for_id': {
        'servant_method_name': 'DeletePostForID',
        'api': True,
        'http_method': 'POST',
    },
    'get_post_safe': {
        'servant_method_name': 'GetPostSafe',
    },
    # recent
    'received_comments_total_count': {
        'servant_method_name': 'ReceivedCommentsTotalCount',
    },
    'mark_all_notification_as_read': {
        'servant_method_name': 'MarkAllNotificationsAsRead',
        'http_method': 'POST',
    },
    'mark_comment_notifications_by_item_ids': {
        'servant_method_name': 'MarkCommentNotificationsByItemIDs',
        'http_method': 'POST',
    },

    # tags
    'get_tags_xml_paged': {
        'servant_method_name': 'GetTagsXMLPaged',
        'needs_ai': False,
    },
    'get_tag_suggest': {
        'servant_method_name': 'GetTagSuggest',
        'needs_ai': False,
    },
    'get_tag_id': {
        'servant_method_name': 'GetTagID',
        'needs_ai': False,
    },
    'get_tag_cloud_to_xml': {
        'servant_method_name': 'GetTagCloud2XML',
        'needs_ai': False,
    },
    'get_tag_cloud_typed': {
        'servant_method_name': 'GetTagCloudTyped',
        'needs_ai': False,
    },

    # community
    'update_community': {
        'servant_method_name': 'UpdateCommunityReq',
        'http_method': 'POST',
    },
    'create_community': {
        'servant_method_name': 'CreateCommunityReq',
        'http_method': 'POST',
        'response_status': 201,
    },
    'get_community_options': {
        'servant_method_name': 'GetCommunityOptions',
        'needs_ai': False
    },
    'get_rubrics_list': {
        'servant_method_name': 'GetRubricsList2',
    },
    'get_communities_rubrics': {
        'servant_method_name': 'GetCommunitiesRubrics2',
    },
    'get_user_communities_paged_light': {
        'servant_method_name': 'GetUserCommunitiesPagedLight',
    },
    'invite_to_group': {
        'servant_method_name': 'InviteToGroup',
        'http_method': 'POST',
    },
    'accept_invitation': {
        'servant_method_name': 'AcceptModerationInvitation',
        'http_method': 'POST',
    },
    'remove_from_community_group': {
        'servant_method_name': 'RemoveFromCommunityGroup',
        'http_method': 'POST',
    },
    'delete_community': {
        'servant_method_name': 'DeleteCommunity',
        'http_method': 'POST',
    },
    'restore_community': {
        'servant_method_name': 'RestoreCommunity',
        'http_method': 'POST',
    },
    'get_community_member_count': {
        'servant_method_name': 'GetCommunityMemberCount',
        'api': True,
        'needs_ai': False,
    },
    'get_my_deleted_communities': {
        'servant_method_name': 'GetMyDeletedCommunities',
    },
    'get_rubric_with_pager': {
        'servant_method_name': 'GetRubricWithPager',
        'needs_ai': False,
    },

    # statistics
    'set_post_rubric': {
        'servant_method_name': 'SetPostRubric',
        'http_method': 'POST',
    },

    # mailing
    'get_mailing_settings_xml': {
        'servant_method_name': 'GetMailingSettingsXML',
    },
    'set_mailing_settings': {
        'servant_method_name': 'SetMailingSettings',
        'http_method': 'POST',
    },
    'get_post_subscription_xml': {
        'servant_method_name': 'GetPostSubscriptionXML',
    },
    'get_blog_subscription_xml': {
        'servant_method_name': 'GetBlogSubscriptionXML',
    },
    'set_blog_subscription': {
        'servant_method_name': 'SetBlogSubscription',
        'http_method': 'POST',
    },
    'set_blog_subscription_public': {
        'servant_method_name': 'SetBlogSubscriptionPublic',
        'http_method': 'POST',
    },
    'set_post_subscription_public': {
        'servant_method_name': 'SetPostSubscriptionPublic',
        'http_method': 'POST',
    },

    # moderation
    'ban_person': {
        'servant_method_name': 'BanPerson',
        'http_method': 'POST',
        'api': True,
    },
    'unban_persons_bulk': {
        'servant_method_name': 'UnbanPersonsBulc',
        'http_method': 'POST',
    },
    'get_banned_persons': {
        'servant_method_name': 'GetBannedPersons',
    },

    # xdecorators
    'deco_get_post_rubrics': {
        'servant_method_name': 'deco_GetPostRubrics',
        'needs_ai': False,
    },
    'deco_get_subscriptions': {
        'servant_method_name': 'deco_GetSubscriptions',
    },
    'deco_get_like_counts': {
        'servant_method_name': 'deco_GetLikeCounts',
    },
    'deco_mark_moderation': {
        'servant_method_name': 'deco_MarkModeration',
    },
    'deco_poll_votes': {
        'servant_method_name': 'deco_PollVotes',
    },
    'deco_feed_items_tags': {
        'servant_method_name': 'deco_FeedItemsTags',
        'api': True,
        'needs_ai': False,
    },
    'deco_community_stat': {
        'servant_method_name': 'deco_CommunityStat',
        'needs_ai': False,
    },

    # user events
    'get_user_events': {
        'servant_method_name': 'GetUserEventsXML2',
    },
    'get_user_event_count_xml': {
        'servant_method_name': 'GetUserEventCountXML2',
        'needs_ai': False,
    },
    'mark_user_events': {
        'servant_method_name': 'MarkUserEvents2',
        'http_method': 'POST',
    },

    # trackers
    'get_trackers_projects': {
        'servant_method_name': 'GetTrackersProjects',
    },
    'get_trackers_candidates': {
        'servant_method_name': 'GetTrackersCandidates',
        'needs_ai': True,
    },

    # feed info
    'get_feed_info_bulk': {
        'servant_method_name': 'GetFeedInfoBulcXMLStr',
        'needs_ai': False,
        'api': True,
    },
    'get_feed_info': {
        'servant_method_name': 'GetFeedInfoXML',
        'needs_ai': False,
        'api': True,
    },

    # feeds
    'get_feed_calendar': {
        'servant_method_name': 'GetFeedCalendar',
    },
    'get_popular_posts': {
        'servant_method_name': 'GetPopularPosts',
    },
    'read_friends_light': {
        'servant_method_name': 'ReadFriendsLight',
        'api': True,
    },
    'get_feed_light': {
        'servant_method_name': 'GetFeed2Light',
    },
    'get_feed_on_moderation': {
        'servant_method_name': 'GetFeedOnModeration',
    },
    'get_feed_trends': {
        'servant_method_name': 'GetFeedTrends',
        'api': True,
    },
    'get_feed_pins': {
        'servant_method_name': 'GetFeedPins',
    },
    'get_comments': {
        'servant_method_name': 'GetCommentsEx',
        'api': True,
    },
    'get_comment': {
        'servant_method_name': 'GetSingleComment',
    },

    # club suggest
    'get_clubs_suggest': {
        'servant_method_name': 'GetClubsSuggest',
    },
    'get_default_clubs_suggest': {
        'servant_method_name': 'GetDefaultClubsSuggest',
    },

    'send_digest': {
        'servant_method_name': 'SendDigest',
        'needs_ai': False,
    }
}


# get alive clubs потеряли когда-то, api пятисотит
