# -*- coding: utf-8 -*-

import lxml.etree as et
from datetime import datetime
from at.api.yaru.utils.odict import odict
import json as json

from at.api.yaru import datacore


def filter_by_map(e, attrmap):
    """Вынимает из elementtree element значения, предписанные xpath'ами из attrmap.
    Возвращает линейный dict со строчными values.
    """
    attrs = odict()
    str_types = [str, str]
    for k, v in attrmap.items():
        if type(v) in str_types:
            match = e.xpath(v)
        else:
            match = v(e)
        if match:
            attrs[k] = match[0]
    return attrs


WHITELIST_META_TYPE = ['friend',  #
                       'unfriend',  #
                       'text',  #
                       'status',  #
                       'rename',
                       'join',  #
                       'unjoin',  #
                       'link',  #
                       'congratulation',
                       'photo',
                       'video',
                       'userpic',
                       ]

META_FIELDS_TO_DROP = ['poll_id', 'source', 'expires',
                       'show-as', 'body', 'body-original',
                       'title']

POST_TAG_IDS = et.XPath('item/tag-list/tag[@id]')
POST_ENTRY_META = et.XPath('item/entry/meta')
POST_ENTRY_CONTENT = et.XPath('item/entry/content')

BASE_ATTRMAP = dict(
    id=et.XPath('id/item_no/text()'),
    posted_in=et.XPath('id/uid/text()'),
    comment_id=et.XPath('id/comment_id/text()'),
    updated=et.XPath('UpdateTime/text()'),
    edited=et.XPath('item/entry/edit_time/text()'),
    published=et.XPath('StoreTime/text()'),
    published_usec=et.XPath('StoreTimeUsec/text()'),
    type_as_digit=et.XPath('PostType/text()'),
    default_type=et.XPath('item/entry/type/text()'),
    filter_type=et.XPath('item/entry/filter_type/text()'),
    agent=et.XPath('item/entry/agent/text()'),
    meta_type=et.XPath('item/entry/meta_type/text()'),
    title=et.XPath('item/entry/content/title/text()'),
    trackback_item_no=et.XPath('item/trackback/in-reply-to-id/item_no/text()'),
    trackback_uid=et.XPath('item/trackback/in-reply-to-id/uid/text()'),
    content=et.XPath('item/entry/content/body'),
    corp_meta=et.XPath('item/entry/content/corp-meta'),
    content_original=et.XPath('item/entry/content/body-original'),
    access_type=et.XPath('item/access/text()'),
    comment_count=et.XPath('CommentCount/text()'),
    comments_disabled=et.XPath('item/block-comments/text()'),
    author_uid=et.XPath('item/author/uid/text()'),
    imported_from=et.XPath('item/imported-from/text()'),
    ## выбирается тот элемент body, который имеет атрибут type
    ## но работает только для новых постов (у которых в body есть type)
    # content = 'item/entry/content/child::*[self::body or self::body-original][@type]'
    # content = 'item/entry/content/body-original | item/entry/content/body',
)
DISPLAY_ATTRMAP = dict(
    label=et.XPath('item/entry/display/label/text()'),
    style=et.XPath('item/entry/display/style/text()'),
    template=et.XPath('item/entry/display/template/text()'),
)


# META_ATTRMAPS = dict(
#    friend = dict(
#        who_id = 'item/entry/content/friender/uid/text()',
#        whom_id = 'item/entry/content/friendee/uid/text()',
#    ),
#    #XXX: incomplete
#    #photo = dict(
#    #    image = 'item/entry/content/images/image',
#    #)
# )


class DummyTagsCache(object):
    def __contains__(self, _):
        return False

    def fill_cache(self, _):
        return

    def get(self, key):
        return []


class TagsCache(object):
    POST_TAG_ID = et.XPath('id/text()')
    POST_TAG_TITLE = et.XPath('title-tag/text()')
    POST_TAGS = et.XPath('post-tag')

    def __init__(self):
        self.cache = {}
        self.cache_keys = set()

    def __contains__(self, item):
        return item in self.cache

    def mk_key(self, tag_id):
        feed_id, item_no = tag_id
        return '%s.%s' % (feed_id, item_no)

    def fill_cache(self, tag_ids):
        keys = [self.mk_key(tag_id) for tag_id in tag_ids]
        tag_ids = set(keys)
        absent_tag_ids = tag_ids.difference(self.cache_keys)
        if absent_tag_ids:
            tags_with_names = datacore.tags_for_posts(','.join(keys))
            tags_elems = TagsCache.POST_TAGS(et.XML(tags_with_names))
            # из всех элементов построить дикт вида {id: title-tag}
            for t in tags_elems:
                if t.attrib['feed-item'] not in self.cache:
                    self.cache[t.attrib['feed-item']] = {}
                self.cache[t.attrib['feed-item']][TagsCache.POST_TAG_ID(t)[0]] = \
                    (
                    TagsCache.POST_TAG_ID(t)[0], TagsCache.POST_TAG_TITLE(t)[0])
            self.cache_keys = set(self.cache)

    def get(self, tag_id):
        key = self.mk_key(tag_id)
        if key not in self.cache_keys:
            self.fill_cache([tag_id])
        return list(self.cache.get(key, {}).values())


class PreviewTagsCache(TagsCache):
    def __init__(self, tags_key, tags):
        super(type(self), self).__init__()
        self.cache_keys = set([tags_key])
        self.cache[tags_key] = dict()
        for tag_id, tag_text in tags:
            self.cache[tags_key][str(tag_id)] = (str(tag_id), tag_text)


def get_posts_tags(elems):
    tags_cache = TagsCache()
    tag_ids = []
    for elem in elems:
        posted_in = BASE_ATTRMAP['posted_in'](elem)[0]
        post_id = BASE_ATTRMAP['id'](elem)[0]
        tag_ids.append((posted_in, post_id))
    tags_cache.fill_cache(tag_ids)
    return tags_cache


def parse_likes(likes):
    return [dict(like.attrib) for like in likes]


def parse_post(elem, tags_cache=None):
    """По технической xml-структуре я.ру-записи строит
    dict с избранными (по POST_ATTR_MAP) атрибутами.
    """
    d = filter_by_map(elem, BASE_ATTRMAP)
    if 'meta_type' in d:
        # Это запись новой структуры (c meta и filter types и т.д.)
        match = POST_ENTRY_META(elem)
        d.raw_meta = match[0] if match else None
    else:
        # Это запись старой структуры

        # meta и filter types восстанавливаются по type
        if 'default_type' not in d:  # скорее всего пост был удалён
            return None

        d.meta_type = d.default_type
        d.filter_type = d.get('filter_type', d.default_type)

        d.raw_meta = POST_ENTRY_CONTENT(elem)[0]

    if d.meta_type not in WHITELIST_META_TYPE:
        d.raw_meta = None

    if elem.xpath("likes/*"):
        d["likes"] = parse_likes(elem.xpath("likes/*"))
    # TODO упрости меня! Очень неоптимально!
    if not tags_cache:
        tags_cache = TagsCache()
    tag_ids = [(d.posted_in, d.id)]
    tags_cache.fill_cache(tag_ids)
    d.tags = tags_cache.get((d.posted_in, d.id))
    # TODO: добавить очистку meta от пустых (или None) значений

    # print text

    # преобразования типов и обработка особых случаев
    # TODO: защита от ошибок преобразований
    d.published = datetime.utcfromtimestamp(float(d.published))
    if 'edited' not in d:
        d.edited = None
    else:
        d.edited = datetime.utcfromtimestamp(float(d.edited))
    if 'updated' not in d:
        d.updated = 0
    d.updated = datetime.utcfromtimestamp(float(d.updated))
    # для старых записей UpdateTime возвращается нулевым
    d.updated = max(d.updated, d.published)

    display = filter_by_map(elem, DISPLAY_ATTRMAP)
    if display:
        d.display = display

    d['access_type'] = d.get('access_type', 'public')
    return d


def retrieve_trends(ai, feed_id, offset, count, post_type=''):
    text = datacore.retrieve_trends(ai, feed_id, offset, count,
                                    post_type)
    items = et.XML(text).findall('feed/item')
    posts = []
    tags_cache = get_posts_tags(items)
    for item in items:
        parsed_post = parse_post(item, tags_cache=tags_cache)
        if parsed_post:
            posts.append(parsed_post)
    return posts


def retrieve_post(ai, feed_id, id, fetch_tags=True):
    text = datacore.retrieve_post(ai, feed_id, id)
    if text:
        item = et.XML(text).xpath('item')
        if not item:
            return
        post = item[0]
        return parse_post(post,
                          tags_cache=TagsCache() if fetch_tags else DummyTagsCache())
    return None


def retrieve_posts(ai, feed_id, offset, count, escape_html=False,
                   post_type='', fetch_tags=True, cat_id=0, min_time=0,
                   max_time=0):
    xml = datacore.retrieve_posts(ai, feed_id, offset, count, escape_html,
                                  post_type, cat_id, min_time, max_time)
    items = et.XML(xml).findall('feed/item')
    posts = []
    tags_cache = get_posts_tags(items) if fetch_tags else DummyTagsCache()
    for item in items:
        parsed_post = parse_post(item, tags_cache=tags_cache)
        if parsed_post:
            posts.append(parsed_post)
    return posts


def retrieve_feed_tags(feed_id, scheme):
    xml = datacore.retrieve_feed_tags(feed_id)
    tree = et.XML(xml)
    tags = []
    for elem in tree.getchildren():
        tags.append({"term": elem.attrib["name"],
                     "scheme": scheme,
                     "id": elem.attrib["id"],
                     "count": elem.attrib["count"]})
    return tags


def retrieve_friends_posts(ai, feed, since, count, post_type='',
                           fetch_tags=True):
    xml = datacore.retrieve_friends_posts(ai, feed, since, count)
    items = et.XML(xml).findall('feed/item')
    posts = []
    tags_cache = get_posts_tags(items) if fetch_tags else DummyTagsCache()
    for item in items:
        parsed_post = parse_post(item, tags_cache)
        if parsed_post:
            posts.append(parsed_post)
    return posts


def get_preview_tags(items, tags):
    tags = enumerate(tags.split(','))
    elem = items[0]
    posted_in = BASE_ATTRMAP['posted_in'](elem)[0]
    post_id = BASE_ATTRMAP['id'](elem)[0]
    return PreviewTagsCache('%s.%s' % (posted_in, post_id), tags)


def create_post(*args, **kw):
    text = datacore.store_post(*args, **kw)
    if text:
        items = et.XML(text).xpath('item')
        if not kw.get('preview'):
            tags_cache = get_posts_tags(items)
        else:
            if 'tags' in kw['params'] and kw['params']['tags'].strip():
                tags_cache = get_preview_tags(items, kw['params']['tags'])
            else:
                tags_cache = None
        return parse_post(items[0], tags_cache)
    return None


def ban_person_in_feed(ai, feed_id, request):
    ban_list = json.loads(request.body)
    answer = {}
    for person, delete_posts in list(ban_list.items()):
        answer[person] = datacore.ban_person_in_feed(ai, feed_id,
                                                     int(person), delete_posts)
    return json.dumps(answer)


def subscribe_post(ai, feed_id, post_no, subscribe):
    xml = et.XML(
        datacore.subscribe_post(ai, feed_id, post_no, subscribe))
    return json.dumps(dict(list(xml.attrib.items())))
