# -*- coding: utf-8 -*-


from django.core.urlresolvers import reverse

from at.common import exceptions
from at.api.yaru import datacore
from at.api.yaru.instances import Blog, Club
from at.api.yaru.person.models import Profile
from at.api.yaru.person.store import PersonAtomStore
from at.api.yaru import urlgen
from at.api.yaru.errors import ClubNotFound
from at.api.yaru.templates import YaruFeed, YaruProfile


class ClubStore(object):
    def __init__(self, ai, feed, format='xml', abs_url=lambda url: url):
        self.abs_url = abs_url
        self.ai = ai
        self.feed = feed
        self.format = format

    def _list(self, typ, next_url_name, page_no, per_page, total_members, members):
        items_left = total_members - page_no * per_page
        links = {}
        if typ == 'clubs':
            links['person'] = [{'href': self.abs_url(reverse('blog_person', args=[self.feed.get_id()]))}]
        else:
            links['club'] = [{'href': self.abs_url(reverse('club', args=[self.feed.get_id()]))}]
        if items_left > 0:
            links['next'] = [{'href': '%s?p=%d&count=%d' % (self.abs_url(reverse(next_url_name, args=[self.feed.get_id()])), page_no+1, per_page)}]

        instance = Club if typ == 'clubs' else Blog

        entries = []
        for member in members or []:
            profile = Profile.from_dict(member)
            entries.append(YaruProfile(instance(member['id'],
                                                profile=profile),
                                       profile,
                                       is_short=True))

        return (entries, links)

    def get(self, feed, with_tags=False):
        club = Profile.from_feed_id(feed.feed_id)
        if not club:
            raise ClubNotFound(feed.get_id())
        if with_tags:
            club.get_tags(urlgen.tag_space_url(feed, abs_url=self.abs_url))
        renderer = YaruProfile(Club(club['id'], profile=club), club)
        return renderer

    def get_atomsvc(self, feed):
        store = PersonAtomStore()
        return store.get_atomsvc(feed)

    def list_members(self,  page_no=1, per_page=50):
        return YaruFeed('urn:ya.ru:club/%s/members' % self.feed.get_id(),
                        *self._list('blogs', 'club_list_members', page_no, per_page, *datacore.club_members(
                            self.feed.get_id(), page_no, per_page)))

    def list_moderators(self,  page_no=1, per_page=50):
        return YaruFeed('urn:ya.ru:club/%s/moderators' % self.feed.get_id(),
                        *self._list('blogs', 'club_list_moderators', page_no, per_page, *datacore.club_moderators(
                            self.feed.get_id(), page_no, per_page)))

    def get_owner(self):
        # FIXME не надо передавать туда blog_person
        try:
            entries, _ = self._list('blogs', 'blog_person', 0, 1, *datacore.club_owner(
                self.feed.get_id()))
        except exceptions.AccessDenied:
            return None
        else:
            return entries[0].feed if entries else None

    def member_of_clubs(self, page_no=1, per_page=50):
        return YaruFeed('urn:ya.ru:person/%s/club/member' % self.feed.get_id(),
                        *self._list('clubs', 'person_member_of_clubs', page_no, per_page, *datacore.member_of_clubs(
                            self.feed.get_id(), page_no, per_page)))

    def owner_of_clubs(self, page_no=1, per_page=50):
        return YaruFeed('urn:ya.ru:person/%s/club/owner' % self.feed.get_id(),
                        *self._list('clubs', 'person_owner_of_clubs', page_no, per_page, *datacore.owner_of_clubs(
                            self.feed.get_id(), page_no, per_page)))

    def moderator_of_clubs(self, page_no=1, per_page=50):
        return YaruFeed('urn:ya.ru:person/%s/club/moderator' % self.feed.get_id(),
                        *self._list('clubs', 'person_moderator_of_clubs', page_no, per_page, *datacore.moderator_of_clubs(
                            self.feed.get_id(), page_no, per_page)))
