# -*- coding: utf-8 -*-

from django.http import HttpResponse

from at.api.yaru.instances import Club, Blog
from at.api.yaru import activity
from at.api.yaru.club.store import ClubStore
from at.api.yaru.utils.django_helpers import to_canonical, get_pagination, \
    require_permission, get_format, JSON_CONTENT_TYPE


def _get_format(request):
    format, forced = get_format(
        request,
        [
            'application/x-yaru+json',
            'application/x-yaru+xml'
        ],
        default_type='application/x-yaru+json'
    )
    return format, forced


@to_canonical(Club, 'ban')
@require_permission('yaru:write', allow_anonymous=False)
def ban_person_in_club(request, feed_id):
    answer = activity.ban_person_in_feed(request.ai, feed_id, request)
    return HttpResponse(answer, content_type=JSON_CONTENT_TYPE)


@to_canonical(Club, 'club')
@require_permission('yaru:read', allow_anonymous=True)
def get_club(request, feed_id, with_tags=False):
    format, forced = _get_format(request)
    club = Club(feed_id, abs_url=request.build_absolute_uri)
    store = ClubStore(request.ai, club, abs_url=request.build_absolute_uri)
    renderer = store.get(club, with_tags)
    if 'xml' in format:
        entry = renderer.to_xml(forced, render=True)
    else:
        entry = renderer.to_json(forced, render=True)
    return HttpResponse(entry, content_type='%s; type=club' % format)



@to_canonical(Club, 'club_with_tags')
@require_permission('yaru:read', allow_anonymous=True)
def get_club_with_tags(request, feed_id):
    return get_club(request, feed_id=feed_id, with_tags=True)


@to_canonical(Club, 'club_list_members')
@require_permission('yaru:read', allow_anonymous=True)
def list_members(request, feed_id):
    format, forced = _get_format(request)
    club = Club(feed_id, abs_url=request.build_absolute_uri)
    store = ClubStore(request.ai, club, abs_url=request.build_absolute_uri)
    page_no, page_size = get_pagination(request)
    renderer = store.list_members(page_no, page_size)
    if 'xml' in format:
        members = renderer.to_xml(forced, 'persons', render=True)
    else:
        members = renderer.to_json(forced, 'persons', render=True)
    return HttpResponse(members, content_type='%s; type=persons' % format)


@to_canonical(Club, 'club_list_moderators')
@require_permission('yaru:read', allow_anonymous=True)
def list_moderators(request, feed_id):
    format, forced = _get_format(request)
    club = Club(feed_id, abs_url=request.build_absolute_uri)
    store = ClubStore(request.ai, club, abs_url=request.build_absolute_uri)
    page_no, page_size = get_pagination(request)
    renderer = store.list_moderators(page_no, page_size)
    if 'xml' in format:
        members = renderer.to_xml(forced, 'persons', render=True)
    else:
        members = renderer.to_json(forced, 'persons', render=True)
    return HttpResponse(members, content_type='%s; type=persons' % format)


@to_canonical(Blog, 'person_member_of_clubs')
@require_permission('yaru:read', allow_anonymous=True)
def member_of_clubs(request, feed_id):
    format, forced = _get_format(request)
    blog = Blog(feed_id, abs_url=request.build_absolute_uri)
    store = ClubStore(request.ai, blog, abs_url=request.build_absolute_uri)
    page_no, page_size = get_pagination(request)
    renderer = store.member_of_clubs(page_no, page_size)
    if 'xml' in format:
        clubs = renderer.to_xml(forced, 'clubs', render=True)
    else:
        clubs = renderer.to_json(forced, 'clubs', render=True)
    return HttpResponse(clubs, content_type='%s; type=clubs' % format)


@to_canonical(Blog, 'person_owner_of_clubs')
@require_permission('yaru:read', allow_anonymous=True)
def owner_of_clubs(request, feed_id):
    format, forced = _get_format(request)
    blog = Blog(feed_id, abs_url=request.build_absolute_uri)
    store = ClubStore(request.ai, blog, abs_url=request.build_absolute_uri)
    page_no, page_size = get_pagination(request)
    renderer = store.owner_of_clubs(page_no, page_size)
    if 'xml' in format:
        clubs = renderer.to_xml(forced, 'clubs', render=True)
    else:
        clubs = renderer.to_json(forced, 'clubs', render=True)
    return HttpResponse(clubs, content_type='%s; type=clubs' % format)


@to_canonical(Blog, 'person_moderator_of_clubs')
@require_permission('yaru:read', allow_anonymous=True)
def moderator_of_clubs(request, feed_id):
    format, forced = _get_format(request)
    blog = Blog(feed_id, abs_url=request.build_absolute_uri)
    store = ClubStore(request.ai, blog, abs_url=request.build_absolute_uri)
    page_no, page_size = get_pagination(request)
    renderer = store.moderator_of_clubs(page_no, page_size)
    if 'xml' in format:
        clubs = renderer.to_xml(forced, 'clubs', render=True)
    else:
        clubs = renderer.to_json(forced, 'clubs', render=True)
    return HttpResponse(clubs, content_type='%s; type=clubs' % format)
