# -*- coding: utf-8 -*-

import re
from datetime import datetime
import lxml.etree as et

from django.conf import settings

from at.api.yaru import datacore
from at.api.yaru.atom import strftime
from at.api.yaru.errors import InvalidInputError, AccessError, EntryNotFound
from at.api.yaru import atom, atomgen
from at.api.yaru.activity import parse_post, DummyTagsCache
from at.api.yaru.post.store import parse_content_from_xml, content_from_yaru_to_atom
from at.api.yaru.templates import AtomComments
from at.api.yaru.instances import FeedsCache


RE_REF = re.compile(r'^urn:ya.ru/comment/(?P<feed_id>[\w-]+)/(?P<post_no>\d+)/(?P<parent_id>\d+)/?$')


class CommentAtomStore(object):
    def __init__(self, ai, feed, post_id, abs_url=lambda url: url):
        self.feed = feed
        self.post_id = post_id
        self.ai = ai
        self.abs_url = abs_url

    def _value_id(self, id):
        #FIXME: может валиться на преобразовании id в строку,
        # если id не правильный -- не число, не ascii
        # так как id может приходит от клиента, нужно проверять и защищаться

        # Experimental URN (http://www.ietf.org/rfc/rfc2141.txt)
        #return 'urn:X-ya-ru:%s,%s' % (self.feed_id, id)
        # tag URI (http://www.faqs.org/rfcs/rfc4151.html)
        return 'urn:ya.ru:post/%s/%s' % (self.feed.get_id(), id)

    def _comment_id(self, post_id, comment_id):
        return 'urn:ya.ru:comment/%s/%s/%s' % (self.feed.get_id(), post_id, comment_id)

    def _create_entry(self, post_id, item_id, content_type, content, reply_to=None, author=None):
        entry = atomgen.create_entry(self._comment_id(post_id, item_id), # entry id
                                     '', # title
                                     '', # updated
                                     )
        if author:
            entry.author = atom.Element('author')
            entry.author.uid = str(author.get_id())
            entry.author.login = author.get_login()
            entry.author.name = author.get_name()
        if not reply_to:
            href = self.feed.get_post_relations(post_id)['self'][0]['href']
            ref = self._value_id(post_id)
        else:
            href = self.feed.get_comment_link(self.feed.get_id(), reply_to)
            ref = self._comment_id(post_id, reply_to)
        entry.append(atom.Element(atomgen._qname('in-reply-to', ns='thr'),
                                  href=href,
                                  ref=ref))
        entry.content = atomgen.E('content', content)
        return entry

    def _serialize_more(self, more):
        if more is not None:
            data = {}
            t = lambda x: x.text if x is not None else None
            data["from"] = t(more.find('from'))
            data["count"] = t(more.find('extra-count'))
            return data

    def _serialize_reply(self, reply, feeds_cache=None):

        feeds_cache = feeds_cache or FeedsCache(abs_url=self.abs_url)
        data = {}
        info = reply.find('item')
        deleted = reply.find('item/deleted')
        if info is None or deleted is not None:
            return data

        t = lambda x: x.text if x is not None else None
        comment_id = t(info.find('id/comment_id'))
        item_no = t(info.find('id/item_no'))
        data['item_no'] = int(item_no)
        data['comment_id'] = int(comment_id)
        data['id'] = self._comment_id(item_no,
                                      comment_id)
        pub = t(info.find('StoreTime')) or ''
        if pub.isdigit():
            data['published'] = strftime(datetime.utcfromtimestamp(float(pub)))
        else:
            data['published'] = ''
        data['author_uid'] = t(info.find('item/author/uid'))
        data['parent'] = t(info.find('item/parent/asStringFull'))
        if data['parent'] is None:
            # Construct asStringFull from parts
            as_string_part = t(info.find('item/parent/asString'))
            parent_comment_id = t(info.find('item/parent/comment_id'))
            if all((as_string_part, parent_comment_id)):
                data['parent'] = "%s.%s" % (as_string_part, parent_comment_id)
        data['reply_type'] = t(info.find('item/entry/type'))
        try:
            data['reply_count'] = int(t(info.find('CommentCount')))
        except (TypeError, ValueError):
            data['reply_count'] = 0
        more = reply.find('more')
        if more:
            data['more'] = self._serialize_more(more)
        trackback = info.find('item/trackback/in-reply-to-id/asStringFull')
        data['trackback'] = t(trackback) or ''

        content = info.find('item/entry/content')
        if content is not None:
            title = content.find('title')
            data['title'] = t(title) or ''

            body = content.find('body')
            if body is not None:
                body.set('type', 'text/html')   # костыли
                data['content_type'], data['content'] = parse_content_from_xml(body, False)

        data['replies'] = []
        content_hide_flags = list((flag_name, reply.find('item/%s' % flag_name))
            for flag_name in ('deleted', 'screened'))
        for flag_name, flag_value in content_hide_flags:
            if flag_value is not None:
                data[flag_name] = True
                data['content'] = ''
            else:
                data[flag_name] = False
        author_uid = data.get('author_uid', 0)
        if author_uid and all((flag_value is None for flag_name, flag_value
                in content_hide_flags)):
            author = feeds_cache.get(int(data.get('author_uid', 0)))
            data['author'] = {
                    'login': author.get_login(),
                    'name': author.get_name(),
                    'uri': author.get_self_link(),
                    'mood': author.get_profile()['mood'],
                    'sex': author.get_profile()['sex']
                }
            # У социального пользователя пустой login
            if not author.get_login():
                data['author']['display_name'] = author.get_display_name()


        for new_reply in reply.iterfind('reply'):
            data['replies'].append(self._serialize_reply(new_reply, feeds_cache))

        data['links'] = self.feed.get_comment_relations(item_no, comment_id)
        return data

    def get_info(self, from_id, parent_id, limit, with_parent=False, escape_html=False):
        comments = datacore.retrieve_comments(
            ai=self.ai,
            feed_id=self.feed.get_id(),
            item_no=self.post_id,
            parent_id=parent_id,
            from_id=from_id,
            limit=limit,
            with_parent=with_parent,
            escape_html=escape_html,
        )
        xml = et.fromstring(comments)
        top_replies = xml.xpath('/aux/replies/reply')
        feeds_cache = FeedsCache(abs_url=self.abs_url)
        t = lambda x: x.text if x is not None else 0
        get_uid = lambda xml_item: int(t(xml_item.find('item/item/author/uid')))
        replies = []
        uids = []
        def fill_replies_and_uids(r):
            replies.append(self._serialize_reply(r, feeds_cache=feeds_cache))
            uids.append(get_uid(r))
        list(map(fill_replies_and_uids, top_replies))
        feeds_cache.fill_cache(uids)
        post = {'feed_id': self.feed.get_id(),
                'id': self._value_id(self.post_id),
                'replies': replies}
        more = xml.find('replies/more')
        if more:
            post['more'] = self._serialize_more(more)
        return AtomComments(post)

    def get_single_comment(self, comment_id, escape_html=False):
        return self.get_info(
                settings.UNDEFINED_COMMENT_ID,
                comment_id,
                0,
                True,
                escape_html=escape_html)

    def store(self, agent, input_entry, escape_html=False):
        in_reply_to_elem = input_entry.find(atomgen._qname('in-reply-to', ns='thr'))
        in_reply_to = in_reply_to_elem.attrib.get(
            'ref') if in_reply_to_elem is not None else None
        if in_reply_to:
            # in_reply_to может быть просто числом, а может быть urn:ya.ru:comment/feed_id/post_no/parent_id
            if not in_reply_to.isdigit():
                m = RE_REF.match(in_reply_to)
                if not m:
                    raise InvalidInputError("ref is invalid") # FIXME magic number
                #print m.groups()
                feed_id, post_no, parent_id = m.groups()
                if self.feed.get_id() != feed_id and self.feed.get_login() != feed_id:
                    raise InvalidInputError("feed_id in url != feed_id in ref")
                if self.post_id != post_no:
                    raise InvalidInputError("post_no in url != post_no in ref")
                in_reply_to = parent_id
            parent = datacore.retrieve_comment(self.ai,
                                               int(self.feed.get_id()),
                                               self.post_id,
                                               in_reply_to,
                                               escape_html=escape_html,
                                               )
            if not parent:
                raise EntryNotFound("parent: %s" % (in_reply_to,))

        content_type, content_text = parse_content_from_xml(input_entry.content)

        result_xml = datacore.create_comment(self.ai,
                                             int(self.feed.get_id()),
                                             self.post_id,
                                             content_text,
                                             parent_id=in_reply_to,
                                             agent=agent,
                                             comment_id=input_entry.id,
                                             escape_html=escape_html,
                                             )
        result_tree = et.XML(result_xml)
        comment_elem = result_tree.xpath('item')[0]
        comment = parse_post(comment_elem, DummyTagsCache())
        """
        '<aux hostname="morbo.dev.yandex.net"><new-reply-id>77</new-reply-id><item-no>72</item-no><trackback-item-no>18446744073709551615</trackback-item-no><retpath>/replies.xml?item_no=72&amp;#reply-77</retpath><status code="Success" hostname="morbo.dev.yandex.net"/></aux>'
        """
        if r"NotFound: Yandex.Blog.Blogger.NotFound(dsc='Could not find post')" in result_xml:
            raise EntryNotFound(self.post_id)
        if 'comment_id' not in comment:
            raise AccessError()
        content_type, content_text = content_from_yaru_to_atom(comment['content'])
        fc = FeedsCache()
        return self._create_entry(self.post_id, comment['comment_id'], content_type, content_text, in_reply_to, fc.get(comment['author_uid']))
