# -*- coding: utf-8 -*-

from django.http import HttpResponse, HttpResponseBadRequest
from lxml.etree import XMLSyntaxError

from at.api.yaru import atom, datacore
from at.api.yaru.utils.django_helpers import response_as_atom_entry, \
    require_permission
from at.api.yaru.instances import Blog, Club
from at.api.yaru.comment.store import CommentAtomStore
from at.api.yaru.utils.django_helpers import get_format


def _post_comment(request, feed, post_no, escape_html=False):
    comment_store = CommentAtomStore(request.ai, feed, post_no)
    try:
        input_entry = atom.ATOM(request.body)
    except XMLSyntaxError:
        return HttpResponseBadRequest("Malformed XML")
    c = comment_store.store(None, input_entry, escape_html=escape_html)
    return response_as_atom_entry(c, status=201)


def _get_comment(request, feed, post_no, comment_id=None, escape_html=False):
    yaru_json = 'application/x-yaru-atom+json'
    yaru_xml = 'application/atom+xml'
    fmt, forced = get_format(
        request=request,
        supported_types=[
            yaru_json,
            yaru_xml
        ],
        default_type=yaru_json,
    )

    comment_store = CommentAtomStore(request.ai, feed, post_no,
                                     abs_url=request.build_absolute_uri)
    if comment_id:
        renderer = comment_store.get_single_comment(comment_id, escape_html=escape_html)
    else:
        params = {}
        for key, default_value in (
                ('limit', None),
                ('from_id', None),
                ('parent_id', None)
        ):
            try:
                params[key] = int(request.GET.get(key))
            except (ValueError, TypeError):
                params[key] = default_value
        renderer = comment_store.get_info(**params)
    if 'xml' in fmt:
        return response_as_atom_entry(renderer.to_xml(render=True))
    else:
        return HttpResponse(renderer.to_json(render=True),
                            content_type=yaru_json)


@require_permission('yaru:write')
def delete_blog_comment(request, feed_id, post_no, comment_id):
    return _delete_comment(request.ai, Blog(feed_id), post_no, comment_id)


@require_permission('yaru:write')
def delete_club_comment(request, feed_id, post_no, comment_id):
    return _delete_comment(request.ai, Club(feed_id), post_no, comment_id)


def _delete_comment(ai, feed, post_no, comment_id):
    datacore.delete_comment(ai, feed.get_id(), post_no, comment_id)
    return HttpResponse('', status=200)


@require_permission('yaru:write')
def post_blog_comment(request, feed_id, post_no):
    return _post_comment(request, Blog(feed_id), post_no)


@require_permission('yaru:read', allow_anonymous=True)
def get_blog_comment(request, feed_id, post_no):
    return _get_comment(request, Blog(feed_id), post_no)


@require_permission('yaru:write')
def post_club_comment(request, feed_id, post_no):
    return _post_comment(request, Club(feed_id), post_no)


@require_permission('yaru:read', allow_anonymous=True)
def get_club_comment(request, feed_id, post_no):
    return _get_comment(request, Club(feed_id), post_no)


@require_permission('yaru:read', allow_anonymous=True)
def get_blog_single_comment(request, feed_id, post_no, comment_id):
    return _get_comment(request, Blog(feed_id), post_no, comment_id)


@require_permission('yaru:read', allow_anonymous=True)
def get_club_single_comment(request, feed_id, post_no, comment_id):
    return _get_comment(request, Club(feed_id), post_no, comment_id)
