# -*- coding: utf-8 -*-

import logging

from django.utils.encoding import force_str

from at.aux_ import BlogsAuxiliary
from at.aux_ import Api
from at.common import exceptions
from at.common import assertions
from at.api.yaru.utils.py2xml import etree2py
from at.api.yaru.errors import InvalidInputError, ClubNotFound, BlogNotFound, \
    EntryNotFound, NotFound
import blackbox

import lxml.etree as et

aux = BlogsAuxiliary.BlogsAuxiliaryMix()
api = Api.Api()

log = logging.getLogger(__name__)


def get_club_id(str_id):
    str_id = str(str_id)
    if str_id.isdigit():
        return str_id
    xml = aux.GetFeedID(str_id)
    tree = et.fromstring(xml)
    feed_ids = tree.xpath('//feed_id')
    if not feed_ids:
        raise ClubNotFound(str_id)
    feed_id = feed_ids.pop(0)
    if feed_id.attrib.get('status') and feed_id.attrib.get(
            'status') == 'NotFound':
        raise ClubNotFound(str_id)
    return feed_id.text


def get_blog_id(str_id):
    if str_id.isdigit():
        return str_id
    uid = blackbox.uid(str_id)
    if not uid:
        raise BlogNotFound(str_id)
    return str(uid)


def get_club_name(club_id):
    fp = retrieve_fast_profile(club_id)
    return str(fp['title'])


def force_str_dict_items(d):
    return ((force_str(k), force_str(v or '')) for k, v in d.items())


def retrieve_fast_profile(id):
    try:
        return aux.GetFeedInfoXML(int(id))
    except exceptions.NotFound:
        return None


def retrieve_fast_profiles(ids):
    """Получить "быстрые" профили пользователей Я.ру"""
    return aux.GetFeedInfoBulcXMLStr(
        ','.join(str(_id) for _id in ids)
    )


def retrieve_trends(ai, feed_id, offset, count, post_type):
    return aux.GetFeedTrends(
        ai=ai,
        source_id=int(feed_id),
        post_types_str=post_type,
        count=int(count),
        tb=int(offset),
    )


def tags_for_posts(posts):
    return aux.deco_FeedItemsTags(posts)


def retrieve_feed_tags(feed_id):
    return api.GetTagsByFeed(feed_id)


PROFILE_GROUP_LIST = ','.join(sorted([
    'Special',
    'Abilities',
    'CV',
    'Favourites',
    'Minimum',
    'Personal',
]))


def retrieve_slow_profile(ai, feed_id):
    return aux.GetProfileCategoriesByID(
        ai=ai,
        feed_id=int(feed_id),
        categories=PROFILE_GROUP_LIST,
    )


def retrieve_post(ai, feed_id, item_no):
    try:
        return api.GetPost(ai, int(feed_id), int(item_no))
    except exceptions.NotFound:
        raise EntryNotFound(item_no)
    except assertions.AssertionFailure as e:
        raise InvalidInputError.from_corba_assertions(e.assertions)


def retrieve_posts(ai, feed_id, offset, count, escape_html=False,
                   post_type='', cat_id=0, min_time=0, max_time=0):
    return api.GetPosts(
        ai=ai,
        source_id=int(feed_id),
        escape_html=bool(escape_html),
        post_types_str=str(post_type),
        cat_id=int(cat_id),
        count=int(count),
        tb=int(offset),
        min_store_time=int(min_time),
        max_store_time=int(max_time),
    )


def delete_post(ai, feed_id, post_id):
    return aux.DeletePostForID(ai, int(feed_id), int(post_id))


def meta_struct(meta_type, meta_dict):
    meta_class = dict(
        offline=Yandex().Blog.OfflineMeta,
        link=Yandex().Blog.LinkMeta,
        custom=Yandex().Blog.CustomMeta,
        friend=Yandex().Blog.RelationMeta,
        join=Yandex().Blog.RelationMeta,
        unfriend=Yandex().Blog.RelationMeta,
        unjoin=Yandex().Blog.RelationMeta,
        congratulation=Yandex().Blog.CongratulationMeta,
    )
    if meta_type == 'poll':
        poll_type = meta_dict['poll_type']
        if poll_type == 'single':
            meta_dict['poll_type'] = Yandex().Blog.PollType._item(
                0)  # POLLTYPE_SINGLE
        elif poll_type == 'multi':
            meta_dict['poll_type'] = Yandex().Blog.PollType._item(
                1)  # POLLTYPE_MULTI
        if 'expires' not in meta_dict:
            meta_dict['expires'] = ''
    elif meta_type == 'link':
        if 'source' not in meta_dict:
            meta_dict['source'] = ''
    elif meta_type in ['friend', 'join', 'unfriend', 'unjoin']:
        meta_dict = {'friendee': meta_dict['uid'],
                     'friender': int(meta_dict['feed'].get_id())}
    elif meta_type == 'congratulation':
        meta_dict['whom'] = meta_dict['whom']['uid'] if meta_dict[
            'whom'] else 0

    if meta_type in meta_class:
        meta_instance = meta_class[meta_type](**meta_dict)
        return meta_instance

    # TODO: отличать нереализованные (not-implemented) мета-типы от неподдерживаемых
    raise NotImplementedError(
        'meta_type "%s" support not implemented yet' % meta_type)


def store_post(ai, feed_id, type, meta_type, meta, params, *args, **kwargs):
    meta = meta and meta_struct(meta_type,
                                dict(force_str_dict_items(meta))) or None
    param_items = params and list(force_str_dict_items(params)) or None
    try:
        result = api.CreatePost(
            ai=ai,
            feed_id=int(feed_id),
            post_type=force_str(type),
            meta=meta,
            params=param_items,
        )
        return result
    except assertions.AssertionFailure as e:
        raise InvalidInputError.from_corba_assertions(e.assertions)
    except exceptions.InvalidParams as e:
        raise InvalidInputError(str(e))


DEFAULT_LIMIT = 3000


def retrieve_comments(ai, feed_id, item_no, parent_id=None, from_id=None,
                      limit=None, with_parent=False, escape_html=False):
    try:
        if limit is None:
            limit = DEFAULT_LIMIT
        return aux.GetCommentsEx(
            ai=ai,
            feed_id=int(feed_id),
            item_no=int(item_no),
            parent_id=int(parent_id or 0),
            last_seen_id=int(from_id or 0),
            with_parent=with_parent,
            limit=int(limit),
            escape_html=bool(escape_html),
        )
    except exceptions.NotFound as e:
        raise NotFound("Invalid bounds %s" % e.dsc)
    except assertions.AssertionFailure as e:
        raise InvalidInputError.from_corba_assertions(e.assertions)


def retrieve_comment(ai, feed_id, item_no, comment_id, escape_html=False):
    try:
        return aux.GetSingleComment(
            ai=ai,
            feed_id=int(feed_id),
            item_no=int(item_no),
            comment_id=int(comment_id),
            escape_html=bool(escape_html),
        )
    except exceptions.NotFound:
        return None
    except assertions.AssertionFailure as e:
        raise InvalidInputError.from_corba_assertions(e.assertions)


def delete_comment(ai, feed_id, item_no, comment_id):
    try:
        aux.DeleteCommentByFeedID(
            ai=ai,
            request={
                'feed_id': int(feed_id),
                'item_no': int(item_no),
                'comment_id': int(comment_id),
            }
        )
    except exceptions.NotFound as e:
        raise EntryNotFound(e.msg)
    return "<ok/>"


def create_comment(ai, feed_id, item_no, body, userpic=None, parent_id=None,
                   agent=None, comment_id=None, escape_html=False):
    args = {'body': body}
    if userpic:
        args['userpic'] = userpic

    if comment_id:
        args['comment_id'] = comment_id.split('/')[-1]

    meta = None
    try:
        return api.CreateComment(
            ai=ai,
            feed_id=int(feed_id),
            item_no=int(item_no),
            parent_id=int(parent_id if parent_id else 0),
            post_type='text',
            meta=meta,
            trackback=False,
            params=list(force_str_dict_items(args)),
            escape_html=escape_html,
        )
    except assertions.AssertionFailure as e:
        raise InvalidInputError.from_corba_assertions(e.assertions)
    except exceptions.InvalidParams as e:
        raise InvalidInputError(str(e))
    except exceptions.NotFound as e:
        raise EntryNotFound('Comment not found')
    except Exception:
        log.exception('author_id=%s, agent=%s, feed_id=%s, type=%s' % (
            int(ai.login),
            force_str(agent),
            int(feed_id),
            force_str(type),
        )
                      )
        raise


def _parse_list_feeds_by_group(xml):
    tree = et.fromstring(xml)
    dom_root = tree.xpath('//feeds')[0]
    dom_feeds = dom_root.getchildren()
    total_items = int(dom_root.attrib['total'])
    feed_id_list = [dom_feed.attrib['feed-id'] for dom_feed in dom_feeds]
    profiles = etree2py(et.XML(retrieve_fast_profiles(feed_id_list)))
    if hasattr(profiles, 'values'):  # см. AT-1292
        profiles = list(profiles.values())
    return (total_items, profiles)


def get_club_members_count(feed_id):
    xml = aux.GetCommunityMemberCount(int(feed_id))
    tree = et.fromstring(xml)
    members_count = tree.xpath('//club_member_count/text()')
    if members_count:
        return int(members_count[0])
    return None


def retrieve_friends(feed_id, page_no, page_size):
    return _parse_list_feeds_by_group(
        api.GetFriends(int(feed_id), page_no, page_size))


def club_members(feed_id, page_no, page_size):
    return _parse_list_feeds_by_group(
        api.GetMembers(int(feed_id), page_no, page_size))


def club_moderators(feed_id, page_no, page_size):
    return _parse_list_feeds_by_group(
        api.GetModerators(int(feed_id), page_no, page_size))


def club_owner(feed_id):
    return _parse_list_feeds_by_group(
        api.GetUsersByGroup(int(feed_id), 'owner', 0, 1, 'blogs'))


def member_of_clubs(feed_id, page_no, page_size):
    return _parse_list_feeds_by_group(
        api.GetClubs(int(feed_id), page_no, page_size))


def owner_of_clubs(feed_id, page_no, page_size):
    return _parse_list_feeds_by_group(
        api.GetOwnedClubs(int(feed_id), page_no, page_size))


def moderator_of_clubs(feed_id, page_no, page_size):
    return _parse_list_feeds_by_group(
        api.GetModeratedClubs(int(feed_id), page_no, page_size))


def retrieve_friends_posts(ai, feed, since, count):
    xml = aux.ReadFriendsLight(
        ai=ai,
        person_id=int(feed.get_id()),
        tb=since,
        count=count,
    )
    return xml


def ban_person_in_feed(ai, feed_id, person, delete_posts):
    try:
        aux.BanPerson(ai, request={
            'feed_id': int(feed_id),
            'uid': int(person)
        })
    except exceptions.AccessDenied:
        return {"Error": "Access Denied"}
    except Exception as er:
        log(logging.WARNING, "Error while banning person %s in club %s: %s" % (
        person, feed_id, er))
        return {"Error": "Unknown error"}
    else:
        answer = {"Status": "Banned"}
        if delete_posts:
            try:
                raise NotImplementedError('ban_person_in_feed')
            except Exception as er:
                log(logging.WARNING,
                    "Error while removing person %s posts in club %s: %s" % (
                    person, feed_id, er))
                answer["warning"] = "Error while flushing posts"
        return answer


def subscribe_post(ai, feed_id, post_no, subscribe):
    return aux.SubscribeToPost(
        ai=ai,
        feed_id=int(feed_id),
        item_id=int(post_no),
        with_comments=int(subscribe),
    )
