from django.http import HttpResponse, HttpResponseNotFound

from at.api.yaru.friend.store import FriendsAtomStore
from at.api.yaru.utils.django_helpers import response_as_atom_feed, to_canonical, get_pagination, require_permission, require_ctypes, check_post_type, get_format
from at.api.yaru.instances import Blog
from at.api.yaru.errors import InvalidInputError, OutOfBoundsError


def _posts_index(request, feed_id, post_type=''):
    format, forced = get_format(request, ['application/x-yaru-atom+json',
                                          'application/atom+xml'],
                                'application/atom+xml')
    if post_type:
        check_post_type(post_type)
    try:
        since = int(request.GET.get('since', 0))
        count = int(request.GET.get('count', 50))
    except ValueError:
        raise InvalidInputError('')
    if count > 50:
        count = 50
    if since < 0:
        raise OutOfBoundsError('')
    store = FriendsAtomStore(request.ai, Blog(feed_id, abs_url=request.build_absolute_uri), abs_url=request.build_absolute_uri)
    if not store:
        return HttpResponseNotFound()
    renderer = store.posts_index(since, count, post_type)
    if 'xml' in format:
        return response_as_atom_feed(renderer.to_xml(forced, render=True))
    else:
        return HttpResponse(renderer.to_json(forced, render=True), content_type='application/x-yaru-atom+json')

@to_canonical(Blog, 'blog_friends')
@require_permission('yaru:read', allow_anonymous=True)
def index(request, feed_id):
    format, forced = get_format(request, ['application/x-yaru+json',
                                          'application/x-yaru+xml'],
                                'application/x-yaru+xml')
    store = FriendsAtomStore(request.ai, Blog(feed_id, abs_url=request.build_absolute_uri), abs_url=request.build_absolute_uri)
    if not store:
        return HttpResponseNotFound()
    page_no, page_size = get_pagination(request)
    renderer = store.index(page_no, page_size)
    if 'xml' in format:
        feed = renderer.to_xml(forced, 'persons', render=True)
    else:
        feed = renderer.to_json(forced, 'persons', render=True)
    return HttpResponse(feed, content_type='%s; type=persons' % format)

@to_canonical(Blog, 'blog_friends_posts')
@require_permission('yaru:read', allow_anonymous=True)
def posts_index(request, feed_id):
    return _posts_index(request, feed_id, '')

@to_canonical(Blog, 'blog_friends_posts_by_type')
@require_permission('yaru:read', allow_anonymous=True)
def posts_index_by_type(request, feed_id, post_type):
    return _posts_index(request, feed_id, post_type)
