# -*- coding: utf-8 -*-
import blackbox

from at.api.yaru import urlgen
from at.api.yaru import datacore
from at.api.yaru.errors import ClubNotFound, BlogNotFound, FeedNotFound
from at.api.yaru.person.models import Profile
from at.api.yaru import atomgen
from django.core.urlresolvers import reverse
from django.conf import settings


import logging

logger = logging.getLogger(__name__)


def from_uid(uid, abs_url=lambda url: url):
    info = Profile.from_feed_id(uid)
    if not info:
        logger.debug('Instance from uid info {} uid {}'.format(info, uid))
        raise FeedNotFound(uid)
    if info['type'] == 'community':
        return Club(uid, abs_url=abs_url)
    else:
        return Blog(uid, abs_url=abs_url)

def is_uid(name):
    return name.isdigit()


json_blackbox = blackbox.JsonBlackbox()

class FeedsCache(object):
    feeds_cache = {}
    def __init__(self, abs_url=lambda url: url):
        self.abs_url = abs_url

    def fill_cache(self, feed_ids):
        absent_ids = [ feed_id for feed_id in feed_ids if feed_id not in self.feeds_cache and feed_id]
        if absent_ids:
            self.feeds_cache.update(from_feed_ids(absent_ids, abs_url=self.abs_url))

    def reset(self):
        self.feeds_cache = {}

    def get(self, uid):
        self.fill_cache([uid])
        return self.feeds_cache[uid]


class Club(object):
    def __init__(self, feed_name, profile=None, abs_url=lambda url: url):
        self.abs_url = abs_url
        self.feed_name = feed_name
        if profile and 'id' in profile:
            self.feed_id = str(profile.get('id'))
        else:
            self.feed_id = datacore.get_club_id(self.feed_name)
        self._got_info = False
        self._got_members_count = False
        self._members_count = None
        self._info = profile
        if self._info:
            self._got_info = True

    def _lazy_get_info(self):
        if self._got_info:
            return
        self._info = Profile.from_feed_id(self.feed_id)
        self._got_info = True
        if self._info is None:
            raise ClubNotFound(self.feed_id)

    def __str__(self):
        return 'Club: %s (%s)' % (self.feed_id, self.feed_name)

    def get_type(self):
        return 'club'

    def get_id(self):
        return self.feed_id

    def get_name(self):
        self._lazy_get_info()
        return str(self._info['title'])

    def get_owner(self):
        from at.api.yaru.club.store import ClubStore
        store = ClubStore(0, self) # 0L - Anonymous
        return store.get_owner()

    def get_login(self):
        self._lazy_get_info()
        return self._info.get('master_url', self._info['id'])

    @property
    def login(self):
        return self.get_login()

    def get_profile(self):
        self._lazy_get_info()
        return self._info

    def get_self_link(self):
        return self.abs_url(reverse('club', args=[self.login]))

    def get_comment_link(self, post_id, comment_id):
        return self.abs_url(reverse('club_comment', args=[self.login, post_id]) + comment_id + '/')

    def get_post_relations(self, item_id):
        return {'self': [{'href': self.abs_url(reverse('club_post', args=[self.login, item_id]))}],
                'edit': [{'href': self.abs_url(reverse('club_post', args=[self.login, item_id]))}],
                'alternate': [{'href': settings.CLUB_POST % {'login': self.login, 'item_no': item_id}}],
                'comments': [{'href': self.abs_url(reverse('club_comment', args=[self.login, item_id]))}],
                }

    def get_comment_relations(self, item_id, comment_id):
        return {'alternate': [{'href': settings.CLUB_COMMENT % {'login': self.login,
                                                                'item_no': item_id,
                                                                'comment_id': comment_id}}]}

    def get_relations(self):

        rels = {'userpic': [{'href': urlgen.club_avatar_url(self.get_id(), self.get_profile()["picture"])}],
                'avatar_url': [{'href': urlgen.club_avatar_url_new(self.get_id(), self.get_profile()["picture"])}],
                'club_members': [{'href': self.abs_url(reverse('club_list_members', args=[self.login]))}],
                'club_moderators': [{'href': self.abs_url(reverse('club_list_moderators', args=[self.login]))}],
                'posts': [{'href': self.abs_url(reverse('club_posts', args=[self.login]))}],
                'www': [{'href': self.get_profile_link(self.login)}],
                'self': [{'href': self.get_self_link()}],
                'service': [{'href': self.abs_url(reverse('club_atomsvc', args=[self.login])), 'type': 'application/atomsvc+xml'}],
                }

        owner = self.get_owner()
        if owner:
            rels['club_owner'] = [{'href': self.abs_url(reverse('blog_person', args=[owner.login]))}]
        return rels

    def get_property_link(self, property):
        return self.abs_url(reverse('club_property', kwargs={'feed_id': self.login, 'property_id': property}))

    def get_members_count(self):
        if self._got_members_count:
            return self._members_count
        self._members_count = datacore.get_club_members_count(self.feed_id)
        self._got_members_count = True
        return self._members_count

    @staticmethod
    def get_profile_link(login):
        return settings.CLUB_URL % {"login": login}

    @staticmethod
    def get_userpic_link(login):
        return ''

    def exists(self):
        # Надо порефакторить этот модуль, чтобы у Blog и Club был общий предок и не дублировать код,
        # но сейчас я хочу быстро выкатить фикс для AT-1142
        if not Profile.exists(self.get_id()):
            raise ClubNotFound(self.get_id()) if self.get_type() == 'club' else BlogNotFound(self.get_id())



class Blog(object):
    def __init__(self, feed_name, profile=None, abs_url=lambda url: url):
        self.abs_url = abs_url
        self.feed_name = str(feed_name)
        if profile and 'id' in profile:
            self.feed_id = str(profile['id'])
        else:
            self.feed_id = datacore.get_blog_id(self.feed_name)
        self.display_name = None
        self._got_info = False
        self._info = profile
        if self._info:
            self._got_info = True

    def exists(self):
        if not Profile.exists(self.get_id()):
            raise ClubNotFound(self.get_id()) if self.get_type() == 'club' else BlogNotFound(self.get_id())

    def _lazy_get_info(self):
        if self._got_info:
            return None
        self._info = Profile.from_feed_id(self.feed_id)
        self._got_info = True
        if self._info is None:
            raise BlogNotFound(self.feed_id)

    def __str__(self):
        return 'Blog: %s (%s)' % (self.feed_id, self.feed_name)

    def get_type(self):
        return 'person'

    def get_id(self):
        return self.feed_id

    def get_name(self):
        self._lazy_get_info()
        return self._info['title']

    def get_login(self):
        self._lazy_get_info()
        return self._info.get('master_url', self._info['id'])

    @property
    def login(self):
        return self.get_login()

    def get_profile(self):
        self._lazy_get_info()
        return self._info

    def get_display_name(self):
        if self.display_name:
            return self.display_name
        self._lazy_get_info()
        user_info= json_blackbox.userinfo(uid=self.feed_id,
                                             userip='128.0.0.1',
                                             social_info=True,
                                             regname=True)
        try:
            self.display_name = user_info['users'][0]['display_name']
        except (KeyError, IndexError):
            return {}

    def get_self_link(self):
        return self.abs_url(reverse('blog_person', args=[self.login]))

    def get_comment_link(self, post_id, comment_id):
        return self.abs_url(reverse('blog_comment', args=[self.login, post_id]) + comment_id + '/')

    def get_post_relations(self, item_id):
        return {'self': [{'href': self.abs_url(reverse('blog_post', args=[self.login, item_id]))}],
                'edit': [{'href': self.abs_url(reverse('blog_post', args=[self.login, item_id]))}],
                'alternate': [{'href': settings.BLOG_POST % {'login': self.login, 'item_no': item_id}}],
                'comments': [{'href': self.abs_url(reverse('blog_comment', args=[self.login, item_id]))}],
                }

    def get_comment_relations(self, item_id, comment_id):
        return {'alternate': [{'href': settings.BLOG_COMMENT % {'login': self.login,
                                                                'item_no': item_id,
                                                                'comment_id': comment_id}}]}

    def get_relations(self):
        return {'userpic': [{'href': urlgen.userpic_url(self.login)}],
                'member_of_clubs': [{'href': self.abs_url(reverse('person_member_of_clubs', args=[self.login]))}],
                'owner_of_clubs': [{'href': self.abs_url(reverse('person_owner_of_clubs', args=[self.login]))}],
                'moderator_of_clubs': [{'href': self.abs_url(reverse('person_moderator_of_clubs', args=[self.login]))}],
                'friends': [{'href': self.abs_url(reverse('blog_friends', args=[self.login]))}],
                'friends_posts': [{'href': self.abs_url(reverse('blog_friends_posts', args=[self.login]))}],
                'posts': [{'href': self.abs_url(reverse('blog_posts', args=[self.login]))}],
                'self': [{'href': self.get_self_link()}],
                'www': [{'href': self.get_profile_link(self.login)}],
                'service': [{'href': self.abs_url(reverse('blog_atomsvc', args=[self.login])), 'type': 'application/atomsvc+xml'}],
            }

    def get_property_link(self, property):
        return self.abs_url(reverse('blog_property', kwargs={'feed_id': self.login, 'property_id': property}))

    @staticmethod
    def get_profile_link(login):
        return urlgen.profile_page_url(login)

    @staticmethod
    def get_userpic_link(login):
        return urlgen.userpic_url(login)


def from_feed_ids(uids, abs_url=lambda url: url):
    feeds = {}
    profiles = Profile.from_feed_ids(uids)
    for uid in uids:
        profile = profiles.get(int(uid))
        if profile:
            if profile['type'] == 'community':
                feeds[uid] = Club(uid, profile=profile, abs_url=abs_url)
            # FIXME а фиг его знает, что там ещё может быть помимо blog и profile
            # elif profile['type'] in ['blog', 'profile']:
            else:
                feeds[uid] = Blog(uid, profile=profile, abs_url=abs_url)
                # У социального пользователя нет логина
                if not feeds[uid].get_login():
                    feeds[uid].get_display_name()

        else:
            feeds[uid] = None
    return feeds
