# -*- coding: utf-8 -*-

import lxml.etree as et
from datetime import datetime
from collections import MutableMapping
from at.api.yaru.utils.py2xml import etree2py
from at.api.yaru.utils.odict import odict

from at.api.yaru import atom, datacore, activity


# Постулаты по которым строится Profile.
#
# * Профиль представляет собой единое пространство имён свойств.
# (В я.ру профиль хранится в bulca группами (которые кое-где называются категориями),
# свойства могут быть получены только группой, изменения тоже могут вноситься
# только для группы целиком. Группировка свойств в группы в я.ру достаточно
# произвольная и обусловлена в основном только потребностями отображения страницы
# профиля пользователя. Может свободно измениться.)
#
# * "Медленный" профиль читается всегда.
# (Обойтись только "быстрым" профилем (хранимым в sql-базе) почти никогда
# не получается; названия "быстрый" и "медленный" когда-то имели отношение
# к скорости получения профиля, но в настоящий момент являются по большей части
# исторической традицией)
#
# * Группировка из я.ру представляется с помощью категорий (тегов)
#


def get_fast_profile(id):
    return _parse_fast_profile(datacore.retrieve_fast_profile(id))


def _parse_fast_profile(text):
    xml = et.XML(text)
    if xml.tag == 'error':
        return None
    return odict([(key, value) for (key, value) in list(etree2py(et.XML(text)).items()) if value])


def get_slow_profile(ai, id):
    '''Получить "медленный" профиль пользователя я.ру.

    Свойства/атрибуты возвращаются в виде odict-словаря:
        name -> odict(name, group, value)

    group -- имя исходной категории (группы) свойств из я.ру
    value -- может быть значением примитивного типа, списком или словарём
    '''
    return _parse_slow_profile(datacore.retrieve_slow_profile(ai, id))

def _parse_slow_profile(text):
    #XXX: использовать парсер с опцией выкидывания незначащих whitespace узлов
    #XXX: перевод строчных значений в unicode?
    doc = et.XML(text)
    entries = doc.xpath('./source-profile/category/entry')
    parsed = {}
    for entry in entries:
        group = entry.findtext('category')
        orig_name = entry.findtext('name')
        name = entry.findtext('name')
        if group == 'Special' and orig_name == 'custom_entry':
            # в custom_entry/value должен xml вида '<xml><name/><value/></xml>'
            # name и value не могут быть пустыми, value снова может содержать <xml/>
            _tmp = entry.xpath('./value/xml')
            if len(_tmp) > 0:
                entry = _tmp[0]
            else:
                entry = None
            #entry = len(_tmp) > 0 and _tmp[0] or None
            if entry is None:
                continue
            name = entry.findtext('name')
        value = entry.find('value')
        if value is None:
            continue
        # convert to structure
        if len(value.getchildren()) > 0:
            value = etree2py(value)
            if issubclass(value.__class__, list):
                value = value[0]
        else:
            value = value.text
        #value = len(value.getchildren()) > 0 and etree2py(value)[0] or value.text
        if not value:
            continue
        #XXX: нужны ли имена в unicode?
        #parsed.setdefault((force_unicode(name), group), []).append(value)
        parsed.setdefault((orig_name, name, group), []).append(value)
    del doc

    # создание выходного словаря атрибутов, с одновременным
    #  * разрешением конфликтов имён
    #  * и с заменой в значениях списков из одного элемента на сам элемент
    #
    # состав атрибута: оригинальное имя, имя, группа, значение (которое может быть значением
    # примитивного типа, списком или словарём):
    # attr = odict(name=name, group=group, value=value)
    #
    names = [i[0] for i in parsed.keys()]
    conflicts = _duplicates(names)
    attrs = odict()
    for (orig_name, name, group), value in parsed.items():
        # конфликт одинакового имени в разных группах разрешается
        # добавлением суффиксов из имён групп ('name' -> 'name#group')
        name = '#'.join([name, group]) if name in conflicts else name
        if len(value) == 1:
            value = value[0]
        if orig_name == 'custom_entry' and group == 'Special':
            # FIXME my eyes!
            attrs.setdefault('special', odict(name=orig_name, group=group, value=odict())).value[name] = value
        else:
            attrs[name] = odict(name=name, group=group, value=value)
    return attrs

def _duplicates(alist):
    ''' Возвращает множество элементов списка, встречающихся в нём более одного раза '''
    unique = set()
    duplicates = set()
    for i in alist:
        if i in unique:
            duplicates.add(i)
        unique.add(i)
    return duplicates


#TODO: Ещё требуется реализация свойств только для чтения.
#TODO: не возвращать отсутствующие или пустые свойства
#TODO: добавить работу по whitelist'у свойств
class Profile(MutableMapping):
    """Единый профиль пользователя.

    Поддерживает интефейс словаря и интерфейс коллекции -- можно
    запрашивать как profile['title'], так и profile.title.

    Однако второй способ не работает для свойств с не ascii именами
    (свойствами из группы Special, чьи имена свободно задаются пользователем);
    также на таких свойствах не будет работать запрос через getattr().
    """

    @staticmethod
    def empty_string_as_default(self, name):
        return str(self.fast_profile.get(name, '')) or ''

    @staticmethod
    def int_or_empty(self, name):
        value = self.fast_profile.get(name)
        return int(value) if value else ''

    @staticmethod
    def list_from_list_or_string(self, name):
        value = self.fast_profile.get(name, [])
        if issubclass(value.__class__, str):
            value = [value]
        return list(map(str, value))

    def __setitem__(self, key, value):
        raise NotImplementedError

    def __delitem__(self, key):
        raise NotImplementedError

    def __len__(self):
        return len(self.fast_profile)

    def __init__(self):
        self.fast_profile = {}
        self.tags = None
        self.out_conversion = {'status': self.empty_string_as_default,
                               'qu': self.int_or_empty,
                               'score_to_level': self.int_or_empty,
                               'mood': self.empty_string_as_default,
                               'qu_up_time': lambda self, name: atom.strftime(datetime.utcfromtimestamp(float(self.fast_profile.get('qu_up_time', 0)))) if 'qu_up_time' in self.fast_profile else '',
                               'name': lambda self, _: self.get_title(),
                               'name_eng': lambda self, _: self.get_title_eng(),
                               'sex': lambda self, x: self.fast_profile.get('sex', '') or 'unknown',
                               'abilities': self.list_from_list_or_string,
                               'languages': self.list_from_list_or_string,
                               'consulting': self.list_from_list_or_string,
                               'account-status': lambda self, _: str(self.fast_profile.get('status', '')),
                               }
        for field in ['email',
                      'mobile_phone',
                      'work_phone',
                      'hair',
                      'height',
                      'weight',
                      'smoking',
                      'marital',
                      'children',
                      'habitat',
                      'constitution',
                      'city',
                      'ya-online',
                      'icq',
                      'm-agent',
                      'g-talk',
                      'skype',
                      'website',
                      'country',
                      'metro',
                      ]:
            self.out_conversion[field] = self.empty_string_as_default
        for field in ['birth_year', 'birth_month', 'birth_day']:
            self.out_conversion[field] = self.int_or_empty


    def _strip(self):
        # email должен браться только из длинного профиля
        # связано это с настройкой, кому показывать email можно, а кому нельзя
        # быстрый профиль не проверяет настройки приватности, поэтому персональные
        # данные нельзя оттуда показывать
        self.fast_profile.pop('email', None)

    @classmethod
    def from_feed_id(cls, profile_id, fetch_fast_profile=True):
        profile = cls()
        if cls.exists(profile_id):
            if fetch_fast_profile:
                profile.fast_profile = get_fast_profile(profile_id)
            profile._strip()
            return profile

    @classmethod
    def from_feed_ids(cls, feed_ids):
        data = etree2py(et.fromstring(datacore.retrieve_fast_profiles(feed_ids)))
        if len(feed_ids) == 1: # Если 1 feed_id, etree2py распарсит в словарь вместо списка
            data = list(data.values())
        return dict((feed_info['id'], cls.from_dict(feed_info)) for feed_info in data)

    @classmethod
    def from_dict(cls, fast_profile):
        profile = cls()
        profile.fast_profile = fast_profile
        profile._strip()
        return profile

    @classmethod
    def exists(cls, feed_id):
        """Есть ли у пользователя полноценный account на Я.ру"""
        p = get_fast_profile(feed_id)
        return True if p and p.status != 'deleted' else False

    def get_relations(self, page_no, page_size):
        total, feeds = datacore.retrieve_friends(self.fast_profile['id'],
                                                 page_no, page_size)
        return total, [ Profile.from_dict(feed) for feed in feeds]

    def get_title(self):
        return self['title'] or self['master_url']

    def get_title_eng(self):
        return self['title_eng'] or self.get_title()

    def get_tags(self, scheme):
        self.tags = activity.retrieve_feed_tags(self.fast_profile['id'], scheme)

    def __contains__(self, name):
        return (name in self.fast_profile)

    def keys(self):
        return list(self.keys())

    def iterkeys(self):
        return self.__iter__()

    def __iter__(self):
        return iter(self.fast_profile.keys())

    def iteritems(self):
        return iter(self.fast_profile.items())

    def items(self):
        return list(self.fast_profile.items())

    def _lookup(self, name):
        if name in self.out_conversion:
            return self.out_conversion[name](self, name)
        return self.fast_profile.get(name)

    def __getattr__(self, name):
        if name in self:
            return self._lookup(name)
        raise AttributeError(name)

    def __getitem__(self, name):
        if isinstance(name, str):
            return self._lookup(name)
        else:
            raise TypeError('String keys accepted only')

    def __str__(self):
        return 'fast: %s\ntags: %s' % (self.fast_profile, self.tags)
