# -*- coding: utf-8 -*-
import lxml.etree as et
from at.api.yaru.templates import ATOM_NAMESPACES as _ATOM_NAMESPACES
from at.api.yaru.atomgen import OLD_YAPI_NS
from at.api.yaru.utils import guards
import re


ATOM_NAMESPACES = dict(_ATOM_NAMESPACES)
ATOM_NAMESPACES['atom'] = ATOM_NAMESPACES.pop('')


RE_FEED_ID = re.compile(r'^urn:ya\.ru:(club|person)/(\d+)/?$')
ET_PERSON_ID = lambda ns: et.XPath('y:person/y:id/text()', namespaces=ns)
ET_CLUB_ID = lambda ns: et.XPath('y:club/y:id/text()', namespaces=ns)
ET_LINK = lambda ns: et.XPath('y:url/text()', namespaces=ns)
ET_EVENT = lambda ns: et.XPath('y:event/text()', namespaces=ns)


XML_OR_JSON = ValueError("request_type must be either xml or json")


def parse_feed_id(feed_id):
    m = RE_FEED_ID.match(feed_id)
    if m:
        feed_type, uid = m.groups()
        return {'type': feed_type,
                'uid': int(uid),
                }

def parse_meta(feed, meta, meta_type, request_type, old_ns=False):
    guards.not_none(feed)
    namespaces = dict(list(ATOM_NAMESPACES.items()))
    if old_ns:
        namespaces['y'] = OLD_YAPI_NS
    if meta_type in ['friend', 'join', 'unfriend', 'unjoin']:
        output = parse_friend_meta(meta, request_type, namespaces)
    elif meta_type == 'link':
        output = parse_link_meta(meta, request_type, namespaces)
    elif meta_type == 'congratulation':
        output = parse_congratulation_meta(meta, request_type, namespaces)
    else:
        raise ValueError("Unsupported meta_type %s" % meta_type)
    guards.not_none(output)
    output['feed'] = feed
    return output

def parse_friend_meta(meta, request_type, namespaces):
    if request_type == 'xml':
        # бред, теперь можно в эл-те club иметь id=urn:ya.ru:person/12345 :-)
        feed_id = ET_PERSON_ID(namespaces)(meta) or ET_CLUB_ID(namespaces)(meta)
        return guards.not_none(parse_feed_id(feed_id[0]))
    elif request_type == 'json':
        return guards.not_none(parse_feed_id(meta['person']['id']))
    else:
        raise XML_OR_JSON

def parse_link_meta(meta, request_type, namespaces):
    if request_type == 'xml':
        return {'URL': ET_LINK(namespaces)(meta)[0]}
    elif request_type == 'json':
        return {'URL': meta['url']}
    else:
        raise XML_OR_JSON

def parse_congratulation_meta(meta, request_type, namespaces):
    if request_type == 'xml':
        feed_id = ET_PERSON_ID(namespaces)(meta)
        feed = parse_feed_id(feed_id[0]) if feed_id else 0
        return {'event': ET_EVENT(namespaces)(meta)[0],
                'whom': feed,
                }
    elif request_type == 'json':
        person = meta.get('person')
        _id = person.get('id') if person else None
        return {'event': meta['event'],
                'whom': parse_feed_id(_id) if _id else {'type': 'person', 'uid': 0}}
    else:
        raise XML_OR_JSON
